package cn.com.duiba.nezha.alg.alg.vo.adx.flowfilter;


import cn.com.duiba.nezha.alg.common.util.AssertUtil;
import cn.com.duiba.nezha.alg.common.util.DataUtil;
import cn.com.duiba.nezha.alg.common.util.MathUtil;
import lombok.Data;

import java.io.Serializable;

/**
 * 实时统计指标
 */
@Data
public class AdxIndexStatDo implements Serializable {

    private static final long serialVersionUID = 4481088624505592801L;

    /**
     * 竞价返回量级
     */
    private Long bid;

    /**
     * 竞价成功量级
     */
    private Long bidSuc;

    /**
     * 曝光量级
     */
    private Long exp;

    /**
     * 点击量级
     */
    private Long click;

    /**
     * adx消耗(分*10^4/cpm)
     */
    private Long adxConsume;

    /**
     * 广告消耗(分)
     */
    private Long advertConsume;

    /**
     * 广告发券数
     */
    private Long advertLaunch;

    /**
     * 活动访问次数
     */
    private Long activityRequest;

    /**
     * 活动参与次数
     */
    private Long activiyJoin;

    /**
     * 广告曝光次数
     */
    private Long advertExposure;

    /**
     * 广告计费点击数
     */
    private Long advertClick;

    /**
     * 预估点击价值总和
     */
    private Double preClickValue;

    /**
     * 累计出价因子factor（真实返回）
     */
    private Double factor;




    //算法计算指标

    /**
     * adx消耗(分)
     */
    private Long adxCost;

    /**
     * ad消耗(分)
     */
    private Long adCost;

    /**
     * 数据是否置信
     */
    private Boolean isConfident;

    /**
     * adx点击率
     */
    private Double ctr;

    /**
     * 每PV发券(发券pv/adx点击PV)
     */
    private Double pvLaunch;

    /**
     * arpu(广告消耗/发券次数)
     */
    private Double adArpu;

    /**
     * 平均出价因子factor
     */
    private Double avgfactor;

    /**
     * adx ecpm(adx消耗/曝光次数*1000)
     */
    private Double cpm;

    /**
     * ad ecpm(广告消耗/曝光次数*1000)
     */
    private Double adEcpm;

    /**
     * 点击价值（分/单次点击）
     */
    private Double clickValue;

    /**
     * 竞价成功率
     */
    public Double sucRate;

    /**
     * 目标偏差率
     */
    public double bias;

    /**
     * 实际roi
     */
    public double actRoi;

    /**
     * 实际cpc
     */
    public double actCpc;

    /**
     * 目标实际值
     */
    public double actValue;



    public static Double getStatCtr(AdxIndexStatDo adxIndexStatDo) {
        Double ret = null;
        if(adxIndexStatDo != null && adxIndexStatDo.getExp() != null && adxIndexStatDo.getExp() > 100) {
            ret = DataUtil.division(adxIndexStatDo.getClick(), adxIndexStatDo.getExp(), 3);
        }
        return ret;
    }

    public static Double getStatLaunchPv(AdxIndexStatDo adxIndexStatDo) {
        Double ret = null;
        if(adxIndexStatDo != null && adxIndexStatDo.getClick() != null && adxIndexStatDo.getClick() > 10) {
            ret = DataUtil.division(adxIndexStatDo.getAdvertLaunch(), adxIndexStatDo.getClick(), 3);
        }
        return ret;
    }

    public static Double getStatEcpm(AdxIndexStatDo adxIndexStatDo) {
        Double ret = null;
        if(adxIndexStatDo != null && adxIndexStatDo.getExp() != null && adxIndexStatDo.getExp() > 1000 && adxIndexStatDo.getAdvertConsume()!=null) {
            ret = DataUtil.division(adxIndexStatDo.getAdvertConsume() * 1000, adxIndexStatDo.getExp(), 3);
        }
        return ret;
    }

    /**
     * 计算当日累计消耗、实时消耗比值
     * @param adxStatBaseDoMin
     * @param adxStatBaseDoDay
     * @return
     */
    public static double getCostWeigh(AdxIndexStatDo adxStatBaseDoMin, AdxIndexStatDo adxStatBaseDoDay) {
        Double ret = null;
        if(AssertUtil.isAllNotEmpty(adxStatBaseDoMin, adxStatBaseDoDay)) {
            ret = MathUtil.division(adxStatBaseDoDay.getAdxConsume3(), adxStatBaseDoMin.getAdxConsume3() , 3);
            ret = MathUtil.log(ret);
        }

        if(ret == null) {
            ret = 1.;
        }

        ret = MathUtil.stdwithBoundary(ret, 1, 10);

        return ret;
    }

    /**
     * 判断消耗是否置信
     * @param adxStatBaseDo
     * @return
     */
    public static boolean isCostConfidence(AdxIndexStatDo adxStatBaseDo) {
        boolean ret = false;
        if(adxStatBaseDo != null) {
            if(adxStatBaseDo.getAdxConsume() != null && adxStatBaseDo.getAdxConsume2() > 3000) {
                ret = true;
            }
        }
        return ret;
    }

    /**
     * 判断曝光是否置信
     * @param adxStatBaseDo
     * @return
     */
    public static boolean isImpConfidence(AdxIndexStatDo adxStatBaseDo) {
        boolean ret = false;
        if(adxStatBaseDo != null) {
            if(adxStatBaseDo.getExp() != null && adxStatBaseDo.getExp() > 1000) {
                ret = true;
            }
        }
        return ret;
    }

    /**
     * 获取竞价成功率
     */
    public static Double getWinRate(AdxIndexStatDo adxStatBaseDo) {
        Double ret = null;
        if(adxStatBaseDo != null) {
            ret = DataUtil.division(adxStatBaseDo.getExp(), adxStatBaseDo.getBid(), 3);
        }
        return ret;
    }

    public static boolean getWinRateStatus(AdxIndexStatDo adxStatBaseDoMin, AdxIndexStatDo adxStatBaseDoDay) {
        boolean ret = false;

        Double winRate = AdxIndexStatDo.getWinRate(adxStatBaseDoMin);
        Double winRateDay = AdxIndexStatDo.getWinRate(adxStatBaseDoDay);

        if(AssertUtil.isAllNotEmpty(winRate, winRateDay)) {
            if(winRate < winRateDay * 0.5 && Math.random() < 0.2) {
                ret = true;
            }
        }
        return ret;

    }

    /**
     * 获取adx消耗（分-每次曝光）
     */
    public Double getAdxConsume2() {
        return MathUtil.division(adxConsume, 10000000., 3);
    }

    /**
     * 获取adx消耗（分-每千次曝光）
     *
     */
    public Double getAdxConsume3() {
        return MathUtil.division(adxConsume, 10000., 3);
    }

    /**
     * 计算统计arpu值
     */
    public static Double getStatArpu(AdxIndexStatDo adxStatBaseDo) {
        Double ret = null;
        if(AdxIndexStatDo.isCostConfidence(adxStatBaseDo)) {
            ret = MathUtil.division(adxStatBaseDo.getAdvertConsume(), adxStatBaseDo.getAdvertLaunch(), 3);
        }
        return ret;
    }

    /**
     * 计算统计cpm值
     */
    public static Double getStatCpm(AdxIndexStatDo adxStatBaseDo) {
        Double ret = null;
        if(AdxIndexStatDo.isCostConfidence(adxStatBaseDo)) {
            ret = MathUtil.division(adxStatBaseDo.getAdxConsume3(), adxStatBaseDo.getExp(), 3);
        }
        return ret;
    }

    /**
     * 计算统计每pv发券
     */
    public static Double getLaunchPv(Double launchPv, AdxIndexStatDo adxStatBaseDo, AdxIndexStatDo statBaseDoDay) {

        Double launchPv1 = AdxIndexStatDo.getStatLaunchPv(adxStatBaseDo);
        Double launchPv2 = AdxIndexStatDo.getStatLaunchPv(statBaseDoDay);
        Double statlaunchPv = MathUtil.mean(launchPv1, launchPv2, 0.8);

        launchPv = MathUtil.mean(launchPv, statlaunchPv, 0.8);

        return launchPv;
    }

    /**
     * 计算统计ctr
     */
    public static Double getStatCtr(Double ctr, AdxIndexStatDo adxStatBaseDo, AdxIndexStatDo statBaseDoDay) {

        Double ctr1 = AdxIndexStatDo.getStatCtr(adxStatBaseDo);
        Double ctr2 = AdxIndexStatDo.getStatCtr(statBaseDoDay);
        Double statCtr = MathUtil.mean(ctr1, ctr2, 0.8);

        ctr = MathUtil.mean(ctr, statCtr, 0.8);

        if(ctr != null) {
            ctr = MathUtil.stdwithBoundary(ctr, 0., 0.2);   //限制ctr范围
        }

        return ctr;
    }

    /**
     * 计算统计cpm
     */
    public static Double getStatCpm(Double cpm, AdxIndexStatDo adxStatBaseDo, AdxIndexStatDo statBaseDoDay) {
        Double cpm1 = AdxIndexStatDo.getStatCpm(adxStatBaseDo);
        Double cpm2 = AdxIndexStatDo.getStatCpm(statBaseDoDay);
        Double statCpm = MathUtil.mean(cpm1, cpm2, 0.8);

        cpm = MathUtil.mean(cpm, statCpm, 0.2);

        cpm = MathUtil.stdwithBoundary(cpm,30, 3000);

        return cpm;
    }

    /**
     * 计算统计arpu值
     */
    public static Double getArpu(Double arpu, AdxIndexStatDo adxStatBaseDo, AdxIndexStatDo statBaseDoDay) {

        Double arpu1 = AdxIndexStatDo.getStatArpu(adxStatBaseDo);
        Double arpu2 = AdxIndexStatDo.getStatArpu(statBaseDoDay);
        Double statArpu = MathUtil.mean(arpu1, arpu2, 0.8);

        arpu = MathUtil.mean(arpu, statArpu, 0.8);

        return arpu;
    }

    /**
     * 计算统计点击价值
     */
    public static Double getStatClickValue(Double clickValue, AdxIndexStatDo adxStatBaseDo, AdxIndexStatDo statBaseDoDay) {
        Double cv1 = AdxIndexStatDo.getStatClickValue(adxStatBaseDo);
        Double cv2 = AdxIndexStatDo.getStatClickValue(statBaseDoDay);
        Double statCv = MathUtil.mean(cv1, cv2, 0.8);

        clickValue = MathUtil.mean(clickValue, statCv, 0.8);

        return clickValue;
    }


    /**
     * 计算统计点击价值
     */
    public static Double getStatClickValue(AdxIndexStatDo adxStatBaseDo) {
        Double ret = null;
        if(AdxIndexStatDo.isCostConfidence(adxStatBaseDo)) {
            ret = MathUtil.division(adxStatBaseDo.getAdvertConsume(), adxStatBaseDo.getClick(), 3);
        }
        return ret;
    }

    /**
     * 计算实际cpc
     */
    public static Double getCpc(AdxIndexStatDo adxStatBaseDo) {

        Double ret = null;
        if(adxStatBaseDo != null) {
            Long click = adxStatBaseDo.getClick();
            Double adxConsume = adxStatBaseDo.getAdxConsume2(); //注意单位
            if(AdxIndexStatDo.isCostConfidence(adxStatBaseDo)) {
                ret = DataUtil.division(adxConsume, click,3);
            }
        }
        return ret;
    }

    /**
     * 计算实际roi(分成)
     * @return
     */
    public static Double getRoi(AdxIndexStatDo adxStatBaseDo) {

        Double ret = null;
        if(adxStatBaseDo != null){
            Long advertConsume = adxStatBaseDo.getAdvertConsume();
            Double adxConsume = adxStatBaseDo.getAdxConsume2(); //注意单位
            if(AdxIndexStatDo.isCostConfidence(adxStatBaseDo)) {
                ret = DataUtil.division(adxConsume,advertConsume,3);
            }
        }

        return ret;

    }




    /**
     * 计算广告消耗/分
     * @return
     */
    public static Long getAdCost(AdxIndexStatDo adxIndexStatDo) {
        Long ret = 0L;
        if(adxIndexStatDo != null && adxIndexStatDo.getAdvertConsume() != null) {
            ret = adxIndexStatDo.getAdvertConsume();
        }
        return ret;
    }

    /**
     * 计算adx消耗/分
     * @return
     */
    public static Long getAdxCost(AdxIndexStatDo adxIndexStatDo) {
        Long ret = 0L;
        if(adxIndexStatDo != null && adxIndexStatDo.getAdxConsume() != null) {
            ret = DataUtil.double2Long(adxIndexStatDo.getAdxConsume2());
        }
        return ret;
    }


    /**
     * 计算竞价次数
     * @return
     */
    public static Long getBidCnt(AdxIndexStatDo adxIndexStatDo) {
        Long ret = 0L;
        if(adxIndexStatDo != null && adxIndexStatDo.getBid() != null) {
            ret = adxIndexStatDo.getBid();
        }
        return ret;
    }

    /**
     * 计算竞价成功次数
     * @return
     */
    public static Long getBidSucCnt(AdxIndexStatDo adxIndexStatDo) {
        Long ret = 0L;
        if(adxIndexStatDo != null && adxIndexStatDo.getBidSuc() != null) {
            ret = adxIndexStatDo.getBidSuc();
        }
        return ret;
    }


    /**
     * 计算曝光次数
     * @return
     */
    public static Long getExpCnt(AdxIndexStatDo adxIndexStatDo) {
        Long ret = 0L;
        if(adxIndexStatDo != null && adxIndexStatDo.getExp() != null) {
            ret = adxIndexStatDo.getExp();
        }
        return ret;
    }


    /**
     * 计算点击次数
     * @return
     */
    public static Long getClickCnt(AdxIndexStatDo adxIndexStatDo) {
        Long ret = 0L;
        if(adxIndexStatDo != null && adxIndexStatDo.getClick() != null) {
            ret = adxIndexStatDo.getClick();
        }
        return ret;
    }


    /**
     * 计算发券次数
     * @return
     */
    public static Long getLaunchCnt(AdxIndexStatDo adxIndexStatDo) {
        Long ret = 0L;
        if(adxIndexStatDo != null && adxIndexStatDo.getAdvertLaunch() != null) {
            ret = adxIndexStatDo.getAdvertLaunch();
        }
        return ret;
    }


}
