package cn.com.duiba.nezha.alg.alg.adx.rcmd2;

import cn.com.duiba.nezha.alg.alg.vo.adx.AdxPriceExploreDo;
import cn.com.duiba.nezha.alg.common.util.AssertUtil;
import cn.com.duiba.nezha.alg.common.util.DataUtil;
import cn.com.duiba.nezha.alg.common.util.MathUtil;
import lombok.Data;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class AdxStatData {

    private static final Logger logger = LoggerFactory.getLogger(AdxStatData.class);

    /**
     * 定时任务，每20分钟调用一次
     * @param priceExploreDoList 探价分桶的统计指标：返回次数、入口素材曝光次数、adx消耗、广告消耗
     * @param lastPriceExploreMap 上一次探价因子表
     * @param minStepSize 最小步长
     * @return 更新后的探价因子表
     */
    public static Map<String, Double> updatePriceExploreList(List<AdxPriceExploreDo> priceExploreDoList,
                                                             Map<String, Double> lastPriceExploreMap,
                                                             Double minStepSize) {

        Map<String, Double> retMap = new HashMap<>();
        // 如果上一次探价因子表为空，则初始化默认值
        if (AssertUtil.isEmpty(lastPriceExploreMap)) {
            logger.info("updatePriceExploreList, lastPriceMap is empty, priceExploreDoList {}", priceExploreDoList);
            retMap.put("1", 1.5);
            retMap.put("2", 1.0);
            retMap.put("3", 1.5);
            retMap.put("4", 2.0);
            retMap.put("5", 2.5);
            retMap.put("6", 3.0);
            return retMap;
        }

        // 找出效果指标最好的分桶
        double bestEffect = Double.NEGATIVE_INFINITY;
        String bestKey = "-1";
        for (AdxPriceExploreDo levelDo : priceExploreDoList) {
            if (AssertUtil.isNotEmpty(levelDo) && !levelDo.getLevel().equals("1")) {
                String key = levelDo.getLevel();
                Double effect = AdxPriceExploreDo.getEffect(levelDo);
                if (AssertUtil.isNotEmpty(effect) && effect > bestEffect) {
                    bestEffect = effect;
                    bestKey = key;
                }
            }
        }

        // 如果不存在最好的分桶，说明数据不置信
        if (bestKey.equals("-1")) {
            logger.info("updatePriceExploreList, data no confidence {}, {}", priceExploreDoList, lastPriceExploreMap);
            return lastPriceExploreMap;
        }
        else {
            // 根据最好的分桶的探价因子，计算新的探价因子步长，更新所有分桶的因子
            double bestFactor = MathUtil.stdwithBoundary(lastPriceExploreMap.get(bestKey), 0.2, 8);
            double oldBase = lastPriceExploreMap.get("1");
            double stepSize = Math.max((bestFactor - oldBase) / 2.0, minStepSize);

            retMap.put("1", bestFactor);
            retMap.put("2", bestFactor - stepSize);
            retMap.put("3", bestFactor);
            retMap.put("4", bestFactor + stepSize);
            retMap.put("5", bestFactor + 2 * stepSize);
            retMap.put("6", bestFactor + 3 * stepSize);
            logger.info("updatePriceExploreList, curPriceMap {}, lastPriceMap{}, minStepSize{}, priceDoList{}", retMap, lastPriceExploreMap, minStepSize, priceExploreDoList);
            return retMap;
        }
    }
}
