package cn.com.duiba.nezha.alg.alg.adx.rcmd2;

import cn.com.duiba.nezha.alg.alg.vo.adx.AdxPriceExploreDo;
import cn.com.duiba.nezha.alg.common.util.AssertUtil;
import cn.com.duiba.nezha.alg.common.util.MathUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;

public class AdxStatData {

    private static final Logger logger = LoggerFactory.getLogger(AdxStatData.class);

    /**
     * 定时任务，每20分钟调用一次
     * @param priceExploreDoList 广告位维度的分桶统计指标：返回次数、入口素材曝光次数、adx消耗、广告消耗，前6个元素为利润模式，后6个为消耗模式
     * @param priceExploreAppDoList 媒体维度的分桶统计指标：返回次数、入口素材曝光次数、adx消耗、广告消耗，前6个元素为利润模式，后6个为消耗模式
     * @param lastPriceExploreMap 上一次探价因子表
     * @param minStepSize 最小步长
     * @return 更新后的探价因子表
     */
    public static Map<String, Double> updatePriceExploreList(List<AdxPriceExploreDo> priceExploreDoList,
                                                             List<AdxPriceExploreDo> priceExploreAppDoList,
                                                             Map<String, Double> lastPriceExploreMap,
                                                             Double minStepSize) {

        Map<String, Double> retMap = new HashMap<>();

        if (minStepSize == null) { minStepSize = 0.1; }

        /**
         * 利润优先模式
        */
        int profitStart = priceExploreLevelEnum.profitLevelStart.getIndex();
        int profitEnd = priceExploreLevelEnum.profitLevelEnd.getIndex();
        Map<String, Double> profitResultMap = getResultMap(priceExploreDoList, priceExploreAppDoList, lastPriceExploreMap, minStepSize, profitStart, profitEnd);
        if (AssertUtil.isEmpty(profitResultMap)) {
            defaultResultMap(profitResultMap, profitStart, profitEnd);
        }



        /**
         * 消耗优先模式
         */
        int consumeStart = priceExploreLevelEnum.consumeLevelStart.getIndex();
        int consumeEnd = priceExploreLevelEnum.consumeLevelEnd.getIndex();
        Map<String, Double> consumeResultMap = getResultMap(priceExploreDoList, priceExploreAppDoList, lastPriceExploreMap, minStepSize, consumeStart, consumeEnd);
        if (AssertUtil.isEmpty(consumeResultMap)) {
            defaultResultMap(consumeResultMap, consumeStart, consumeEnd);
        }

        retMap.putAll(profitResultMap);
        retMap.putAll(consumeResultMap);
        return retMap;
    }

    public static void defaultResultMap(Map<String, Double> resultMap, int start, int end) {
        for (int i=start; i <= end; i++) {
            double defaultValue = (start+"").equals(i+"") ? 2.0 : 2.0 + (i-(end+start+1)/2.0) * 0.5;
            resultMap.put(i+"", defaultValue);
        }
    }


    public static Map<String, Double> getResultMap(List<AdxPriceExploreDo> priceExploreDoList,
                                      List<AdxPriceExploreDo> priceExploreAppDoList,
                                      Map<String, Double> lastPriceExploreMap,
                                      Double minStepSize, int start, int end) {

        Map<String, Double> retMap = new HashMap<>();
        String bestKey;
        if (priceExploreDoList.size()>=end && priceExploreDoList.get(end-1).getLevel().equals(end+"") && priceExploreAppDoList.size()>=end && priceExploreAppDoList.get(end-1).getLevel().equals(end+"")) {

            bestKey = getBestKey(priceExploreDoList.subList(start, end-1), priceExploreAppDoList.subList(start, end-1), start+"");
        } else {
            logger.info("updatePriceExploreList, Mode{} input data error {}, {}", start, priceExploreDoList, priceExploreAppDoList);
            return retMap;
        }

        // 如果不存在最好的分桶，说明广告位和媒体维度数据均不置信，探价因子表会重置为默认值
        if (bestKey.equals("-1") || !lastPriceExploreMap.containsKey(start+"")) {
            if (AssertUtil.isNotEmpty(priceExploreDoList.get(start).getSlotId()) && priceExploreDoList.get(start).getSlotId().equals("7330181")) {
                logger.info("updatePriceExploreList, Mode{} app_data no confidence {}, {}", start, priceExploreAppDoList.subList(start-1, end-1), lastPriceExploreMap);
            }
            else {
                if (Math.random() < 0.001) {
                    logger.info("updatePriceExploreList, Mode{} app_data no confidence {}, {}", start, priceExploreAppDoList.subList(start-1, end-1), lastPriceExploreMap);
                }
            }
        }
        else {
            // 根据最好的分桶的探价因子，计算新的探价因子步长，更新所有分桶的因子
            Map<String, Double> factorMap = getFactorMap(bestKey, minStepSize, lastPriceExploreMap, start);
            double bestFactor = factorMap.getOrDefault("bestFactor", 2.0);
            double stepSize = factorMap.getOrDefault("stepSize", 0.5);
            double idx = factorMap.getOrDefault("idx"+start, 0.0);
            for (int level = start; level <= end; level++) {
                double factor = level==start ? bestFactor : bestFactor + (level - (end+start+1)/2.0) * stepSize;
                retMap.put(level+"", factor);
            }
            retMap.put("idx"+start, idx);

            logger.info("updatePriceExploreList, curPriceMap {}, lastPriceMap{}, priceDoList{}, Mode{}, bestKey{}",
                    retMap, lastPriceExploreMap, priceExploreDoList, start, bestKey);
        }
        return retMap;
    }


    public static String getBestKey(List<AdxPriceExploreDo> priceExploreDoList,
                                    List<AdxPriceExploreDo> priceExploreAppDoList,
                                    String effectMode) {
        // 找出效果指标最好的分桶
        String bestKey;
        bestKey = findKey(priceExploreDoList, effectMode);

        // 如果不存在最好的分桶，说明广告位维度数据不置信，取媒体维度的统计数据重新计算
        if (bestKey.equals("-1")) {
            bestKey = findKey(priceExploreAppDoList, effectMode);

            if (priceExploreDoList.get(0).getSlotId().equals("7330181")) {
                logger.info("updatePriceExploreList, slot_data no confidence {}", priceExploreDoList);
            }
            else {
                if (Math.random() < 0.01) {
                    logger.info("updatePriceExploreList, slot_data no confidence {}", priceExploreDoList);
                }
            }
        }
        return bestKey;
    }

    public static String findKey(List<AdxPriceExploreDo> priceExploreDoList, String effectMode) {
        double bestEffect = Double.NEGATIVE_INFINITY;
        String bestKey = "-1";

        for (AdxPriceExploreDo levelDo : priceExploreDoList) {
            if (AssertUtil.isNotEmpty(levelDo)) {
                String key = levelDo.getLevel();
                Double effect = AdxPriceExploreDo.getEffect(levelDo, effectMode);
                if (AssertUtil.isNotEmpty(effect) && effect > bestEffect) {
                    bestEffect = effect;
                    bestKey = key;
                }
            }
        }
        return bestKey;
    }


    public static Map<String, Double> getFactorMap(String bestKey, double minStepSize, Map<String, Double> lastPriceExploreMap, int start) {
        Map<String, Double> tmpMap = new HashMap<>();
        double upper = 5.0; // 探价因子上限
        double lower = 0.3; // 探价因子下限
        double bestFactor = MathUtil.stdwithBoundary(lastPriceExploreMap.getOrDefault(bestKey, 2.0), lower, upper);
        double oldBase;
        double idx;

        oldBase = lastPriceExploreMap.getOrDefault(start+"", 2.0);
        idx = lastPriceExploreMap.getOrDefault("idx"+start, 0.0);

        double stepSize;
        // 用idx记录探价因子连续未更新的次数，如果连续4次未更新则扩大探索范围
        if (bestFactor == oldBase) { idx = idx + 1.0; } else { idx = 0.0; }
        if (idx >= 4.0) {
            if (bestFactor >= 3.0) {
                stepSize = 5 * minStepSize;
                bestFactor = bestFactor - 2 * stepSize;
                idx = 0.0;
            } else {
                stepSize = 2 * minStepSize;
            }
        }
        else {
            double stepSizeU;
            if (bestFactor - oldBase > 0) {
                stepSizeU = Math.min((upper - bestFactor)/2.0, 1.0);
            } else {
                stepSizeU = Math.min((bestFactor - lower)/2.0, 1.0);
            }
            stepSize = MathUtil.stdwithBoundary(Math.abs(bestFactor - oldBase), minStepSize, stepSizeU);
        }
        tmpMap.put("bestFactor", bestFactor);
        tmpMap.put("stepSize", stepSize);
        tmpMap.put("idx"+start, idx);
        return tmpMap;
    }
}