package cn.com.duiba.nezha.alg.alg.adx.directly;

import cn.com.duiba.nezha.alg.alg.adx.rcmd2.priceExploreLevelEnum;
import cn.com.duiba.nezha.alg.alg.vo.adx.directly.meituan.ExplorePriceDo;
import cn.com.duiba.nezha.alg.common.util.AssertUtil;
import cn.com.duiba.nezha.alg.common.util.MathUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class MeiTuanExplorePrice {
    private static final Logger logger = LoggerFactory.getLogger(MeiTuanExplorePrice.class);

    /**
     * 定时任务，每20分钟调用一次
     * @param priceExploreDoList 探价id维度的分桶统计指标：返回次数、券曝光次数、券点击次数、媒体消耗、广告消耗
     * @param lastPriceExploreMap 上一次探价因子表
     * @param liftFactor 提价幅度（从盘古获取）
     * @param realFloorPrice 手动设置底价（从盘古获取）
     * @return 更新后的探价因子表
     */
    public static Map<String, Double> updatePriceExploreList(List<ExplorePriceDo> priceExploreDoList,
                                                             Map<String, Double> lastPriceExploreMap,
                                                             Double liftFactor,
                                                             Double realFloorPrice) {

        if (lastPriceExploreMap == null) {
            lastPriceExploreMap = new HashMap<>();
        }

        if (liftFactor == null) { liftFactor = 1.0; }  // 消耗与roi的权重系数
        if (realFloorPrice == null) { realFloorPrice = 1.0; }  // 选择cpc / ocpc模式

        /**
         * 消耗优先模式
         */
        int profitStart = priceExploreLevelEnum.profitLevelStart.getIndex();
        int profitEnd = priceExploreLevelEnum.profitLevelEnd.getIndex();
        Map<String, Double> consumeResultMap = getResultMap(priceExploreDoList, lastPriceExploreMap, liftFactor, realFloorPrice, profitStart, profitEnd);
        if (AssertUtil.isEmpty(consumeResultMap)) {
            defaultResultMap(consumeResultMap, profitStart, profitEnd);
        }

        return consumeResultMap;
    }


    public static void defaultResultMap(Map<String, Double> resultMap, int start, int end) {
        // 1-5是探价桶，6是基准桶
        for (int i = start; i <= end; i++) {
            double defaultValue = i == end ? 2.0 : (i + 1) * 0.5;
            resultMap.put(i+"", defaultValue);
        }
    }


    public static Map<String, Double> getResultMap(List<ExplorePriceDo> priceExploreDoList,
                                                   Map<String, Double> lastPriceExploreMap,
                                                   Double liftFactor,
                                                   Double realFloorPrice,
                                                   int start, int end) {

        Map<String, Double> retMap = new HashMap<>();
        String bestKey;
        if (priceExploreDoList.size() == end) {
            bestKey = getBestKey(priceExploreDoList, liftFactor);
        } else {
            logger.info("updatePriceExploreList input data error {}", priceExploreDoList);
            return retMap;
        }

        // 根据最好的分桶的探价因子，计算新的探价因子步长，更新所有分桶的因子
        Map<String, Double> factorMap = getFactorMap(bestKey, lastPriceExploreMap);
        double bestFactor = factorMap.getOrDefault("bestFactor", 2.0);
        double stepSize = factorMap.getOrDefault("stepSize", 0.5);
        double idx = factorMap.getOrDefault("idx", 0.0);
        for (int level = start; level <= end; level++) {
            double factor = level == end ? bestFactor : bestFactor + (level - 3) * stepSize;
            retMap.put(level+"", factor);
        }
        retMap.put("idx", idx);

        logger.info("updatePriceExploreList, curPriceMap {}, lastPriceMap{}, priceDoList{}, bestKey{}, liftFactor{}",
                retMap, lastPriceExploreMap, priceExploreDoList, bestKey, liftFactor);

        return retMap;
    }


    public static String getBestKey(List<ExplorePriceDo> priceExploreDoList, Double liftFactor) {
        // 找出效果指标最好的分桶
        double bestEffect = Double.NEGATIVE_INFINITY;
        String bestKey = "-1";

        for (ExplorePriceDo levelDo : priceExploreDoList.subList(0, 4)) {
            if (AssertUtil.isNotEmpty(levelDo)) {
                String key = levelDo.getLevel();
                Double effect = ExplorePriceDo.getEffect(levelDo, liftFactor);
                if (AssertUtil.isNotEmpty(effect) && effect > bestEffect) {
                    bestEffect = effect;
                    bestKey = key;
                }
            }
        }
        return bestKey;
    }


    public static Map<String, Double> getFactorMap(String bestKey, Map<String, Double> lastPriceExploreMap) {
        Map<String, Double> tmpMap = new HashMap<>();
        double upper = 3.0; // 探价因子上限
        double lower = 1.0; // 探价因子下限
        double minStepSize = 0.05;
        double bestFactor = MathUtil.stdwithBoundary(lastPriceExploreMap.getOrDefault(bestKey, 2.0), lower, upper);
        double oldBase;
        double idx;

        oldBase = lastPriceExploreMap.getOrDefault("3", 2.0);
        idx = lastPriceExploreMap.getOrDefault("idx", 0.0);  // 连续未更新的次数

        double stepSize;
        // 用idx记录探价因子连续未更新的次数，如果连续4次未更新则扩大探索范围
        if (bestFactor == oldBase) { idx = idx + 1.0; } else { idx = 0.0; }
        if (idx >= 4.0) {
            stepSize = idx * minStepSize;
        }
        else {
            double stepSizeU;
            if (bestFactor - oldBase > 0) {
                stepSizeU = Math.min((upper - bestFactor)/2.0, 0.5);
            } else {
                stepSizeU = Math.min((bestFactor - lower)/2.0, 0.5);
            }
            stepSize = MathUtil.stdwithBoundary(Math.abs(bestFactor - oldBase)/1.5, minStepSize, stepSizeU);
        }
        tmpMap.put("bestFactor", bestFactor);
        tmpMap.put("stepSize", stepSize);
        tmpMap.put("idx", idx);
        return tmpMap;
    }

}
