package cn.com.duiba.nezha.alg.alg.adx;

import cn.com.duiba.nezha.alg.alg.enums.AdxIndex;
import cn.com.duiba.nezha.alg.alg.enums.AdxLevel;
import cn.com.duiba.nezha.alg.alg.enums.AdxStrategy;
import cn.com.duiba.nezha.alg.alg.vo.*;
import cn.com.duiba.nezha.alg.common.util.AssertUtil;
import cn.com.duiba.nezha.alg.common.util.DataUtil;
import cn.com.duiba.nezha.alg.common.util.LocalDateUtil;
import com.alibaba.fastjson.JSON;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;




public class FactorExploration {



    /**
     * 策略3-因子试探（根据缓存数据，调用频率：1min调用1次）
     *
     * @param adxRoiControlDoInfo 人工设置，统计数据，上一次缓存数据
     * @return 因子试探表
     */


    private static final Logger logger = LoggerFactory.getLogger(AdxAlgoBidding.class);

    public static AdxFactorExplorationDo getExploreFactor(AdxRoiControlDo adxRoiControlDoInfo) {

        AdxFactorExplorationDo ret = new AdxFactorExplorationDo();


        /**
         *
         * 步骤：
         * 步骤1、设置默认值
         * 步骤2、当前对象是否合法
         * 步骤3、计算adx消耗、roi、竞价成功率
         * 步骤4、获取上一次最优level作为基准值
         * 步骤5、判断基准值是否重置
         * 步骤6、判断基准值是否回归
         * 步骤7、调整基准流量比例
         * 步骤8、更新基准值和基准流量比例
         *
         */


        try {


            // 1、设置默认值
            Double defaultFactor = 1.0;                               // 默认ROI调节因子
            Double defaultMinRoi = 1.0;                               // 默认最低ROI
            Double lowerLimit = 0.70;                                 // ROI调节因子下限
            Double upperLimit = 1.40;                                 // ROI调节因子上限

            Double backStep = 0.003;                                  // 回归步长
            Double stepFlow1 = 0.07;                                  // 流量调节步长1
            Double stepFlow2 = 0.03;                                  // 流量调节步长2
            Double[] stepList = {-0.03, -0.01, 0.00, 0.01, 0.03};     // 基准值调节步长
            Double[] bucketDiffList = {0.2, 0.5, 0.8, 1.0};           // ROI偏差分桶值
            Double adxConsumeMsLimit = 2.0 * 100;                     // 策略实时adx消耗阈值（分）


            Double baseValue = defaultFactor;                         // 基准值
            Double[] baseFlowRate = {0.05, 0.15, 0.60, 0.15, 0.05};   // 基准流量比例
            Double[] lastRealRoi = {0.0, 0.0, 0.0, 0.0, 0.0};         // 默认上一次实际roi

            Map<String, Double> factorMap = new HashMap<>((AdxLevel.values().length - 1));
            Map<String, Double> flowRateMap = new HashMap<>((AdxLevel.values().length - 1));
            Map<String, Double> lastRoiMap  = new HashMap<>((AdxLevel.values().length - 1));



            //2、当前对象是否合法
            if (AssertUtil.isNotEmpty(adxRoiControlDoInfo)) {


                //3、计算adx消耗、roi、竞价成功率
                Long defaultConsume = 0L;
                ArrayList<AdxStrategyDo> strategyMsDoList = adxRoiControlDoInfo.getStrategyMsDoList();
                List<AdxLevelDo> strategyThiMs = StrategyBid.getStrategyInfo(strategyMsDoList, AdxStrategy.ADX_STRATEGY_THI.getCode());

                //策略3-分level-历史30min-ROI,竞价成功率,adx消耗(分*10000/cpm)
                Map<String, Double> roiLevelMs = StrategyBid.getLevelRoi(strategyThiMs);
                Map<String, Double> sucRateLevelMs = StrategyBid.getLevelSucRate(strategyThiMs);
                Map<String, Long> adxConsumeLevelMs = StrategyBid.getLevelIndex(strategyThiMs, AdxIndex.ADX_CONSUME.getCode(), defaultConsume);

                //策略3-历史30min-ROI,竞价成功率,adx消耗
                Double roiMs = StrategyBid.getRoi(strategyThiMs);
                Double sucRateMs = StrategyBid.getSucRate(strategyThiMs);
                Double adxConsumeMs = DataUtil.toDouble(StrategyBid.getSumLevelIndex(strategyThiMs, AdxIndex.ADX_CONSUME.getCode(), defaultConsume))/10000/1000;

                //资源位维度-历史30min-ROI,竞价成功率
                Double resoRoiMs = 0.0;
                Double resoSucRateMs = 0.0;
                Map<String, Long> resoMapInfo = adxRoiControlDoInfo.getResourceIndexMap();
                if (AssertUtil.isNotEmpty(resoMapInfo)) {
                    resoRoiMs = StrategyBid.getRoi(resoMapInfo);
                    resoSucRateMs = StrategyBid.getSucRate(resoMapInfo);
                }


                //4、获取上一次最优level作为基准值
                Double minRoi = StrategyBid.nullToDefault(adxRoiControlDoInfo.getMinRoi(), defaultMinRoi);
                AdxFactorExplorationDo lastFactorDoInfo = adxRoiControlDoInfo.getLastFactorExplorationDo();
                if (AssertUtil.isNotEmpty(lastFactorDoInfo)) {
                    String bestLevel = StrategyBid.selectBestLevel(minRoi, roiLevelMs, sucRateLevelMs);
                    Map<String, Double> lastFactorMap = lastFactorDoInfo.getFactorExploreMap();
                    if (AssertUtil.isNotEmpty(lastFactorMap) && AssertUtil.isNotEmpty(lastFactorMap.get(bestLevel))) {
                        baseValue = lastFactorMap.get(bestLevel);
                    }
                }


                //5、判断基准值是否重置
                //重置--人工设置ROI与上一次ROI不相同; 更新时间为00:00:00-00:01:00
                Double diff = StrategyBid.bucket(Math.abs(roiMs / minRoi - 1.00), bucketDiffList);
                Double lastMinRoi = StrategyBid.nullToDefault(adxRoiControlDoInfo.getLastMinRoi(), defaultMinRoi);
                Long currentTime = DataUtil.string2Long(LocalDateUtil.getCurrentLocalDateTime("HHmm"));

                if (minRoi.compareTo(lastMinRoi) != 0 || currentTime == null || (currentTime >= 0L && currentTime < 1L)) {
                    baseValue = defaultFactor;

                } else {


                    //6、判断基准值是否回归
                    if (adxConsumeMs < adxConsumeMsLimit || sucRateMs < resoSucRateMs * 0.5) {

                        if (baseValue > defaultFactor) {
                            baseValue += -backStep * diff;
                        } else if (baseValue < defaultFactor) {
                            baseValue += backStep * diff;
                        }
                    }
                }

                //7、调整基准流量比例
                if (roiMs > 0.0) {

                    if (roiMs > minRoi) {
                        baseFlowRate[0] += stepFlow2 * diff;
                        baseFlowRate[1] += stepFlow1 * diff;
                        baseFlowRate[3] += -stepFlow1 * diff;
                        baseFlowRate[4] += -stepFlow2 * diff;

                    } else if (roiMs < minRoi) {

                        baseFlowRate[0] += -stepFlow2 * diff;
                        baseFlowRate[1] += -stepFlow1 * diff;
                        baseFlowRate[3] += stepFlow1 * diff;
                        baseFlowRate[4] += stepFlow2 * diff;

                    }
                    baseFlowRate[2] = 1-baseFlowRate[0]-baseFlowRate[1]-baseFlowRate[3]-baseFlowRate[4];
                }

                //更新上一次实际ROI
                for (int i = 1; i < AdxLevel.values().length; i++) {
                    lastRealRoi[i-1] = roiLevelMs.get(DataUtil.Integer2String(i));
                }

            }

            //8、更新基准值和基准流量比例
            //调节因子范围控制
            baseValue = DataUtil.formatDouble(StrategyBid.getNormalValue(baseValue, defaultFactor, lowerLimit, upperLimit),6);

            for (AdxLevel adxLevel : AdxLevel.values()) {
                String key = adxLevel.getCode();
                if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())) {
                    int i = DataUtil.toInt(DataUtil.string2Long(key));
                    Double factor = baseValue + stepList[i-1];
                    Double flowRate = baseFlowRate[i-1];
                    Double lastRoi = lastRealRoi[i-1];

                    factorMap.put(key,factor);
                    flowRateMap.put(key,flowRate);
                    lastRoiMap.put(key, lastRoi);
                }
            }

            ret.setFactorExploreMap(factorMap);
            ret.setFactorFlowRateMap(flowRateMap);
            ret.setLastRealRoiMap(lastRoiMap);


        } catch(Exception e){
            logger.error("FactorExploration.getExploreFactor error:" + e);
        }

        return ret;
    }






    //单元测试
    public static void main(String[] args) {

        try {

            AdxRoiControlDo adxRoiControlDoInfo1 = new AdxRoiControlDo();
            adxRoiControlDoInfo1.setBasePrice(900.0);


            AdxFactorExplorationDo ret2 = FactorExploration.getExploreFactor(adxRoiControlDoInfo1);
            System.out.println("ret2:" + JSON.toJSONString(ret2));


        } catch (Exception e) {
            e.printStackTrace();
        }

    }
}