package cn.com.duiba.nezha.alg.alg.adx;

import cn.com.duiba.nezha.alg.alg.enums.AdxIndex;
import cn.com.duiba.nezha.alg.alg.enums.AdxLevel;
import cn.com.duiba.nezha.alg.alg.enums.AdxStrategy;
import cn.com.duiba.nezha.alg.alg.vo.*;
import cn.com.duiba.nezha.alg.common.util.AssertUtil;
import cn.com.duiba.nezha.alg.common.util.DataUtil;
import com.alibaba.fastjson.JSON;

import java.security.acl.LastOwnerException;
import java.util.*;



public class StrategyBid {


    /**
     * 策略一：实时调节ROI因子，预估出价
     *
     * @param adxDoInfo
     * @param adxRoiFactorDoInfo
     * @param advertConsumeInfo
     * @return level, price
     */
    public static Map<String, String> getStrategyFirBid(AdxDo adxDoInfo,
                                                        AdxRoiFactorDo adxRoiFactorDoInfo,
                                                        Map<String, Long> advertConsumeInfo) {

        // ROI调节因子默认值
        Double defaultRoiFactor = 1.00;

        Map<String, String> retMap = new HashMap<>();
        String level = AdxLevel.ADX_LEVEL_ZER.getCode();
        Long price = AdxBidding.getAdxParPrice(adxDoInfo, defaultRoiFactor, advertConsumeInfo);

        if (AssertUtil.isNotEmpty(adxRoiFactorDoInfo)) {

            Double roiFactor = adxRoiFactorDoInfo.getAdxRoiFactor();
            price = AdxBidding.getAdxParPrice(adxDoInfo, roiFactor, advertConsumeInfo);
        }

        retMap.put("level", level);
        retMap.put("price", DataUtil.Long2String(price));

        return retMap;
    }


    /**
     * 策略二：价格试探，寻找最优出价
     *
     * @param priceExplorationDoInfo
     * @param defaultPrice
     * @return level, price
     */
    public static Map<String, String> getStrategySecBid(AdxPriceExplorationDo priceExplorationDoInfo,
                                                        Long defaultPrice) {


        Map<String, String> retMap = new HashMap<>();
        String level = AdxLevel.ADX_LEVEL_THR.getCode();
        Long price = defaultPrice;

        if (AssertUtil.isNotEmpty(priceExplorationDoInfo)) {

            Map<String, Long> priceMap = priceExplorationDoInfo.getPriceExploreMap();
            Map<String, Double> FlowRateMap = priceExplorationDoInfo.getPriceFlowRateMap();
            Map<String, String> result = flowSplit(FlowRateMap, priceMap, defaultPrice);

            level = result.get("level");
            price = DataUtil.string2Long(result.get("price"));

        }

        retMap.put("level", level);
        retMap.put("price", DataUtil.Long2String(price));

        return retMap;
    }


    /**
     * 策略三：因子试探，寻找最优，预估出价
     *
     * @param adxDoInfo
     * @param factorExplorationDoInfo
     * @param advertConsumeInfo
     * @return level, price
     */
    public static Map<String, String> getStrategyThiBid(AdxDo adxDoInfo,
                                                        AdxFactorExplorationDo factorExplorationDoInfo,
                                                        Map<String, Long> advertConsumeInfo) {


        // ROI调节因子默认值
        Double defaultRoiFactor = 1.00;

        Map<String, String> retMap = new HashMap<>();
        String level = AdxLevel.ADX_LEVEL_THR.getCode();
        Long price = AdxBidding.getAdxParPrice(adxDoInfo, defaultRoiFactor, advertConsumeInfo);

        if (AssertUtil.isNotEmpty(factorExplorationDoInfo)) {

            Map<String, Double> factorMap = factorExplorationDoInfo.getFactorExploreMap();
            Map<String, Double> FlowRateMap = factorExplorationDoInfo.getFactorFlowRateMap();
            Map<String, String> result = flowSplit(FlowRateMap, factorMap, defaultRoiFactor);

            level = result.get("level");
            Double roiFactor = DataUtil.string2Double(result.get("factor"));
            price = AdxBidding.getAdxParPrice(adxDoInfo, roiFactor, advertConsumeInfo);
        }

        retMap.put("level", level);
        retMap.put("price", DataUtil.Long2String(price));

        return retMap;
    }


    /**
     * 策略二：流量分配，返回level和试探出价
     *
     * @param flowRateMap
     * @param priceMap
     * @param defaultValue
     * @return
     */
    public static Map<String, String> flowSplit(Map<String, Double> flowRateMap, Map<String, Long> priceMap, Long defaultValue) {

        Map<String, String> retMap = new HashMap<>();
        String level = AdxLevel.ADX_LEVEL_THR.getCode();
        Long price = defaultValue;

        Double sumRate = 0.00;
        Double[] sumRateList = new Double[AdxLevel.values().length - 1];

        if (AssertUtil.isAllNotEmpty(flowRateMap, priceMap)) {

            for (AdxLevel adxLevel : AdxLevel.values()) {
                String key = adxLevel.getCode();
                if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())){
                    int i = DataUtil.toInt(DataUtil.string2Long(key));
                    sumRate += nullToDefault(flowRateMap.get(key), 0.0);
                    sumRateList[i - 1] = sumRate;
                }
            }

            double random = Math.random();

            for (int i = 0; i < sumRateList.length; i++) {
                double bound = sumRateList[i];

                if (random <= bound) {
                    level = DataUtil.Integer2String(i + 1);
                    price = priceMap.get(level);
                    if (price == null) {
                        price = defaultValue;
                    }
                    break;
                }
            }
        }

        retMap.put("level", level);
        retMap.put("price", DataUtil.Long2String(price));
        return retMap;
    }


    /**
     * 策略三：流量分配，返回level和试探因子
     *
     * @param flowRateMap
     * @param factorMap
     * @param defaultValue
     * @return
     */
    public static Map<String, String> flowSplit(Map<String, Double> flowRateMap, Map<String, Double> factorMap, Double defaultValue) {

        Map<String, String> retMap = new HashMap<>();
        String level = AdxLevel.ADX_LEVEL_THR.getCode();
        Double factor = defaultValue;

        Double sumRate = 0.00;
        Double[] sumRateList = new Double[AdxLevel.values().length - 1];

        if (AssertUtil.isAllNotEmpty(flowRateMap, factorMap)) {

            for (AdxLevel adxLevel : AdxLevel.values()) {
                String key = adxLevel.getCode();
                if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())){
                    int i = DataUtil.toInt(DataUtil.string2Long(key));
                    sumRate += nullToDefault(flowRateMap.get(key), 0.0);
                    sumRateList[i - 1] = sumRate;
                }
            }

            double random = Math.random();

            for (int i = 0; i < sumRateList.length; i++) {
                double bound = sumRateList[i];

                if (random <= bound) {
                    level = DataUtil.Integer2String(i + 1);
                    factor = factorMap.get(level);
                    if (factor == null) {
                        factor = defaultValue;
                    }
                    break;
                }
            }
        }

        retMap.put("level", level);
        retMap.put("factor", DataUtil.double2String(factor));
        return retMap;
    }


    /**
     * 获取指定策略数据
     *
     * @param arrayListInfo
     * @param strategyLabel
     * @return 获取指定策略数据
     */
    public static List<AdxLevelDo> getStrategyInfo(ArrayList<AdxStrategyDo> arrayListInfo, String strategyLabel) {

        List<AdxLevelDo> ret = new ArrayList<>();
        if (AssertUtil.isNotEmpty(arrayListInfo)) {
            for (int i = 0; i < AdxStrategy.values().length; i++) {
                AdxStrategyDo ind = arrayListInfo.get(i);
                if (strategyLabel.equals(ind.getStrategy())) {
                    ret = ind.getLevelDoList();
                }
            }
        }
        return ret;
    }


    /**
     * 合计所有level中相应指标数据
     *
     * @param levelListInfo
     * @param indexLabel
     * @param defaultValue
     * @return 默认值
     */
    public static Long getSumLevelIndex(List<AdxLevelDo> levelListInfo, String indexLabel, Long defaultValue) {

        Long ret = 0L;
        if (AssertUtil.isNotEmpty(levelListInfo)) {
            for (AdxLevelDo levelDo : levelListInfo) {
                if (AssertUtil.isNotEmpty(levelDo)) {
                    Map<String, Long> valueMap = levelDo.getValueMap();

                    if (AssertUtil.isNotEmpty(valueMap)
                            && valueMap.get(indexLabel) != null
                            && valueMap.get(indexLabel) >= 0L) {

                        ret += valueMap.get(indexLabel);
                    }

                } else {
                    ret += defaultValue;
                }
            }
        }
        return ret;
    }


    /**
     * 获取不同level的相应指标数据
     *
     * @param levelListInfo
     * @param indexLabel
     * @param defaultValue
     * @return 默认值
     */
    public static Map<String, Long> getLevelIndex(List<AdxLevelDo> levelListInfo, String indexLabel, Long defaultValue) {

        Map<String, Long> retMap = new HashMap<>();
        Long val = defaultValue;
        String key;
        for (AdxLevel adxLevel : AdxLevel.values()) {
            key = adxLevel.getCode();

            if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())) {
                if (AssertUtil.isNotEmpty(levelListInfo)) {

                    int i = DataUtil.toInt(DataUtil.string2Long(key));
                    AdxLevelDo ind = levelListInfo.get(i);

                    if (AssertUtil.isNotEmpty(ind)) {
                        Map<String, Long> indMap = ind.getValueMap();

                        if (AssertUtil.isNotEmpty(indMap)
                                && indMap.get(indexLabel) != null
                                && indMap.get(indexLabel) >= 0L) {

                            val = indMap.get(indexLabel);
                        }
                    }
                }
                retMap.put(key, val);
            }
        }
        return retMap;
    }


    /**
     * 计算指定策略的竞价成功率
     *
     * @param levelListInfo
     * @return 默认值
     */
    public static Double getSucRate(List<AdxLevelDo> levelListInfo) {

        Double ret = 0.0;
        Long defaultValue = 0L;

        if(AssertUtil.isNotEmpty(levelListInfo)) {
            //竞价量
            Long bidCnt = StrategyBid.getSumLevelIndex(levelListInfo, AdxIndex.BID.getCode(), defaultValue);
            //竞价成功量
            Long sucCnt = StrategyBid.getSumLevelIndex(levelListInfo, AdxIndex.BID_SUC.getCode(), defaultValue);

            ret = DataUtil.division(sucCnt, bidCnt, 6);
        }
        return ret;
    }


    /**
     * 计算创意/资源位整体竞价成功率
     *
     * @param mapInfo
     * @return 默认值
     */
    public static Double getSucRate(Map<String, Long> mapInfo) {

        Double ret = 0.0;
        Long defaultValue = 0L;

        if(AssertUtil.isNotEmpty(mapInfo)){
            //竞价量
            Long bidCnt = StrategyBid.nullToDefault(mapInfo.get(AdxIndex.BID.getCode()), defaultValue);
            //竞价成功量
            Long sucCnt = StrategyBid.nullToDefault(mapInfo.get(AdxIndex.BID_SUC.getCode()), defaultValue);

            ret = DataUtil.division(sucCnt, bidCnt, 6);

        }
        return ret;
    }


    /**
     * 计算不同level的竞价成功率
     *
     * @param levelListInfo
     * @return 默认值
     */
    public static Map<String, Double> getLevelSucRate(List<AdxLevelDo> levelListInfo) {

        Map<String, Double> retMap = new HashMap<>();
        Long defaultValue = 0L;

        for (AdxLevel adxLevel : AdxLevel.values()) {
            String key = adxLevel.getCode();
            if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())) {

                if (AssertUtil.isNotEmpty(levelListInfo)) {
                    //竞价量
                    Map<String, Long> bidCnt = StrategyBid.getLevelIndex(levelListInfo, AdxIndex.BID.getCode(), defaultValue);
                    //竞价成功量
                    Map<String, Long> sucCnt = StrategyBid.getLevelIndex(levelListInfo, AdxIndex.BID_SUC.getCode(), defaultValue);

                    Double ret = DataUtil.division(sucCnt.get(key), bidCnt.get(key), 6);
                    retMap.put(key, ret);

                } else {
                    retMap.put(key, 0.0);
                }
            }
        }
        return retMap;
    }


    /**
     * 计算指定策略的roi
     *
     * @param levelListInfo
     * @return 默认值
     */
    public static Double getRoi(List<AdxLevelDo> levelListInfo) {

        Double ret = 0.0;
        Long defaultValue = 0L;

        if(AssertUtil.isNotEmpty(levelListInfo)){
            //adx消耗(分*10000/cpm)
            Long adxConsume = StrategyBid.getSumLevelIndex(levelListInfo, AdxIndex.ADX_CONSUME.getCode(), defaultValue);
            //广告消耗(分)
            Long advertConsume = StrategyBid.getSumLevelIndex(levelListInfo, AdxIndex.ADVERT_CONSUME.getCode(), defaultValue);

            ret = DataUtil.formatDouble((DataUtil.division(advertConsume, adxConsume, 13) * 10000 * 1000),6);
        }
        return ret;
    }


    /**
     * 计算创意/资源位整体的roi
     *
     * @param mapInfo
     * @return 默认值
     */
    public static Double getRoi(Map<String, Long> mapInfo) {

        Double ret = 0.0;
        Long defaultValue = 0L;

        if (AssertUtil.isNotEmpty(mapInfo)) {
            //adx消耗(分*10000/cpm)
            Long adxConsume = StrategyBid.nullToDefault(mapInfo.get(AdxIndex.ADX_CONSUME.getCode()), defaultValue);
            //广告消耗(分)
            Long advertConsume = StrategyBid.nullToDefault(mapInfo.get(AdxIndex.ADVERT_CONSUME.getCode()), defaultValue);

            ret = DataUtil.formatDouble((DataUtil.division(advertConsume, adxConsume, 13) * 10000 * 1000),6);
        }

        return ret;
    }


    /**
     * 计算不同level的roi
     *
     * @param levelListInfo
     * @return 默认值
     */
    public static Map<String, Double> getLevelRoi(List<AdxLevelDo> levelListInfo) {

        Map<String, Double> retMap = new HashMap<>();
        Long defaultValue = 0L;

        for (AdxLevel adxLevel : AdxLevel.values()) {
            String key = adxLevel.getCode();
            if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())) {

                if (AssertUtil.isNotEmpty(levelListInfo)) {

                    //adx消耗(分*10000/cpm)
                    Map<String, Long> adxConsume = StrategyBid.getLevelIndex(levelListInfo, AdxIndex.ADX_CONSUME.getCode(), defaultValue);
                    //广告消耗(分)
                    Map<String, Long> advertConsume = StrategyBid.getLevelIndex(levelListInfo, AdxIndex.ADVERT_CONSUME.getCode(), defaultValue);
                    Double ret = DataUtil.formatDouble((DataUtil.division(advertConsume.get(key), adxConsume.get(key), 13) * 10000 * 1000),6);
                    retMap.put(key, ret);

                } else {
                    retMap.put(key, 0.0);
                }
            }
        }
        return retMap;
    }


    /**
     * 缺失情况，重置为defaultValue
     *
     * @param value
     * @param defaultValue
     * @return 默认值
     */
    public static Long nullToDefault(Long value, Long defaultValue) {

        Long ret = defaultValue;
        if (value != null && value > defaultValue) {
            ret = value;
        }
        return ret;
    }


    /**
     * 缺失情况，重置为defaultValue
     *
     * @param value
     * @param defaultValue
     * @return 默认值
     */
    public static Double nullToDefault(Double value, Double defaultValue) {

        Double ret = defaultValue;
        if (value != null && value > defaultValue) {
            ret = value;
        }
        return ret;
    }


    /**
     * 获取正常范围内的值
     *
     * @param value
     * @param defaultValue
     * @return 默认值
     */
    public static Double getNormalValue(Double value, Double defaultValue, Double lower, Double upper) {

        Double ret = defaultValue;
        if (value != null) {
            ret = value < lower ? lower : (value > upper ? upper : value);
        }
        return ret;
    }


    /**
     * 分桶函数
     * <p>
     * 左开右闭区间
     * 其他情况下的闭合区间设计需注意！！
     *
     * @param value
     * @param bucketList 不为空，且不含有空值（未判断）
     * @return 分桶值
     */
    public static Double bucket(Double value, Double[] bucketList) {
        Double ret = 0.00;

        if (value != null && bucketList != null && bucketList.length > 0) {
            for (int i = 0; i < bucketList.length; i++) {
                double bound = bucketList[i];
                if (value <= bound) {
                    ret = bound;
                    break;
                }
            }

            if (value > bucketList[bucketList.length - 1]) {
                ret = bucketList[bucketList.length - 1];
            }
        }
        return ret;
    }


    /**
     * 选取最优level
     *
     * @param minRoi
     * @param roiLevel
     * @param sucRateLevel
     * @return level
     */
    public static String selectBestLevel(Double minRoi, Map<String, Double> roiLevel, Map<String, Double> sucRateLevel) {
        String ret = AdxLevel.ADX_LEVEL_THR.getCode();

        if (AssertUtil.isAllNotEmpty(roiLevel, sucRateLevel)) {
            Double tmpRoi = roiLevel.get(ret);
            Double tmpSucRate = sucRateLevel.get(ret);
            for (AdxLevel adxLevel : AdxLevel.values()) {
                String key = adxLevel.getCode();
                if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())) {
                    Double roi = roiLevel.get(key);
                    Double sucRate = sucRateLevel.get(key);

                    if (tmpRoi > minRoi * 0.9) {
                        if (roi > minRoi * 0.9) {
                            if (sucRate > tmpSucRate) {
                                ret = key;
                            }
                        }

                    } else {
                        if (roi > tmpRoi) {
                            ret = key;
                        }
                    }

                    if (roi > tmpRoi) {
                        tmpRoi = roi;
                        tmpSucRate = sucRate;
                    }
                }
            }
        }
        return ret;
    }


    //单元测试
    public static void main(String[] args) {

        try {

            Map<String, Double> roiLevel1 = new HashMap<>();
            roiLevel1.put("1", 0.95);
            roiLevel1.put("2", 1.05);
            roiLevel1.put("3", 1.10);
            roiLevel1.put("4", 1.02);
            roiLevel1.put("5", 1.11);

            Map<String, Double> sucRateLevel1 = new HashMap<>();
            sucRateLevel1.put("1", 0.75);
            sucRateLevel1.put("2", 0.50);
            sucRateLevel1.put("3", 0.86);
            sucRateLevel1.put("4", 0.80);
            sucRateLevel1.put("5", 0.61);

            Map<String, Long> map1 = new HashMap<>();
            map1.put(AdxIndex.ADX_CONSUME.getCode(), 1200*1000*10000L);
            map1.put(AdxIndex.ADVERT_CONSUME.getCode(), 1400L);

            System.out.println("selectBestLevel:" + JSON.toJSONString(selectBestLevel(1.10,null,null)));

            System.out.println("flowSplit:" + JSON.toJSONString(flowSplit(null,null,1.00)));

            System.out.println("getSumLevelIndex:" + JSON.toJSONString(getSumLevelIndex(null,null,1L)));

            System.out.println("getLevelIndex:" + JSON.toJSONString(getLevelIndex(null,null,0L)));

            System.out.println("getLevelSucRate:" + JSON.toJSONString(getLevelSucRate(null)));

            System.out.println("getLevelRoi:" + JSON.toJSONString(getLevelRoi(null)));

            System.out.println("getRoi:" + JSON.toJSONString(getRoi(map1)));

            Double roiFactor = getNormalValue(0.6, 1.00, 0.7, 1.4);
            System.out.println("getNormalValue:" + JSON.toJSONString(roiFactor));

        } catch (Exception e) {
            e.printStackTrace();
        }
    }

}

