package cn.com.duiba.nezha.alg.alg.adx.rcmd;

import cn.com.duiba.nezha.alg.alg.adx.AdxStatData;
import cn.com.duiba.nezha.alg.alg.vo.adx.directly.AdxIndexStatsDo;
import cn.com.duiba.nezha.alg.alg.vo.adx.pd.AdxStatsDo;
import cn.com.duiba.nezha.alg.alg.vo.adx.rcmd.*;
import cn.com.duiba.nezha.alg.common.util.DataUtil;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;
import java.util.stream.Collectors;


public class AdxMaterialRecallAlg {

    private static final Logger logger = LoggerFactory.getLogger(AdxMaterialRecallAlg.class);

    /**
     * ADX素材推荐-召回接口
     * 5min执行1次，存储redis，打印info日志
     * 按 <资源位，素材尺寸比例区间，素材图片类型> 维度维护
     *
     * @param
     * @return
     */
    public static AdxMaterialRecallDo adxMaterialRecall(AdxMaterialRecallReqDo recallRequestDo) {


        Integer tagRecallSize = 10;       //Top标签召回集大小
        Integer recallSize = 30;          //素材单路召回集大小
        Integer multiRecallSize = 50;     //素材多路召回集大小
        AdxMaterialRecallDo ret = new AdxMaterialRecallDo();


        //1. 入参校验
        if (!valid(recallRequestDo)) {
            return ret;
        }


        //2. 试投素材池
        List<Long> materialList = recallRequestDo.getMaterialList();
        Map<Long, AdxStatsDo> resoMaterialStatInfo = Optional.ofNullable(recallRequestDo.getResoMaterialStatInfo()).orElse(new HashMap<>());
        Set<Long> rawList = getRawList(materialList, resoMaterialStatInfo);


        //3. 可投素材池过小，全召回
        if (materialList.size() <= recallSize) {
            ret.setRecallList(new HashSet<>(materialList));
            ret.setMaterialSize(materialList.size());
            ret.setRawList(rawList);
            return ret;
        }


        //4.1 素材指标融合计算：资源位+素材维度
        List<AdxMaterialStatDo> materialStatsList = new ArrayList();
        materialList.forEach(materialId -> {
            AdxStatsDo materialInfo = Optional.ofNullable(resoMaterialStatInfo.get(materialId)).orElse(new AdxStatsDo());
            AdxMaterialStatDo materialStats = getMergeStats(materialId, materialInfo);
            materialStatsList.add(materialStats);
        });

        //4.2 ctr召回
        List<Long> ctrFilterList = materialStatsList.stream()
                .sorted(Comparator.comparing(AdxMaterialStatDo::getWilsonCtr).reversed())
                .limit(recallSize).map(k -> k.getMaterialId()).collect(Collectors.toList());

        //4.3 rpm召回
        List<Long> rpmFilterList = materialStatsList.stream()
                .sorted(Comparator.comparing(AdxMaterialStatDo::getRpm).reversed())
                .limit(recallSize).map(k -> k.getMaterialId()).collect(Collectors.toList());


        //5.1 标签指标计算：资源位+素材标签维度
        List<AdxMaterialTagStatDo> tagStatsList = new ArrayList();
        Map<Long, AdxStatsDo> resoTagStatInfo = Optional.ofNullable(recallRequestDo.getResoTagStatInfo()).orElse(new HashMap<>());
        resoTagStatInfo.forEach((k, v) -> {
            AdxIndexStatsDo last7DayInfo = AdxStatData.getAdxTimeIndex(v, "7day");
            Double wilsonCtr = AdxStatData.getWilsonScore(last7DayInfo.getClickCnt(), last7DayInfo.getExpCnt());
            AdxMaterialTagStatDo tagStats = new AdxMaterialTagStatDo(k, last7DayInfo.getExpCnt(), last7DayInfo.getClickCnt(), wilsonCtr);
            tagStatsList.add(tagStats);
        });

        //5.2 Top标签:资源位维度
        List<AdxMaterialTagStatDo> topTagInfo = tagStatsList.stream()
                .sorted(Comparator.comparing(AdxMaterialTagStatDo::getWilsonCtr).reversed())
                .limit(tagRecallSize).collect(Collectors.toList());
        List<Long> topTagList = topTagInfo.stream().map(AdxMaterialTagStatDo::getTagId).collect(Collectors.toList());

        //5.3 标签召回
        List<AdxMaterialTopTagDo> hitTagInfo = new ArrayList();
        Map<Long, List<Long>> materialTagList = Optional.ofNullable(recallRequestDo.getMaterialTagList()).orElse(new HashMap<>());
        materialList.forEach(materialId -> {
            List<Long> tagList = Optional.ofNullable(materialTagList.get(materialId)).orElse(new ArrayList<>());
            List<Long> intersection = tagList.stream().filter(tag -> topTagList.contains(tag)).collect(Collectors.toList());
            AdxMaterialTopTagDo hitTagDo = new AdxMaterialTopTagDo(materialId, intersection.size(), intersection);
            hitTagInfo.add(hitTagDo);
        });
        List<AdxMaterialTopTagDo> tagFilterList = hitTagInfo.stream()
                .sorted(Comparator.comparing(AdxMaterialTopTagDo::getHitTagCnt).reversed())
                .limit(recallSize).collect(Collectors.toList());


        //6. 多路召回
        Set<Long> recallList = mergeMultiRecallList(multiRecallSize, ctrFilterList, rpmFilterList, tagFilterList);


        ret.setRecallList(recallList);
        ret.setRawList(rawList);
        ret.setMaterialSize(materialList.size());
        ret.setMaterialStatsList(materialStatsList);
        ret.setCtrFilterList(ctrFilterList);
        ret.setRpmFilterList(rpmFilterList);
        ret.setTagFilterList(tagFilterList);
        ret.setTopTagList(topTagInfo);
        return ret;
    }


    /**
     * 校验基础信息是否非法
     *
     * @param
     * @return
     */
    private static Boolean valid(AdxMaterialRecallReqDo recallRequestDo) {

        Boolean ret = true;

        if (recallRequestDo == null) {
            logger.warn("AdxMaterialRecallAlg.adxMaterialRecall() input params valid, params recallRequestDo is null");
            return false;

        } else if (CollectionUtils.isEmpty(recallRequestDo.getMaterialList())) {
            logger.warn("AdxMaterialRecallAlg.adxMaterialRecall() input params valid, params materialList is null, params resourceId is {}, params adxRatioType is {}, params pictureType is {}",
                    recallRequestDo.getResourceId(), recallRequestDo.getAdxRatioType(), recallRequestDo.getPictureType());
            return false;
        }

        return ret;
    }


    /**
     * 统计数据融合
     * (近1h，当天，近3天)
     *
     * @param
     * @return
     */
    private static AdxMaterialStatDo getMergeStats(Long materialId, AdxStatsDo materialInfo) {

        //融合比例：近1h，当天，近3天
        Double r1 = 0.7, r2 = 0.2, r3 = 0.1;

        AdxIndexStatsDo last1HourInfo = AdxStatData.getAdxTimeIndex(materialInfo, "1hour");
        AdxIndexStatsDo last1DayInfo = AdxStatData.getAdxTimeIndex(materialInfo, "1day");
        AdxIndexStatsDo last3DayInfo = AdxStatData.getAdxTimeIndex(materialInfo, "3day");

        Long exp = DataUtil.double2Long(r1 * last1HourInfo.getExpCnt() + r2 * last1DayInfo.getExpCnt() + r3 * last3DayInfo.getExpCnt());
        Long click = DataUtil.double2Long(r1 * last1HourInfo.getClickCnt() + r2 * last1DayInfo.getClickCnt() + r3 * last3DayInfo.getClickCnt());
        Double wilsonCtr = AdxStatData.getWilsonScore(click, exp);

        Long bid = DataUtil.double2Long(r1 * last1HourInfo.getBidCnt() + r2 * last1DayInfo.getBidCnt() + r3 * last3DayInfo.getBidCnt());
        Long adxConsume = DataUtil.double2Long(r1 * last1HourInfo.getAdxConsume() + r2 * last1DayInfo.getAdxConsume() + r3 * last3DayInfo.getAdxConsume());
        Long advertConsume = DataUtil.double2Long(r1 * last1HourInfo.getAdvertConsume() + r2 * last1DayInfo.getAdvertConsume() + r3 * last3DayInfo.getAdvertConsume());
        Double rpm = AdxStatData.getRpm(bid, adxConsume, advertConsume);

        AdxMaterialStatDo ret = new AdxMaterialStatDo(materialId, exp, click, wilsonCtr, bid, adxConsume, advertConsume, rpm);

        return ret;
    }


    /**
     * @description merge多路召回结果
     * @return 返回值说明
     * @date 2020/7/24
     */
    private static Set<Long> mergeMultiRecallList(Integer multiRecallSize,
                                                  List<Long> ctrFilterList,
                                                  List<Long> rpmFilterList,
                                                  List<AdxMaterialTopTagDo> tagFilterList) {
        List<Long> mergeList = new ArrayList<>();
        ctrFilterList.forEach(material -> mergeList.add(material));
        rpmFilterList.forEach(material -> mergeList.add(material));
        tagFilterList.forEach(materialDo -> mergeList.add(materialDo.getMaterialId()));
        Set<Long> ret = mergeList.stream().distinct().limit(multiRecallSize).collect(Collectors.toSet());
        return ret;
    }


    /**
     * 素材试投池
     *
     * @param
     * @return
     */
    public static Set<Long> getRawList(List<Long> materialList, Map<Long, AdxStatsDo> resoMaterialStatInfo) {

        long expLimit = 200L; //资源位+素材 近3天曝光阈值

        List<Long> rawList = new ArrayList<>();
        materialList.forEach(materialId -> {
            AdxStatsDo materialInfo = Optional.ofNullable(resoMaterialStatInfo.get(materialId)).orElse(new AdxStatsDo());
            AdxIndexStatsDo last3DayInfo = AdxStatData.getAdxTimeIndex(materialInfo, "3day");
            if (last3DayInfo.getExpCnt() < expLimit) {
                rawList.add(materialId);
            }
        });
        Set<Long> ret = rawList.stream().collect(Collectors.toSet());
        return ret;
    }
}