package cn.com.duiba.nezha.alg.alg.adx.rtbbid;

import cn.com.duiba.nezha.alg.alg.adx.AdxStatData;
import cn.com.duiba.nezha.alg.alg.enums.AdxIndex;
import cn.com.duiba.nezha.alg.alg.enums.AdxLevel;
import cn.com.duiba.nezha.alg.alg.enums.AdxStrategy;
import cn.com.duiba.nezha.alg.alg.vo.adx.AdxDo;
import cn.com.duiba.nezha.alg.alg.vo.adx.AdxLevelDo;
import cn.com.duiba.nezha.alg.alg.vo.adx.AdxStrategyDo;
import cn.com.duiba.nezha.alg.alg.vo.adx.directly.AdxIndexStatsDo;
import cn.com.duiba.nezha.alg.alg.vo.adx.pd.AdxStatsDo;
import cn.com.duiba.nezha.alg.alg.vo.adx.rtb.CpcControlDo;
import cn.com.duiba.nezha.alg.alg.vo.adx.rtb.CpcControlReqDo;
import cn.com.duiba.nezha.alg.common.util.AssertUtil;
import cn.com.duiba.nezha.alg.common.util.DataUtil;
import cn.com.duiba.nezha.alg.common.util.LocalDateUtil;
import com.alibaba.fastjson.JSON;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

public class CpcControlTask {

    private static final Logger logger = LoggerFactory.getLogger(CpcControlTask.class);

    /**
     * 定时任务–cpc维稳(20min执行1次)
     *
     * @param cpcControlReqDo
     * @return
     */
    public static CpcControlDo getCpcControl(CpcControlReqDo cpcControlReqDo) {

        CpcControlDo ret = new CpcControlDo();

        /**
         *
         * 步骤：
         * 步骤1、获取统计指标
         * 步骤2、判断启动状态
         * 步骤3、获取上次维稳信息
         * 步骤4、更新基准因子
         * 步骤5、调节试探幅度
         * 步骤6、分配试探流量
         *
         */

        try {

            //设置默认值
            Double defaultFactor = 1.0;           //cpc维稳因子默认值
            Double lowerFactor = 0.1;             //cpc维稳因子下限
            Double upperFactor = 1.9;             //cpc维稳因子上限
            Double targetCpc = 1.0;               //目标cpc(分/单次点击)
            Integer tryLabel = 0;                 //冷启动试探标记

            Double baseValue = defaultFactor;           //基准值
            Double[] stepList = {-0.05, 0.00, 0.05};    //试探调节幅度
            Double[] flowRateList = {0.2, 0.6, 0.2};    //试探流量比例
            Double[] diffList = {0.02, 0.05, 0.08, 0.10, 0.15, 0.20, 0.30};  //目标偏差分桶值


            //因子试探表
            Integer len = AdxLevel.values().length - 1;
            Map<String, Double> factorMap = new HashMap<>(len);
            Map<String, Double> flowRateMap = new HashMap<>(len);
            Map<String, Double> cpcMap = new HashMap<>(len);
            Map<String, Double> sucRateMap = new HashMap<>(len);


            //调节主要依据
            Long bidCntMin = 0L, sucCntMin = 0L, clickCntMin = 0L, adxCostMin = 0L, adConsumeMin = 0L;
            Long bidCntDay = 0L, sucCntDay = 0L, clickCntDay = 0L, adxCostDay = 0L, adConsumeDay = 0L;
            Double cpcMin = null, sucMin = null, cpcDay = null, sucDay = null;
            Double conCpcMin = null;



            if (AssertUtil.isNotEmpty(cpcControlReqDo)) {

                //1.获取统计指标
                //资源位维度
                AdxStatsDo resoStats = Optional.ofNullable(cpcControlReqDo.getResoStats()).orElse(new AdxStatsDo());
                AdxIndexStatsDo resoStatsMin = AdxStatData.getAdxTimeIndex(resoStats, "20min");
                //创意维度
                AdxStatsDo ideaStats = Optional.ofNullable(cpcControlReqDo.getIdeaStats()).orElse(new AdxStatsDo());
                AdxIndexStatsDo ideaMinStats = AdxStatData.getAdxTimeIndex(ideaStats, "20min");
                AdxIndexStatsDo ideaDayStats = AdxStatData.getAdxTimeIndex(ideaStats, "1day");
                //策略维度(策略1)
                Map<String, AdxStrategyDo> strategyInfo = cpcControlReqDo.getStrategyLevelStats();
                List<AdxLevelDo> firStrategyInfo = AdxStatData.getStrategyInfo(strategyInfo, AdxStrategy.ADX_STRATEGY_FIR.getCode());
                AdxIndexStatsDo strategyMinStats = AdxStatData.getStrategyIndex(firStrategyInfo, "20min");
                AdxIndexStatsDo strategyDayStats = AdxStatData.getStrategyIndex(firStrategyInfo, "1day");
                //level维度(level=1,2,3)-实时
                Map<String, Long> bidCntMinLevel = AdxStatData.getLevelIndex(firStrategyInfo, AdxIndex.BID.getCode(), "20min", 1L);
                Map<String, Long> clickCntMinLevel = AdxStatData.getLevelIndex(firStrategyInfo, AdxIndex.CLICK.getCode(), "20min", 1L);
                Map<String, Long> adxCostMinLevel = AdxStatData.getLevelIndex(firStrategyInfo, AdxIndex.ADX_CONSUME.getCode(), "20min", 1L);
                Map<String, Long> adConsumeMinLevel = AdxStatData.getLevelIndex(firStrategyInfo, AdxIndex.ADVERT_CONSUME.getCode(), "20min", 1L);
                Map<String, Double> sucRateMinLevel = AdxStatData.getLevelIndex(firStrategyInfo, "sucRate", "20min");
                Map<String, Double> cpcMinLevel = AdxStatData.getLevelIndex(firStrategyInfo, "cpc", "20min");


                //2.判断启动状态：0-正常；1-冷启动；2-暂停投放
                targetCpc = AdxStatData.nullToDefault(cpcControlReqDo.getTargetCpc(), 1.0);
                tryLabel = getTryLabel(strategyMinStats.getBidCnt(), targetCpc, adxCostMinLevel, sucRateMinLevel, cpcMinLevel);



                //3.获取上次维稳信息
                Integer lastTryLabel = 0;
                Double lastTargetCpc = 1.0;
                Double lastBaseValue = defaultFactor;
                Double lastLowValue = defaultFactor + stepList[0];
                Double lastUppValue = defaultFactor + stepList[2];
                Map<String, Double> lastFactorMap = new HashMap<>(len), lastFlowRateMap = new HashMap<>(len);
                String bestLevel = AdxLevel.ADX_LEVEL_TWO.getCode();

                CpcControlDo lastControlInfo = cpcControlReqDo.getCpcControlInfo();
                if (AssertUtil.isNotEmpty(lastControlInfo)) {
                    lastTargetCpc = AdxStatData.nullToDefault(lastControlInfo.getTargetCpc(), 1.0);
                    lastTryLabel = AdxStatData.nullToDefault(lastControlInfo.getTryLabel(), 0);
                    lastFlowRateMap = lastControlInfo.getFlowRateMap();

                    lastFactorMap = lastControlInfo.getFactorMap();
                    if (AssertUtil.isNotEmpty(lastFactorMap)) {
                        lastBaseValue = lastFactorMap.get(AdxLevel.ADX_LEVEL_TWO.getCode());
                        lastLowValue = lastFactorMap.get(AdxLevel.ADX_LEVEL_ONE.getCode());
                        lastUppValue = lastFactorMap.get(AdxLevel.ADX_LEVEL_THR.getCode());
                    }

                    // 最优level：根据每千次竞价收益
                    bestLevel = AdxStatData.selectCpcBestLevel(bidCntMinLevel, clickCntMinLevel, adxCostMinLevel, targetCpc);
                }


                //4.更新基准因子
                Double conCpc = AdxStatData.getConCpc(strategyMinStats, strategyDayStats, targetCpc, 0.7);
                Double conCpcDiff = DataUtil.division(conCpc, targetCpc, 3);
                Map<String, Double> conCpcMinLevel = AdxStatData.getConCpcLevel(clickCntMinLevel, adxCostMinLevel, targetCpc);
                cpcMap = conCpcMinLevel; conCpcMin = conCpc;

                Long currentTime = DataUtil.string2Long(LocalDateUtil.getCurrentLocalDateTime("HHmm"));
                if (targetCpc.compareTo(lastTargetCpc) != 0
                        || currentTime == null || (currentTime >= 0L && currentTime < 20L)
                        || !strategyDayStats.getConfident()) {

                    //重置：目标cpc变更; 次日0点; 全天数据不置信;
                    baseValue = defaultFactor;

                } else {

                    //更新：计算调节系数
                    Double adjustCoeff = getAdjustCoeff(lastBaseValue, bestLevel, conCpcDiff,  diffList, lastFactorMap, strategyMinStats.getSucRate());
                    baseValue = getBaseValue(tryLabel, lastTryLabel, adjustCoeff, lastBaseValue, lastUppValue, lowerFactor, upperFactor);
                }



                //5.调节试探幅度
                stepList = getStep(tryLabel, targetCpc, baseValue, lastLowValue, lastUppValue, conCpcDiff, conCpcMinLevel, strategyMinStats.getAdxConsume(), adxCostMinLevel);


                //6.分配试探流量
                flowRateList = getFlowRate(tryLabel, targetCpc, conCpcDiff, conCpcMinLevel, adxCostMinLevel, lastFlowRateMap);



                bidCntMin = strategyMinStats.getBidCnt();
                sucCntMin = strategyMinStats.getSucCnt();
                clickCntMin = strategyMinStats.getClickCnt();
                adxCostMin = strategyMinStats.getAdxConsume();
                adConsumeMin = strategyMinStats.getAdvertConsume();
                sucMin = strategyMinStats.getSucRate();
                cpcMin = strategyMinStats.getCpc();

                bidCntDay = strategyDayStats.getBidCnt();
                sucCntDay = strategyDayStats.getSucCnt();
                clickCntDay = strategyDayStats.getClickCnt();
                adxCostDay = strategyDayStats.getAdxConsume();
                adConsumeDay = strategyDayStats.getAdvertConsume();
                sucDay = strategyDayStats.getSucRate();
                cpcDay = strategyDayStats.getCpc();

            }

            // 封装分level调节信息
            for (AdxLevel adxLevel : AdxLevel.values()) {
                String key = adxLevel.getCode();
                if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())) {
                    int i = Integer.valueOf(key);
                    Double factor = AdxStatData.getNormalValue((baseValue + stepList[i - 1]), defaultFactor, lowerFactor, upperFactor,6);
                    Double flowRate = DataUtil.formatDouble(flowRateList[i - 1], 3);

                    factorMap.put(key, factor);
                    flowRateMap.put(key, flowRate);
                }
            }

            ret.setFactorMap(factorMap);
            ret.setFlowRateMap(flowRateMap);
            ret.setTryLabel(tryLabel);
            ret.setTargetCpc(targetCpc);
            ret.setCpcMap(cpcMap);
            ret.setConCpcMin(conCpcMin);

            ret.setBidCntMin(bidCntMin);
            ret.setSucCntMin(sucCntMin);
            ret.setClickCntMin(clickCntMin);
            ret.setAdxCostMin(adxCostMin);
            ret.setAdConsumeMin(adConsumeMin);
            ret.setSucMin(sucMin);
            ret.setCpcMin(cpcMin);

            ret.setBidCntDay(bidCntDay);
            ret.setSucCntDay(sucCntDay);
            ret.setClickCntDay(clickCntDay);
            ret.setAdxCostDay(adxCostDay);
            ret.setAdConsumeDay(adConsumeDay);
            ret.setSucDay(sucDay);
            ret.setCpcDay(cpcDay);


        } catch (Exception e) {
            logger.error("RtbBidAlg.getCpcControl error", e);
        }
        return ret;
    }




    /**
     * 判断启动状态tryLabel：0-正常；1-冷启动；2-暂停投放
     *
     * @param
     * @return
     */
    private static Integer getTryLabel(Long bidCnt,
                                       Double targetCpc,
                                       Map<String, Long> adxCostLevel,
                                       Map<String, Double> sucRateLevel,
                                       Map<String, Double> cpcLevel) {

        Integer ret = 0;
        Long baseAdxCost = 0L, uppAdxCost = 0L;
        Double baseSucRate = null, uppSucRate = null;
        Double baseCpc = null;

        if (AssertUtil.isAllNotEmpty(sucRateLevel, adxCostLevel, cpcLevel)) {
            baseAdxCost = AdxStatData.nullToDefault(adxCostLevel.get(AdxLevel.ADX_LEVEL_TWO.getCode()), 0L);
            uppAdxCost = AdxStatData.nullToDefault(adxCostLevel.get(AdxLevel.ADX_LEVEL_THR.getCode()), 0L);
            baseSucRate = sucRateLevel.get(AdxLevel.ADX_LEVEL_TWO.getCode());
            uppSucRate = sucRateLevel.get(AdxLevel.ADX_LEVEL_THR.getCode());
            baseCpc = cpcLevel.get(AdxLevel.ADX_LEVEL_TWO.getCode());
        }

        if (bidCnt.equals(0L)) {
            //实时竞价量级为0:暂停投放
            ret = 2;

        } else if ((AssertUtil.isEmpty(uppSucRate) || uppSucRate < 0.02) && uppAdxCost < 200L) {
            //试探高价组数据不置信
            ret = 1;

        } else if ((AssertUtil.isEmpty(baseSucRate) || baseSucRate < 0.02) && baseAdxCost < 200L) {
            if ((AssertUtil.isEmpty(baseCpc) || (baseCpc > targetCpc * 1.05 || baseCpc < targetCpc * 0.95))) {
                //基准组数据不置信
                ret = 1;
            }
        }

        return ret;
    }



    /**
     * 计算基准因子调节系数
     * 最优调控+维稳调控
     *
     * @param
     * @return
     */
    private static Double getAdjustCoeff(Double lastBaseValue, String bestLevel, Double diff, Double[] diffList,
                                         Map<String, Double> lastFactorMap, Double sucRateMs) {

        Double ret = 1.0;

        //最优调控-增量1
        lastFactorMap = Optional.ofNullable(lastFactorMap).orElse(new HashMap<>());
        Double lastBestValue = AdxStatData.nullToDefault(lastFactorMap.get(bestLevel), lastBaseValue);
        Double incre1 = DataUtil.division(lastBestValue, lastBaseValue, 3);
        incre1 = AdxStatData.getNormalValue(incre1, 1.00, 0.90, 1.10);

        //最优调控-权重1
        Double weight1 =  Math.abs(diff - 1.0) < 0.1 ? 1.0 : (Math.abs(diff - 1.0) < 0.2 ? 0.8 : ((Math.abs(diff - 1.0) < 0.4 ? 0.5 : 0.0)));
        if (AssertUtil.isEmpty(sucRateMs) || sucRateMs < 0.02) { weight1 = 0.0; }

        //维稳调控-增量2: 1/(实际cpc/(目标cpc*0.98)) = diff/0.98
        Double incre2 = DataUtil.division(0.98, diff, 3);
        Double remainStableLimit = diff < 0.9 ? 0.30 : (diff < 1.1 ? 0.20 : 0.30);
        incre2 = AdxStatData.getNormalValue(incre2, 1.00, (1 - remainStableLimit), (1 + remainStableLimit));

        //维稳调控-权重2
        Double weight2 = 1.00;
        if (AssertUtil.isEmpty(sucRateMs) || sucRateMs < 0.02) { weight2 = 0.0; }


        //计算综合调节系数
        ret = 1 + (incre1-1) * weight1 *0.5 + (incre2-1) * weight2 * 0.5;
        if (weight1 == 0.0) { ret = 1 + (incre2-1) * weight2; }

        Double range = AdxStatData.bucket(Math.abs(diff - 1.0), diffList);
        ret = DataUtil.formatDouble(AdxStatData.getNormalValue(ret, 1.0, (1.0 - range), (1.0 + range)),6);

        return ret;

    }



    /**
     * 更新基准因子
     *
     * @param
     * @return
     */
    private static Double getBaseValue(Integer tryLabel, Integer lastTryLabel, Double coeff,
                                       Double lastBaseValue, Double lastUppValue,
                                       Double lowerLimit, Double upperLimit) {

        Double defaultFactor = 1.0;
        Double ret = defaultFactor;

        if (tryLabel.equals(2)) {
            //暂停投放：基准值回归至默认值
            ret -= (lastBaseValue - defaultFactor) * 0.20;

        } else if (tryLabel.equals(1)) {
            //冷启动：基准值不更新
            ret = lastBaseValue;

        } else if (tryLabel.equals(0)) {

            if (lastTryLabel.equals(1)) {
                //冷启动跳转正常：更新，使用试探高价组因子
                ret = lastUppValue;

            } else {
                //正常：更新，基于上次基准值和调节系数
                ret = lastBaseValue * coeff;

            }
        }

        ret = AdxStatData.getNormalValue(ret, defaultFactor, lowerLimit, upperLimit,6);
        return ret;
    }



    /**
     * 分配试探流量
     *
     * @param
     * @return
     */
    private static Double[] getStep(Integer tryLabel, Double targetCpc,
                                    Double baseValue, Double lastLowValue, Double lastUppValue,
                                    Double diff, Map<String, Double> cpcLevel,
                                    Long adxCostMin, Map<String, Long> adxCostLevel) {

        Double[] retList = {-0.05, 0.00, 0.05};
        Double defaultFactor = 1.0;

        if (tryLabel.equals(2)) {

            //暂停投放：均回归至默认值
            Double uppDiff = lastUppValue - defaultFactor;
            retList[2] = (lastUppValue - uppDiff * 0.20) - baseValue;

            Double lowDiff = lastLowValue - defaultFactor;
            retList[0] = (lastLowValue - lowDiff * 0.20) - baseValue;

        } else if (tryLabel.equals(1)) {

            //step1：根据试探高价组的消耗决定
            Double step1 = 0.0, step1Limit = 0.1;
            Long tryAdxCost = adxCostLevel.get(AdxLevel.ADX_LEVEL_THR.getCode());
            if (tryAdxCost <= 200L) {
                step1 = (1.0 - DataUtil.division(tryAdxCost, 200L)) * step1Limit;
            } else {
                step1 = Math.max(1.0 - DataUtil.division(tryAdxCost, 200L), -1) * step1Limit;
            }
            step1 = DataUtil.formatDouble(AdxStatData.getNormalValue(step1,0.0, -step1Limit, step1Limit),3);

            //step2：根据试探高价组的cpc偏差决定
            Double uppDiff = DataUtil.division(cpcLevel.get(AdxLevel.ADX_LEVEL_THR.getCode()), targetCpc, 3);
            Double step2 = uppDiff < 0.8 ? 0.1 : (uppDiff < 0.95 ? 0.07 : (uppDiff < 1.05 ? 0.05 : (uppDiff < 1.20 ? -0.07 : -0.1)));

            //无消耗时，固定试探高价幅度
            if (adxCostMin.equals(0L)) { step1 = 0.1; step2 = 0.1; }

            //冷启动：高价试探幅度增大，低价试探幅度不变
            retList[2] = Math.max(lastUppValue - baseValue + (0.4 * step1 + 0.6 * step2), 0.0);
            retList[0] = lastLowValue - baseValue;


        } else if (tryLabel.equals(0)) {

            //正常：根据目标偏离程度，调节幅度
            retList[2] += diff < 0.7 ? 0.05 : (diff < 0.8 ? 0.03 : (diff < 0.9 ? 0.00 : (diff < 0.95 ? -0.02 : -0.03)));
            retList[0] -= diff < 1.05 ? -0.03 : (diff < 1.1 ? -0.02 : (diff < 1.2 ? 0.00 : (diff < 1.3 ? 0.03 : 0.05)));

        }

        retList[2] = Math.max(retList[2], retList[1]);
        retList[0] = Math.min(retList[0], retList[1]);

        return retList;
    }




    /**
     * 分配试探流量
     *
     * @param
     * @return
     */
    private static Double[] getFlowRate(Integer tryLabel, Double targetCpc, Double diff,
                                        Map<String, Double> cpcLevel,
                                        Map<String, Long> adxCostLevel,
                                        Map<String, Double> lastFlowRateMap) {

        Double[] retList = new Double[3];
        retList[0] = 0.2; //试探低价组占比
        retList[2] = 0.2; //试探高价组占比

        Double tryDiff = 1.0, lastTryRate = 0.2;
        Long tryAdxCost = 0L;
        if (AssertUtil.isAllNotEmpty(cpcLevel, adxCostLevel, lastFlowRateMap)) {
            tryDiff = AdxStatData.nullToDefault(DataUtil.division(cpcLevel.get(AdxLevel.ADX_LEVEL_THR.getCode()), targetCpc, 3),1.0);
            tryAdxCost = AdxStatData.nullToDefault(adxCostLevel.get(AdxLevel.ADX_LEVEL_THR.getCode()), 0L);
            lastTryRate = AdxStatData.nullToDefault(lastFlowRateMap.get(AdxLevel.ADX_LEVEL_THR.getCode()), 0.2);
        }

        if (tryLabel.equals(2)) {
            retList[2] = 0.2;
            retList[0] = 0.2;


        } else if (tryLabel.equals(1)) {
            retList[2] = AdxStatData.getTryFlowRate(tryDiff, tryAdxCost, lastTryRate);
            retList[0] = retList[2] > 0.8 ? 0.0 : (retList[2] > 0.7 ? 0.1 : 0.2);


        } else if (tryLabel.equals(0) && AssertUtil.isNotEmpty(diff)) {

            if (diff > 1.1) {
                retList[2] = 0.1;
                retList[0] = diff > 1.2 ? 0.3 : 0.2;

            } else if (diff < 0.9) {
                retList[2] = diff < 0.8 ? 0.3 : 0.2;
                retList[0] = 0.1;

            } else if (diff > 0.95 && diff < 1.05) {
                retList[2] = 0.10;
                retList[0] = 0.10;

            } else {
                retList[2] = 0.15;
                retList[0] = 0.15;
            }

        }

        //基准组占比
        retList[1] = DataUtil.formatDouble((1.0 - retList[0] - retList[2]),3);

        return retList;
    }






    //单元测试
    public static void main(String[] args) {

        try {

            System.out.println("getTryLabel:" + JSON.toJSONString(getTryLabel(10L,1.0,null, null, null)));

        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}
