package cn.com.duiba.nezha.alg.alg.adxhd.parse;

import cn.com.duiba.nezha.alg.alg.adxhd.stat.AHStatCalcDto;
import cn.com.duiba.nezha.alg.alg.adxhd.stat.AHStatDto;
import cn.com.duiba.nezha.alg.alg.adxhd.stat.AHSubStatDto;
import cn.com.duiba.nezha.alg.common.util.AssertUtil;
import cn.com.duiba.nezha.alg.common.util.DataUtil;
import cn.com.duiba.nezha.alg.common.util.MathUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

public class AHStatParse extends BaseParse {
    private static final Logger logger = LoggerFactory.getLogger(AHStatParse.class);
    private static final Map<Long, AHStatDto> nullMap = new HashMap<>();

    private static final AHSubStatDto nullDto = new AHSubStatDto();


    public static AHStatCalcDto parse(AHStatDto ahStatDo, Double targetRoi) throws Exception {


        AHStatCalcDto ahStatCalcDto = stat(ahStatDo);

        if (AssertUtil.isEmpty(ahStatDo)) {
            return ahStatCalcDto;
        }

        Double weight = ahStatCalcDto.getWeight();

        /**
         * ADX入口素材点击率
         * sCtr
         * 计算更新
         */
        Boolean sCtrConf = ahStatCalcDto.getAdxExpPv() > 200;
        Double sCtr = sCtrConf ? DataUtil.division(ahStatCalcDto.getAdxClickPv(), ahStatCalcDto.getAdxExpPv(), 5) : null;
        ahStatCalcDto.setSCtr(sCtr);
        ahStatCalcDto.setSCtrConf(sCtrConf);

        /**
         * ADX点击目标转化率
         * sCvr
         * 计算更新
         */
        Boolean sCvrConf = ahStatCalcDto.getAdxClickPv() > 5000;
        Double sCvr = sCvrConf ? DataUtil.division(ahStatCalcDto.getAdConvertPv(), ahStatCalcDto.getAdxClickPv(), 5) : null;
        ahStatCalcDto.setSCvr(sCvr);
        ahStatCalcDto.setSCvrConf(sCvrConf);

        /**
         * UV点券收益
         * 计算更新
         */
        Double uvCost = sCvrConf ? DataUtil.division(ahStatCalcDto.getAdTargetCost(), ahStatCalcDto.getAdxClickPv(), 5) : null;
        ahStatCalcDto.setUvCost(uvCost);


        /**
         * 实际ROI
         * 计算更新
         */
        Boolean sRoiConf = ahStatCalcDto.getAdxCost() > 20000;
        Double sRoi = sRoiConf ? DataUtil.division(ahStatCalcDto.getAdTargetCost(), ahStatCalcDto.getAdxCost(), 2) : null;
        ahStatCalcDto.setSRoi(sRoi);
        ahStatCalcDto.setSRoiConf(sRoiConf);

        /**
         * 平均成本控制因子
         * 计算更新
         */
        Double cFactoAvg = sRoiConf ? DataUtil.division(ahStatCalcDto.getCFactor(), ahStatCalcDto.getAdxExpPv(), 2) : null;
        ahStatCalcDto.setCFactoAvg(cFactoAvg);

        /**
         * 成本控制因子
         * 计算: 平均因子 * 实际Roi/目标Roi
         */
        targetRoi = targetRoi != null ? targetRoi : 1.1;
        Double cFactor = sRoiConf ? cFactoAvg * DataUtil.division(sRoi, targetRoi, 2) : 1.0;
        ahStatCalcDto.setCFactor(cFactor);


        /**
         * 特征计算
         */
        Map<String, String> featureMap = new HashMap<>(8);

        if (AssertUtil.isNotEmpty(ahStatDo.getAppPkgName())) {
            // 特征 创意计划+媒体包维度
            putMap(featureMap, "fas_1004", bucket(ahStatCalcDto.getSCtr(), ctrBucket));//CTR
            putMap(featureMap, "fas_1005", bucket(ahStatCalcDto.getSCvr(), cvrBucket));//CVR
            putMap(featureMap, "fas_1006", bucket(ahStatCalcDto.getUvCost(), uvCostBucket));//uvCost
        } else {
            // 特征 创意计划维度
            putMap(featureMap, "fas_1001", bucket(ahStatCalcDto.getSCtr(), ctrBucket));//CTR
            putMap(featureMap, "fas_1002", bucket(ahStatCalcDto.getSCvr(), cvrBucket));//CVR
            putMap(featureMap, "fas_1003", bucket(ahStatCalcDto.getUvCost(), uvCostBucket));//uvCost
        }

        ahStatCalcDto.setFeatureMap(featureMap);

        return ahStatCalcDto;

    }


    private static AHStatCalcDto stat(AHStatDto ahStatDo) {
        AHStatCalcDto ahStatCalcDto = new AHStatCalcDto();

        /**
         * 近20分钟统计数据
         */
        double w1 = acc(ahStatCalcDto, ahStatDo.getLast20MinStat(), 2000, 3.0, 0.1, 6.0);

        /**
         * 近1小时统计数据
         */
        double w2 = acc(ahStatCalcDto, ahStatDo.getLast1HourStat(), 2000, 2.0, 0.2, 4.0);

        /**
         * 近3小时统计数据
         */
        double w3 = acc(ahStatCalcDto, ahStatDo.getLast3HourStat(), 2000, 1.5, 1.0, 3.0);

        /**
         * 近1天统计数据
         */
        double w4 = acc(ahStatCalcDto, ahStatDo.getLast1DayStat(), 2000, 1.0, 1.0, 1.0);

        /**
         * 近3天统计数据
         */
        double w5 = acc(ahStatCalcDto, ahStatDo.getLast3DayStat(), 2000, 0.3, 0.3, 0.3);

        return ahStatCalcDto;
    }


    /**
     * @param cDto           v1
     * @param sDto           v2
     * @param baseAdxClickPv 置信基准量级
     * @param ratio          权重系数
     * @param minWeight      权重下限
     * @param maxWeight      权重上限
     * @return
     */
    private static double acc(AHStatCalcDto cDto, AHSubStatDto sDto, int baseAdxClickPv, double ratio, double minWeight, double maxWeight) {
        sDto = Optional.ofNullable(sDto).orElse(nullDto);

        double weight = 1.0;
        Long adxClickPv = sDto.getAdxClickPv();
        if (adxClickPv != null) {
            weight = MathUtil.formatDouble(ratio * (adxClickPv + 0.01) / baseAdxClickPv, 5);
        }

        if (weight <= minWeight) {
            weight = minWeight;
        }
        if (weight >= maxWeight) {
            weight = maxWeight;
        }

        System.out.println("w=" + weight);
        cDto.setWeight(add(cDto.getWeight(), weight, 1.0));

        /**
         * adx竞价成功量级
         */
        cDto.setAdxSucPv(add(cDto.getAdxSucPv(), sDto.getAdxSucPv(), weight));


        /**
         * adx曝光量级
         */
        cDto.setAdxExpPv(add(cDto.getAdxExpPv(), sDto.getAdxExpPv(), weight));

        /**
         * adx点击量级
         */
        cDto.setAdxClickPv(add(cDto.getAdxClickPv(), sDto.getAdxClickPv(), weight));

        /**
         * 广告位访问量级
         */
        cDto.setSlotRequestPV(add(cDto.getSlotRequestPV(), sDto.getSlotRequestPV(), weight));

        /**
         * 广告曝光次数
         */
        cDto.setAdExpPv(add(cDto.getAdExpPv(), sDto.getAdExpPv(), weight));

        /**
         * 广告计费点击数
         */
        cDto.setAdClickPv(add(cDto.getAdClickPv(), sDto.getAdClickPv(), weight));

        /**
         * 广告目标转化PV
         */
        cDto.setAdConvertPv(add(cDto.getAdConvertPv(), sDto.getAdConvertPv(), weight));

        /**
         * 累计出价因子factor（口径：曝光量级）
         */
        cDto.setFactorAcc(add(cDto.getFactorAcc(), sDto.getFactorAcc(), weight));

        /**
         * 累计预估cvr（口径：曝光量级）
         */
        cDto.setPCvrAcc(add(cDto.getPCvrAcc(), sDto.getPCvrAcc(), weight));

        /**
         * 累计预估ctr（口径：曝光量级）
         */
        cDto.setPCtrAcc(add(cDto.getPCtrAcc(), sDto.getPCtrAcc(), weight));

        /**
         * 累计出价（口径：曝光量级）
         */
        cDto.setPriceAcc(add(cDto.getPriceAcc(), sDto.getPriceAcc(), weight));

        /**
         * adx消耗(分)
         */
        cDto.setAdxCost(add(cDto.getAdxCost(), sDto.getAdxCost(), weight));

        /**
         * 广告理论消耗(分)
         */
        cDto.setAdTargetCost(add(cDto.getAdTargetCost(), sDto.getAdTargetCost(), weight));

        return weight;
    }

    private static <T extends Number> double add(Double v1, T v2, double v2Weight) {
        double ret = 0.0;
        if (v1 != null) {
            ret += v1;
        }
        if (v2 != null) {
            ret += v2.doubleValue() * v2Weight;
        }
        return ret;
    }


}
