package cn.com.duiba.nezha.alg.alg.adx.rcmd;

import cn.com.duiba.nezha.alg.alg.base.MathBase;
import cn.com.duiba.nezha.alg.alg.base.Roulette;
import cn.com.duiba.nezha.alg.alg.vo.adx.AdxIdeaParamsDo;
import cn.com.duiba.nezha.alg.alg.vo.adx.AdxIdeaStatDo;
import cn.com.duiba.nezha.alg.alg.vo.adx.AdxResourceRcmdDo;
import cn.com.duiba.nezha.alg.common.enums.DateStyle;
import cn.com.duiba.nezha.alg.common.util.AssertUtil;
import cn.com.duiba.nezha.alg.common.util.DataUtil;
import cn.com.duiba.nezha.alg.common.util.LocalDateUtil;
import cn.com.duiba.nezha.alg.common.util.MathUtil;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class AdxIdeaBaseRcmder {

    private static long ADX_MULTIPLIER = 10000000;

    /**
     * ctr权重分配
     */
    static double[] ctrBucket = {0.5, 0.7, 0.8, 0.90, 0.95, 0.97, 1};
    static double[] ctrWeight = {0.5, 1, 2, 5, 10, 50, 100};

    /**
     * pcpm权重分配1 top1>0
     */
    static double[] pCpmBucket = {0.5, 0.7, 0.8, 0.90, 0.95, 0.97, 1};
    static double[] pCpmWeight = {0.5, 1, 2, 5, 10, 50, 100};

    /**
     * pcpm权重分配1 top1<0
     */
    static double[] pCpmBucket2 = {1, 1.1, 1.5, 2, 3};
    static double[] pCpmWeight2 = {100, 20, 5, 1, 0.5};


    /**
     * 更新间隔权重分配
     */
    static double[] updateDelayBucket = {60, 60 * 6, 60 * 24, 60 * 24 * 3};
    static double[] updateDelayWeight = {0.3, 0.5, 0.7, 1.0};


    /**
     * 权重更新
     * 对长时间未更新特征，加大学习步长
     *
     * @param oldWeight
     * @param newWeight
     * @param lastUpdateTime
     * @return
     */
    public static Double getUpdateWeight(Double oldWeight, Double oldWeightSum, Double newWeight, Double newWeightSum, String lastUpdateTime) {

        Double ret = null;


        if (oldWeight != null) {
            // 有旧模型融合

            Long delayMinutes = LocalDateUtil.getIntervalMinutes(lastUpdateTime, DateStyle.YYYYMMDDHHMMSS);
            Double learnRatio = MathBase.getConfidenceWeight(delayMinutes, updateDelayBucket, updateDelayWeight, 0.3);

            Double newWeight2 = newWeight * 100 / newWeightSum;
            Double oldWeight2 = oldWeight * 100 / oldWeightSum;

            ret = (1 - learnRatio) * oldWeight2 + learnRatio * newWeight2;
        } else {
            //无旧模型，赋值
            ret = newWeight * 100 / newWeightSum;
        }


        return MathUtil.formatDouble(ret, 2);
    }


    /**
     * 权重融合
     *
     * @param minWeight
     * @param dayWeight
     * @return
     */
    public static Double getMergeWeight(Double minWeight, Double dayWeight, Long historyTimes, Long currentTimes, Long size, Double currentWeightRato) {
        Double ret = null;

        // 历史无数据 && 实时有数据，实时为准
        if (dayWeight == null && minWeight != null) {
            ret = minWeight;
        }

        // 历史有数据 && 实时无数据，历史位准
        if (dayWeight != null && minWeight == null) {
            ret = dayWeight;
        }

        // 历史无数据 && 实时无数据，默认
        if (dayWeight == null && minWeight == null) {
            if (currentTimes == null || currentTimes <= 5) {

                Double limit = 10.0;

                if (historyTimes != null && historyTimes > 5) {
                    limit = 5.0;
                }
                ret = Math.min(100.0 / size, limit);

            } else {
                ret = 1.0;
            }

        }
        // 历史有数据 && 实时有数据，融合（时间维度）
        if (dayWeight != null && minWeight != null) {

            ret = (1 - currentWeightRato) * dayWeight + currentWeightRato * minWeight;
        }
//            System.out.println("dayWeight=" + dayWeight + "，minWeight=" + minWeight + ",mergeWeight=" + ret);
        return ret;
    }

    /**
     * 权重分配，根据CTR、PCPM
     *
     * @param adxIdeaStatDo
     * @param adxBestIdeaStatDo
     * @return
     */
    public static Double getIdeaWeight(AdxIdeaStatDo adxIdeaStatDo, AdxIdeaStatDo adxBestIdeaStatDo) {
        Double ret = null;
        // 数据不置信(创意、最优组)
        if (adxIdeaStatDo == null || !adxIdeaStatDo.getConfidence() || adxBestIdeaStatDo == null) {
            return ret;
        }


        Double bestCtr = adxBestIdeaStatDo.getCtr();
        Double bestProfitEcpm = adxBestIdeaStatDo.getProfitEcpm();

        Double ideaCtr = adxIdeaStatDo.getCtr();
        Double ideaProfitEcpm = adxIdeaStatDo.getProfitEcpm();

        boolean ideaConfidence = adxIdeaStatDo.getConfidence();

        Double ctrWeight = getCtrWeight(ideaCtr, bestCtr, ideaConfidence);
        Double profitCpmWeight = getProfitEcpmWeight(ideaProfitEcpm, bestProfitEcpm, ideaConfidence);


        if (ctrWeight != null && profitCpmWeight != null) {


            Double newCtrWeight = MathBase.noiseSmoother(ctrWeight, 0.6 * profitCpmWeight, 1.2 * profitCpmWeight);

            ret = 0.2 * newCtrWeight + 0.8 * profitCpmWeight;


        }
//        System.out.println("ctrWeight="+ctrWeight+"，profitCpmWeight="+profitCpmWeight);
        return ret;
    }


    /**
     * @param ctr
     * @param bestCtr
     * @param isConfidence
     * @return
     */
    public static Double getCtrWeight(Double ctr, Double bestCtr, boolean isConfidence) {
        Double ret = null;
        if (bestCtr != null && isConfidence && ctr != null) {
            Double ratio = DataUtil.division(ctr, bestCtr, 3);
            ret = MathBase.getConfidenceWeight(Math.min(ratio, 1.0), ctrBucket, ctrWeight);
        }
        return ret;
    }

    /**
     * @param pCpm
     * @param bestPCpm
     * @param isConfidence
     * @return
     */
    public static Double getProfitEcpmWeight(Double pCpm, Double bestPCpm, boolean isConfidence) {
        Double ret = null;

        if (pCpm != null && isConfidence && bestPCpm != null) {

            if (bestPCpm >= 0) {
                Double ratio = DataUtil.division(pCpm, bestPCpm + 0.000001, 3);
                ret = MathBase.getConfidenceWeight(Math.min(ratio, 1.0), pCpmBucket, pCpmWeight);
            } else {
                Double ratio = DataUtil.division(pCpm, bestPCpm, 3);
                ret = MathBase.getConfidenceWeight(Math.max(ratio, 1.0), pCpmBucket2, pCpmWeight2);
            }
        }

        return ret;
    }

    /**
     * 计算最大CTR && PECPM
     *
     * @param adxStatDoMap
     * @return
     */
    public static AdxIdeaStatDo getBestCtrAndProfitEcpm(List<Long> ideaList, Map<Long, AdxIdeaStatDo> adxStatDoMap) {
        AdxIdeaStatDo ret = new AdxIdeaStatDo();
        if (AssertUtil.isEmpty(adxStatDoMap)) {
            return null;
        }

        for (int i = 0, size = ideaList.size(); i < size; i++) {
            Long ideaId = ideaList.get(i);
            AdxIdeaStatDo adxIdeaStatDo = adxStatDoMap.get(ideaId);

//            System.out.println(JSON.toJSONString(adxIdeaStatDo));

            if (adxIdeaStatDo != null && adxIdeaStatDo.getConfidence()) {

                if (ret.getCtr() == null || ret.getCtr() < adxIdeaStatDo.getCtr()) {
                    ret.setCtr(adxIdeaStatDo.getCtr());
                }

                if (ret.getProfitEcpm() == null || ret.getProfitEcpm() < adxIdeaStatDo.getProfitEcpm()) {
                    ret.setProfitEcpm(adxIdeaStatDo.getProfitEcpm());
                }

            }
        }
//        System.out.println(JSON.toJSONString(ret));

        return ret;
    }


    /**
     * 置信、指标计算
     *
     * @param adxIdeaStatDo
     */
    public static void ideaStatCompute(AdxIdeaStatDo adxIdeaStatDo) {

        if (adxIdeaStatDo != null) {

            boolean isConfidence = false;

            // 判断是否置信
            if (DataUtil.isLarger(adxIdeaStatDo.getBid(), 50L) &&
                    DataUtil.isLarger(adxIdeaStatDo.getBidSuc(), 20L) &&
                    DataUtil.isLarger(adxIdeaStatDo.getExp(), 10L) &&
                    DataUtil.isLarger(adxIdeaStatDo.getClick(), 2L) &&
                    DataUtil.isLarger(adxIdeaStatDo.getAdxConsume(), 50 * ADX_MULTIPLIER) &&
                    DataUtil.isLarger(adxIdeaStatDo.getAdvertConsume(), 50L)) {
                isConfidence = true;
            }


            if (isConfidence) {
                // roi
                Double roi = MathUtil.division(adxIdeaStatDo.getAdvertConsume(), adxIdeaStatDo.getAdxConsume() / ADX_MULTIPLIER, 3);

                // ctr
                Double ctr = MathUtil.division(adxIdeaStatDo.getClick(), adxIdeaStatDo.getExp(), 4);


                // 千次竞价收益
                Double profit = adxIdeaStatDo.getAdvertConsume() / 1.15 - adxIdeaStatDo.getAdxConsume() / ADX_MULTIPLIER;
                Double pEcpm = MathUtil.division(profit * 1000 * roiPenaltyFactor(roi), adxIdeaStatDo.getBid(), 4);

//                System.out.println("idea="+adxIdeaStatDo.getIdeaId()+",profit="+profit+",roi="+roi+",roiPenaltyFactor(roi)="+roiPenaltyFactor(roi)+",bid="+ adxIdeaStatDo.getBid()+",pecpm="+pEcpm);
                adxIdeaStatDo.setRoi(roi);
                adxIdeaStatDo.setCtr(ctr);
                adxIdeaStatDo.setProfitEcpm(pEcpm);

            }

            adxIdeaStatDo.setConfidence(isConfidence);


        }
    }


    /**
     * 惩罚因子，当ROI过大时生效
     * 分段函数
     * 0~1.5   1
     * 1.5~2   0.9
     * 2~      0.8
     *
     * @param roi
     * @return
     */
    public static double roiPenaltyFactor(Double roi) {
        double ret = 1;

        if (roi != null) {

            if (roi >= 1.5 && roi < 2.0) {
                ret = (1.5 + (roi - 1.5) * 0.9) / roi;
            }
            if (roi > 2.0) {
                ret = (1.5 + (2.0 - 1.5) * 0.9 + (roi - 2.0) * 0.8) / roi;
            }
        }

        return ret;
    }


    /**
     * @param rpm
     * @param bestRpm
     * @return
     */
    public static Double getRpmWeight(Double rpm, Double bestRpm) {
        Double ret = null;

        if (rpm != null && bestRpm != null) {

            Double ratio = DataUtil.division(rpm, bestRpm, 3);
            if (bestRpm >= 0) {
                ret = MathBase.getConfidenceWeight(Math.min(ratio, 1.0), pCpmBucket, pCpmWeight);
            } else {
                ret = MathBase.getConfidenceWeight(Math.max(ratio, 1.0), pCpmBucket2, pCpmWeight2);
            }
        }

        return ret;
    }

    public static void updateHistoryRpm(AdxIdeaParamsDo adxIdeaParamsDo, AdxIdeaStatDo adxMinIdeaStatDo) {
        if (AssertUtil.isAllNotEmpty(adxIdeaParamsDo, adxMinIdeaStatDo)) {

            Double decayFactor = getDecayFactor(adxMinIdeaStatDo.getAdvertConsume(), 500);

            Long bidPv = merge(adxIdeaParamsDo.getBidPv(), adxMinIdeaStatDo.getBid(), decayFactor);
            Long adxConsume = merge(adxIdeaParamsDo.getAdxConsume(), adxMinIdeaStatDo.getAdxConsume(), decayFactor);
            Long adConsume = merge(adxIdeaParamsDo.getAdvertConsume(), adxMinIdeaStatDo.getAdvertConsume(), decayFactor);

            Double rpm = null;

            // 判断是否置信
            if (bidPv > 200 && adxConsume > 200 * ADX_MULTIPLIER && adConsume > 200) {

                rpm = DataUtil.formatDouble((adConsume / 1.15 - adxConsume / ADX_MULTIPLIER)/bidPv, 5);

            }


            adxIdeaParamsDo.setBidPv(bidPv);
            adxIdeaParamsDo.setAdxConsume(adxConsume);
            adxIdeaParamsDo.setAdvertConsume(adConsume);
            adxIdeaParamsDo.setRpm(rpm);
        }

    }


    public static Long merge(Long v1, Long v2, Double decayFactor) {

        if (v1 == null) {
            v1 = 0L;
        }
        if (v2 == null) {
            v2 = 0L;
        }

        return DataUtil.double2Long(decayFactor * v1 + v2);

    }

    public static Double getDecayFactor(Long value, long baseValue) {
        Double ret = 1.0;
        if (value != null) {
            ret = 1.0 - 0.3 * Math.min(value / baseValue, 1.0);
        }
        return ret;
    }

}
