package cn.com.duiba.nezha.alg.alg.adx;

import cn.com.duiba.nezha.alg.alg.enums.AdxIndex;
import cn.com.duiba.nezha.alg.alg.enums.AdxLevel;
import cn.com.duiba.nezha.alg.alg.enums.AdxStrategy;
import cn.com.duiba.nezha.alg.alg.vo.AdxLevelDo;
import cn.com.duiba.nezha.alg.alg.vo.AdxStrategyDo;
import cn.com.duiba.nezha.alg.alg.vo.adx.directly.AdxDirectlyFactorDo;
import cn.com.duiba.nezha.alg.alg.vo.adx.directly.AdxIndexStatsDo;
import cn.com.duiba.nezha.alg.alg.vo.adx.flowfilter.AdxIndexStatDo;
import cn.com.duiba.nezha.alg.common.util.AssertUtil;
import cn.com.duiba.nezha.alg.common.util.DataUtil;
import cn.com.duiba.nezha.alg.common.util.MathUtil;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class AdxStatData {


    private static long ADX_MULTIPLIER = 10000000;


    /**
     * 计算策略各指标数据
     *
     * @param strategy
     * @param strategyDoList
     * @return 策略各指标数据（key：指标）
     */
    public static Map<String, Long> getStrategyStatData(String strategy,
                                                        ArrayList<AdxStrategyDo> strategyDoList) {


        Map<String, Long> retMap = new HashMap<>();

        Long defaultCnt = 0L;
        Long defaultConsume = 0L;
        Long bidCnt = defaultCnt, sucCnt = defaultCnt, expCnt = defaultCnt, clickCnt = defaultCnt;
        Long adxConsume = defaultConsume, advertConsume = defaultConsume;

        if (AssertUtil.isAllNotEmpty(strategy, strategyDoList)) {

            List<AdxLevelDo> strategyDo = StrategyBid.getStrategyInfo(strategyDoList, strategy);
            bidCnt = StrategyBid.getSumLevelIndex(strategyDo, AdxIndex.BID.getCode(), defaultCnt);
            sucCnt = StrategyBid.getSumLevelIndex(strategyDo, AdxIndex.BID_SUC.getCode(), defaultCnt);
            expCnt = StrategyBid.getSumLevelIndex(strategyDo, AdxIndex.EXP.getCode(), defaultCnt);
            clickCnt = StrategyBid.getSumLevelIndex(strategyDo, AdxIndex.CLICK.getCode(), defaultCnt);
            adxConsume = DataUtil.double2Long((StrategyBid.getSumLevelIndex(strategyDo, AdxIndex.ADX_CONSUME.getCode(), defaultConsume) / ADX_MULTIPLIER), 1L);
            advertConsume = StrategyBid.getSumLevelIndex(strategyDo, AdxIndex.ADVERT_CONSUME.getCode(), defaultConsume);

        }

        retMap.put(AdxIndex.BID.getCode(), bidCnt);                   //竞价量
        retMap.put(AdxIndex.BID_SUC.getCode(), sucCnt);               //竞价成功量
        retMap.put(AdxIndex.EXP.getCode(), expCnt);                   //曝光量
        retMap.put(AdxIndex.CLICK.getCode(), clickCnt);               //点击量
        retMap.put(AdxIndex.ADX_CONSUME.getCode(), adxConsume);       //adx消耗(分)
        retMap.put(AdxIndex.ADVERT_CONSUME.getCode(), advertConsume); //广告消耗(分)

        return retMap;
    }


    /**
     * 计算策略-level指定指标数据
     *
     * @param strategy
     * @param strategyDoList
     * @return 策略-level指定指标数据（key：level）
     */
    public static Map<String, Long> getlevelIndexStat(String strategy, String indexLabel,
                                                      ArrayList<AdxStrategyDo> strategyDoList) {

        Map<String, Long> retMap = new HashMap<>();

        Long defaultValue = 0L;

        List<AdxLevelDo> strategyDo = StrategyBid.getStrategyInfo(strategyDoList, strategy);
        retMap = StrategyBid.getLevelIndex(strategyDo, indexLabel, defaultValue);

        if (indexLabel.equals(AdxIndex.ADX_CONSUME.getCode())) {
            retMap.replaceAll((k, v) -> DataUtil.double2Long(DataUtil.division(v, ADX_MULTIPLIER)));
        }

        return retMap;
    }


    /**
     * 计算策略/创意/资源位整体的roi
     *
     * @param mapInfo
     * @return roi
     */
    public static Double getRoi(Map<String, Long> mapInfo, Double targetRoi) {

        Double ret = 0.0;
        Long defaultValue = 0L;
        Long defaultAdxConsume = 50L;

        if (AssertUtil.isNotEmpty(mapInfo)) {
            //adx消耗(分)
            Long adxConsume = nullToMinDefault(mapInfo.get(AdxIndex.ADX_CONSUME.getCode()), defaultValue);
            //广告消耗(分)
            Long advertConsume = nullToMinDefault(mapInfo.get(AdxIndex.ADVERT_CONSUME.getCode()), defaultValue);

            if (adxConsume >= defaultAdxConsume) {
                ret = DataUtil.division(advertConsume, adxConsume, 6);

            } else {
                ret = DataUtil.division((advertConsume + defaultAdxConsume * targetRoi), (adxConsume + defaultAdxConsume), 6);
            }
        }

        return ret;
    }


    /**
     * 计算不同level的roi
     *
     * @param adxMap
     * @param advertMap
     * @param targetRoi
     * @return 默认值
     */
    public static Map<String, Double> getLevelRoi(Map<String, Long> adxMap,
                                                  Map<String, Long> advertMap,
                                                  Double targetRoi) {

        Map<String, Double> retMap = new HashMap<>();

        Double ret = 0.0;
        Long defaultAdxConsume = 25L;

        for (AdxLevel adxLevel : AdxLevel.values()) {
            String key = adxLevel.getCode();
            if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())) {

                if (AssertUtil.isAllNotEmpty(adxMap, advertMap)) {

                    if (adxMap.get(key) >= defaultAdxConsume) {
                        ret = DataUtil.division(advertMap.get(key), adxMap.get(key), 6);
                        retMap.put(key, ret);

                    } else {
                        ret = DataUtil.division((advertMap.get(key) + defaultAdxConsume * targetRoi), (adxMap.get(key) + defaultAdxConsume), 6);
                        retMap.put(key, ret);
                    }

                } else {
                    retMap.put(key, 0.0);
                }
            }
        }
        return retMap;
    }


    /**
     * ROI置信处理（根据分level调节因子）
     *
     * @param roiMap
     * @param adxConsumeLevelMs
     * @param directlyFactorDoInfo
     * @return 分level-ROI
     */
    public static Map<String, Double> getRoiFactorRectify(Map<String, Double> roiMap,
                                                          Map<String, Long> adxConsumeLevelMs,
                                                          AdxDirectlyFactorDo directlyFactorDoInfo) {

        Map<String, Double> retMap = new HashMap<>();

        Long adxConsumeLimit = 50L;
        Long adxConsumeBaseLimit = 25L;
        Double baseRoi = nullToMinDefault(roiMap.get(AdxLevel.ADX_LEVEL_TWO.getCode()), 0.0);

        for (AdxLevel adxLevel : AdxLevel.values()) {
            String key = adxLevel.getCode();
            if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())) {
                retMap.put(key, roiMap.get(key));
            }
        }

        if (AssertUtil.isNotEmpty(directlyFactorDoInfo)) {
            Map<String, Double> factorMap = directlyFactorDoInfo.getFactorExploreMap();
            Double baseFactor = 1.0;
            if (AssertUtil.isNotEmpty(factorMap)) {
                baseFactor = nullToMinDefault(factorMap.get(AdxLevel.ADX_LEVEL_TWO.getCode()), 1.0);
            }

            for (AdxLevel adxLevel : AdxLevel.values()) {
                String key = adxLevel.getCode();
                if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())) {

                    if (AssertUtil.isNotEmpty(adxConsumeLevelMs)) {

                        if (adxConsumeLevelMs.get(key) < adxConsumeLimit
                                && adxConsumeLevelMs.get(AdxLevel.ADX_LEVEL_TWO.getCode()) >= adxConsumeBaseLimit) {

                            Double diff = DataUtil.division(factorMap.get(key), baseFactor);
                            Double roi = baseRoi * diff;

                            retMap.put(key, DataUtil.formatDouble(roi,6));
                        }
                    }
                }
            }
        }

        return retMap;
    }


    /**
     * 计算竞价成功率
     *
     * @param mapInfo
     * @return 默认值
     */
    public static Double getSucRate(Map<String, Long> mapInfo, Long defaultCnt) {

        Double ret = 0.0;
        Long defaultValue = 0L;

        if (AssertUtil.isNotEmpty(mapInfo)) {

            //竞价量
            Long bidCnt = nullToMinDefault(mapInfo.get(AdxIndex.BID.getCode()), defaultValue);
            //竞价成功量
            Long sucCnt = nullToMinDefault(mapInfo.get(AdxIndex.BID_SUC.getCode()), defaultValue);

            if (bidCnt >= defaultCnt) {
                ret = getNormalValue(DataUtil.division(sucCnt, bidCnt, 6), 0.0, 0.0, 1.0);
            }
        }
        return ret;
    }


    /**
     * 计算不同level的竞价成功率
     *
     * @param bidCntMap
     * @param sucCntMap
     * @return 默认值
     */
    public static Map<String, Double> getLevelSucRate(Map<String, Long> bidCntMap,
                                                      Map<String, Long> sucCntMap) {

        Map<String, Double> retMap = new HashMap<>();
        Double ret = 0.0;
        Long defaultCnt = 10L;

        for (AdxLevel adxLevel : AdxLevel.values()) {
            String key = adxLevel.getCode();
            if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())) {

                if (AssertUtil.isAllNotEmpty(bidCntMap, sucCntMap)) {

                    if (bidCntMap.get(key) >= defaultCnt) {
                        ret = getNormalValue(DataUtil.division(sucCntMap.get(key), bidCntMap.get(key), 6), 0.0, 0.0, 1.0);
                        retMap.put(key, ret);

                    } else {
                        retMap.put(key, 0.0);
                    }

                } else {
                    retMap.put(key, 0.0);
                }
            }
        }
        return retMap;
    }


    /**
     * 计算策略流量配比
     *
     * @param flowRateMap
     * @param ind
     * @param step
     * @return
     */
    public static Map<String, Double> getFlowRateAdjust(Map<String, Double> flowRateMap,
                                                        Integer ind, Double step) {

        Map<String, Double> retMap = new HashMap<>();

        Double defaultFlowRate = 0.33;  // 默认流量配比
        Double firFlowRate = defaultFlowRate, secFlowRate = defaultFlowRate, thiFlowRate = defaultFlowRate;

        if (AssertUtil.isNotEmpty(flowRateMap)) {
            firFlowRate = getNormalValue(flowRateMap.get(AdxStrategy.ADX_STRATEGY_FIR.getCode()), defaultFlowRate, 0.1, 0.8);
            secFlowRate = getNormalValue(flowRateMap.get(AdxStrategy.ADX_STRATEGY_SEC.getCode()), defaultFlowRate, 0.1, 0.8);
            thiFlowRate = getNormalValue(flowRateMap.get(AdxStrategy.ADX_STRATEGY_THI.getCode()), defaultFlowRate, 0.1, 0.8);
        }

        if (AssertUtil.isAllNotEmpty(ind, step)) {

            if (ind == 1) {
                firFlowRate += step;
            } else if (ind == 2) {
                secFlowRate += step;
            } else if (ind == 3) {
                thiFlowRate += step;
            }

        }

        retMap.put(AdxStrategy.ADX_STRATEGY_FIR.getCode(), firFlowRate);
        retMap.put(AdxStrategy.ADX_STRATEGY_SEC.getCode(), secFlowRate);
        retMap.put(AdxStrategy.ADX_STRATEGY_THI.getCode(), thiFlowRate);

        return retMap;
    }


    /**
     * 缺失情况，重置为defaultValue
     *
     * @param value
     * @param defaultValue
     * @return 默认值
     */
    public static Double nullToDefault(Double value, Double defaultValue) {

        Double ret = defaultValue;
        if (value != null) {
            ret = value;
        }
        return ret;
    }


    /**
     * 缺失情况，重置为defaultValue
     *
     * @param value
     * @param defaultValue
     * @return 默认值
     */
    public static Long nullToDefault(Long value, Long defaultValue) {

        Long ret = defaultValue;
        if (value != null) {
            ret = value;
        }
        return ret;
    }


    /**
     * 缺失情况，重置为defaultValue
     *
     * @param value
     * @param defaultValue
     * @return 默认值
     */
    public static Double nullToMinDefault(Double value, Double defaultValue) {

        Double ret = defaultValue;
        if (value != null && value > defaultValue) {
            ret = value;
        }
        return ret;
    }


    /**
     * 缺失情况，重置为defaultValue
     *
     * @param value
     * @param defaultValue
     * @return 默认值
     */
    public static Long nullToMinDefault(Long value, Long defaultValue) {

        Long ret = defaultValue;
        if (value != null && value > defaultValue) {
            ret = value;
        }
        return ret;
    }


    /**
     * 获取正常范围内的值
     *
     * @param value
     * @param defaultValue
     * @return 默认值
     */
    public static Double getNormalValue(Double value, Double defaultValue, Double lower, Double upper) {

        Double ret = defaultValue;
        if (value != null) {
            ret = value < lower ? lower : (value > upper ? upper : value);
        }
        return ret;
    }


    /**
     * 获取正常范围内的值
     *
     * @param value
     * @param defaultValue
     * @return 默认值
     */
    public static Long getNormalValue(Long value, Long defaultValue, Long lower, Long upper) {

        Long ret = defaultValue;
        if (value != null) {
            ret = value < lower ? lower : (value > upper ? upper : value);
        }
        return ret;
    }


    /**
     * 判断是否比阈值大
     *
     * @param value1
     * @param value2
     * @return
     */
    public static boolean isLarger(Long value1, Long value2) {

        boolean ret = false;

        if (value1 != null && value2 != null && value1 > value2) {
            ret = true;
        }

        return ret;
    }


    /**
     * 判断是否比阈值大
     *
     * @param value1
     * @param value2
     * @return
     */
    public static boolean isLarger(Double value1, Double value2) {

        boolean ret = false;

        if (value1 != null && value2 != null && value1 > value2) {
            ret = true;
        }

        return ret;
    }


    /**
     * 判断两者是否相等
     *
     * @param value1
     * @param value2
     * @return
     */
    public static boolean isEqual(Long value1, Long value2) {

        boolean ret = false;

        if (value1 != null && value2 != null && value1.equals(value2)) {
            ret = true;
        }

        return ret;
    }


    /**
     * 判断两者是否相等
     *
     * @param value1
     * @param value2
     * @return
     */
    public static boolean isEqual(Double value1, Double value2) {

        boolean ret = false;

        if (value1 != null && value2 != null && value1.equals(value2)) {
            ret = true;
        }

        return ret;
    }


    /**
     * 计算流量配比得分
     *
     * @param mapInfo
     * @return roi
     */
    public static Double getFlowRateScore(Map<String, Long> mapInfo) {

        Double ret = 0.0;
        Long defaultValue = 0L;
        Long bidCntLimit = 20L;

        if (AssertUtil.isNotEmpty(mapInfo)) {

            //竞价量
            Long bidCnt = nullToMinDefault(mapInfo.get(AdxIndex.BID.getCode()), defaultValue);
            //adx消耗(分)
            Long adxConsume = nullToMinDefault(mapInfo.get(AdxIndex.ADX_CONSUME.getCode()), defaultValue);
            //广告消耗(分)
            Long advertConsume = nullToMinDefault(mapInfo.get(AdxIndex.ADVERT_CONSUME.getCode()), defaultValue);

            if (bidCnt >= bidCntLimit) {
                ret = DataUtil.division((advertConsume / 1.15 - adxConsume), bidCnt) * 1000;
            }
        }

        return DataUtil.formatDouble(ret,3);
    }


    /**
     *冷启动高价组-因子试探流量占比
     *
     * @param roiMsDiff
     * @param adxConsumeTryMs(分/cpm)
     * @param directlyFactorDoInfo
     * @return
     */
    public static Double getTryFlowRate(Double roiMsDiff, Long adxConsumeTryMs,
                                        AdxDirectlyFactorDo directlyFactorDoInfo) {

        Double ret = 0.20;
        Double lastRate = 0.20;

        if (AssertUtil.isNotEmpty(directlyFactorDoInfo)) {
            Map<String, Double> flowRateMap = directlyFactorDoInfo.getFactorFlowRateMap();
            if (AssertUtil.isNotEmpty(flowRateMap)) {
                lastRate = getNormalValue(flowRateMap.get(AdxLevel.ADX_LEVEL_ONE.getCode()), lastRate, 0.02, 1.0);
            }
        }

        if (AssertUtil.isAllNotEmpty(roiMsDiff, adxConsumeTryMs)) {

            Long adxConsumeMs = nullToDefault(adxConsumeTryMs, 0L);
            Double baseRate = getNormalValue((DataUtil.division(100.0, adxConsumeMs) * lastRate), lastRate, 0.02,0.8);

            if (roiMsDiff > 0.98) {
                ret = baseRate + Math.min(DataUtil.division(roiMsDiff, 0.98) - 1.0, 1.0) * (1.0 - baseRate);

            } else {

                if (adxConsumeMs < 100) {
                    ret = baseRate;

                } else if (adxConsumeMs > 1000) {
                    ret = baseRate * 1.5;

                } else {
                    ret = baseRate * 1.2;

                }

            }

        }

        return getNormalValue(ret,0.20, 0.02, 1.0);
    }


    /**
     * 消耗置信保障
     *
     * @param roi
     * @return 消耗置信保障(分)
     */
    public static Double getConfidentAdxConsume(Double roi) {

        Double ret = 100.0;

        if (AssertUtil.isNotEmpty(roi)){

            ret = 100.0 + DataUtil.division(roi-1, 1.15-1) * 100.0 ;
        }

        return getNormalValue(ret,100.0,100.0,200.0);

    }


    /**
     * 冷启动-试探涨价幅度
     *
     * @param adxConsume(分/cpm)
     * @param roi
     * @param limit
     * @return
     */
    public static Double getTryUppStepCon(Long adxConsume, Double roi, Double limit) {

        Double ret = 0.0;

        if (AssertUtil.isAllNotEmpty(adxConsume, roi, limit)) {

            Double confidentConsume = getConfidentAdxConsume(roi);

            if (adxConsume <= confidentConsume) {
                ret = (1.0 - DataUtil.division(adxConsume, confidentConsume)) * limit;

            } else {
                ret = Math.max(1.0 - DataUtil.division(adxConsume, confidentConsume), -1) * limit;

            }

        }

        return DataUtil.formatDouble(getNormalValue(ret,0.0, -limit, limit),3);

    }



    /**
     * 选取最优level（左闭右开区间）
     *
     * @param bidMap
     * @param adxMap
     * @param advertMap
     * @return level
     */
    public static String selectBestLevel(Map<String, Long> bidMap,
                                         Map<String, Long> adxMap,
                                         Map<String, Long> advertMap) {

        String ret = AdxLevel.ADX_LEVEL_TWO.getCode();

        if (AssertUtil.isAllNotEmpty(bidMap, adxMap, advertMap)) {
            Double tmpRpm = -1000000.0;

            for (AdxLevel adxLevel : AdxLevel.values()) {
                String key = adxLevel.getCode();

                if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())) {
                    Long bid = nullToMinDefault(bidMap.get(adxLevel),0L);
                    Long adx =  nullToMinDefault(adxMap.get(adxLevel),0L);
                    Long advert =  nullToMinDefault(advertMap.get(adxLevel),0L);

                    if (isLarger(bid, 10L) && isLarger(adx, 25L) && isLarger(advert, 25L)) {

                        Double profit = advert / 1.15 - adx;
                        Double rpm = MathUtil.division((profit * 1000L), bid,6);

                        if (rpm > tmpRpm) {
                            ret = key;
                            tmpRpm = rpm;
                        }
                    }

                }
            }
        }
        return ret;
    }


    /**
     * 指标计算(创意维度)
     *
     * @param indexStatInfo
     */
    public static AdxIndexStatsDo indexStatCompute(Map<String, Long> indexStatInfo) {

        AdxIndexStatsDo ret = new AdxIndexStatsDo();

        boolean isConfident = false;
        Long bidCnt = 0L, sucCnt = 0L, expCnt = 0L, clickCnt = 0L , adxConsume = 0L, advertConsume = 0L;

        if (AssertUtil.isNotEmpty(indexStatInfo)) {

            //指标:竞价量级,竞价成功量级,曝光量级,点击量级,adx消耗(分),广告消耗(分)
            bidCnt = nullToMinDefault(indexStatInfo.get(AdxIndex.BID.getCode()), 0L);
            sucCnt = nullToMinDefault(indexStatInfo.get(AdxIndex.BID_SUC.getCode()), 0L);
            expCnt = nullToMinDefault(indexStatInfo.get(AdxIndex.EXP.getCode()), 0L);
            clickCnt = nullToMinDefault(indexStatInfo.get(AdxIndex.CLICK.getCode()), 0L);
            adxConsume = DataUtil.double2Long((nullToMinDefault(indexStatInfo.get(AdxIndex.ADX_CONSUME.getCode()), 0L) / ADX_MULTIPLIER), 1L);
            advertConsume = nullToMinDefault(indexStatInfo.get(AdxIndex.ADVERT_CONSUME.getCode()), 0L);

            // 判断是否置信
            if (isLarger(bidCnt, 50L) && isLarger(sucCnt, 20L)
                    && isLarger(expCnt, 10L) && isLarger(clickCnt, 2L)
                    && isLarger(adxConsume, 50L) && isLarger(advertConsume, 50L)) {
                isConfident = true;
            }


            if (isConfident) {

                // roi：广告消耗/adx消耗
                Double roi = MathUtil.division(advertConsume, adxConsume, 6);
                // sucRate：竞价成功率
                Double sucRate = MathUtil.division(sucCnt, bidCnt, 6);
                // ctr：adx点击率
                Double ctr = MathUtil.division(clickCnt, expCnt, 6);
                // rpm: 千次竞价收益
                Double profit = advertConsume / 1.15 - adxConsume;
                Double rpm = MathUtil.division(profit * 1000L, bidCnt, 6);

                ret.setRoi(roi);
                ret.setSucRate(sucRate);
                ret.setCtr(ctr);
                ret.setRpm(rpm);

            }
        }

        ret.setConfident(isConfident);
        ret.setBidCnt(bidCnt);
        ret.setSucCnt(sucCnt);
        ret.setExpCnt(expCnt);
        ret.setClickCnt(clickCnt);
        ret.setAdxConsume(adxConsume);
        ret.setAdvertConsume(advertConsume);

        return ret;
    }



    /**
     * 获取ctr预估区间preInterval
     *
     * @param preValue  ctr预估值
     */
    public static Integer getPreInterval(Double preValue) {

        Integer ret = null;
        Double[] preList = {0.002, 0.004, 0.007, 0.01};  // 预估ctr区间划分

        if (AssertUtil.isNotEmpty(preValue) && preValue >= 0) {

            ret = preValue < preList[0] ? 1
                    : (preValue < preList[1] ? 2
                    : (preValue < preList[2] ? 3
                    : (preValue < preList[3] ? 4
                    : 5)));
        }

        return ret;
    }



    /**
     * 指标计算(创意维度)
     *
     * @param adxIndexStats
     */
    public static AdxIndexStatsDo adxIndexCompute(AdxIndexStatDo adxIndexStats) {

        AdxIndexStatsDo ret = new AdxIndexStatsDo();

        boolean isConfident = false;
        Long bidCnt = 0L, sucCnt = 0L, expCnt = 0L, clickCnt = 0L , adxConsume = 0L, advertConsume = 0L;

        if (AssertUtil.isNotEmpty(adxIndexStats)) {

            //指标:竞价量级,竞价成功量级,曝光量级,点击量级,adx消耗(分),广告消耗(分)
            bidCnt = nullToMinDefault(adxIndexStats.getBid(), 0L);
            sucCnt = nullToMinDefault(adxIndexStats.getSuc(), 0L);
            expCnt = nullToMinDefault(adxIndexStats.getExp(), 0L);
            clickCnt = nullToMinDefault(adxIndexStats.getClick(), 0L);
            adxConsume = DataUtil.double2Long((nullToMinDefault(adxIndexStats.getAdxConsume(), 0L) / ADX_MULTIPLIER), 1L);
            advertConsume = nullToMinDefault(adxIndexStats.getAdvertConsume(), 0L);

            // 判断是否置信
            if (isLarger(bidCnt, 50L) && isLarger(sucCnt, 20L)
                    && isLarger(expCnt, 10L) && isLarger(clickCnt, 2L)
                    && isLarger(adxConsume, 50L) && isLarger(advertConsume, 50L)) {
                isConfident = true;
            }

            if (isConfident) {

                // roi：广告消耗/adx消耗
                Double roi = MathUtil.division(advertConsume, adxConsume, 6);
                // sucRate：竞价成功率
                Double sucRate = MathUtil.division(sucCnt, bidCnt, 6);
                // ctr：adx点击率
                Double ctr = MathUtil.division(clickCnt, expCnt, 6);
                // rpm: 千次竞价收益
                Double profit = advertConsume / 1.15 - adxConsume;
                Double rpm = MathUtil.division(profit * 1000L, bidCnt, 6);

                ret.setRoi(roi);
                ret.setSucRate(sucRate);
                ret.setCtr(ctr);
                ret.setRpm(rpm);

            }
        }

        ret.setConfident(isConfident);
        ret.setBidCnt(bidCnt);
        ret.setSucCnt(sucCnt);
        ret.setExpCnt(expCnt);
        ret.setClickCnt(clickCnt);
        ret.setAdxConsume(adxConsume);
        ret.setAdvertConsume(advertConsume);

        return ret;
    }



    /**
     * 指标计算(创意维度)
     *
     * @param intervalStats
     */
    public static Map<Integer, AdxIndexStatsDo> intervalIndexCompute(Map<Integer, AdxIndexStatDo> intervalStats) {

        Integer preIntervalSize = 5;// ctr预估区间个数
        Map<Integer, AdxIndexStatsDo> retMap = new HashMap<>(preIntervalSize);

        boolean isConfident = false;
        Long bidCnt = 0L, sucCnt = 0L, expCnt = 0L, clickCnt = 0L , adxConsume = 0L, advertConsume = 0L;

        for (int key = 1; key <= preIntervalSize; ++key) {

            AdxIndexStatsDo ret = new AdxIndexStatsDo();

            if (AssertUtil.isNotEmpty(intervalStats)) {
                AdxIndexStatDo intervalInfo = intervalStats.get(key);

                if (AssertUtil.isNotEmpty(intervalInfo)) {
                    ret = adxIndexCompute(intervalInfo);
                }
            }

            ret.setConfident(isConfident);
            ret.setBidCnt(bidCnt);
            ret.setSucCnt(sucCnt);
            ret.setExpCnt(expCnt);
            ret.setClickCnt(clickCnt);
            ret.setAdxConsume(adxConsume);
            ret.setAdvertConsume(advertConsume);

            retMap.put(key, ret);

        }

        return retMap;
    }


}
