package cn.com.duiba.nezha.alg.alg.adx.flowfilter;

import cn.com.duiba.nezha.alg.alg.adx.AdxStatData;
import cn.com.duiba.nezha.alg.alg.vo.adx.directly.AdxIndexStatsDo;
import cn.com.duiba.nezha.alg.alg.vo.adx.flowfilter.*;
import cn.com.duiba.nezha.alg.common.util.AssertUtil;
import cn.com.duiba.nezha.alg.common.util.DataUtil;
import cn.com.duiba.nezha.alg.feature.parse.AdxFeatureParse;
import cn.com.duiba.nezha.alg.feature.vo.AdxFeatureDo;
import cn.com.duiba.nezha.alg.feature.vo.FeatureMapDo;
import cn.com.duiba.nezha.alg.model.CODER;
import cn.com.duiba.nezha.alg.model.tf.LocalTFModel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

public class LaunchPvFiltering {

    /**
     * 每pv发券过滤
     *
     * @param flowFilterRequestDo
     * @return
     */

    private static final Logger logger = LoggerFactory.getLogger(LaunchPvFiltering.class);

    public static FlowFilterResultDo getLaunchPvFiltering(FlowFilterRequestDo flowFilterRequestDo) {

        FlowFilterResultDo ret = new FlowFilterResultDo();

        /**
         *
         * 步骤：
         * 步骤0、预估每PV发券
         * 步骤1、获取统计数据
         * 步骤2、获取历史过滤比例
         * 步骤3、判断流量过滤
         * 步骤4、控制整体放弃比例
         *
         */

        try {

            Integer flowFilterLabel = 0;        //流量过滤标记(0：不过滤；1：过滤)
            Integer filterType = 0;             //算法过滤原因(0-不过滤，1-流量预过滤，2-每pv发券过滤)
            boolean isConfident = false;        //统计指标是否置信

            Double preWeight = 0.8;             //每PV发券模型预估权重
            Double flowerLimit = 0.0;           //每PV发券统计值*比例下限
            Double fUpperLimit = 4.0;           //每PV发券统计值*比例上限
            Double defaultValue = 0.01;         //每PV发券默认值


            //0.预估每PV发券
            Double preLaunchPv = null;
            if(valid(flowFilterRequestDo)) {

                AdxFeatureDo adxFeatureDo = flowFilterRequestDo.getAdxFeatureDo();
                Map<String, String> staticFeatureMap = AdxFeatureParse.generateFeatureMapStatic(adxFeatureDo);
                Map<String, String> dynamicFeatureMap = AdxFeatureParse.generateFeatureMapDynamic(adxFeatureDo, adxFeatureDo);
                FeatureMapDo featureMapDo = new FeatureMapDo();
                featureMapDo.setDynamicFeatureMap(dynamicFeatureMap);
                featureMapDo.setStaticFeatureMap(staticFeatureMap);

                Map<Long, FeatureMapDo> featureMap = new HashMap<>();
                Long ideaId = flowFilterRequestDo.getFlowInfo().getIdeaId(); //默认1个创意
                featureMap.put(ideaId, featureMapDo);

                CODER launchPvCoderModel = flowFilterRequestDo.getLaunchPvCoderModel();
                LocalTFModel launchPvLtfModel = flowFilterRequestDo.getLaunchPvLtfModel();
                Map<Long, Double> preValueMap = launchPvCoderModel.predictWithLocalTFNew(featureMap, launchPvLtfModel);
                preLaunchPv = preValueMap.get(ideaId);
            }


            AdxFlowInfoDo flowInfo = flowFilterRequestDo.getFlowInfo();
            if (AssertUtil.isNotEmpty(flowInfo)) {

                //1.获取统计数据
                AdxIndexStatDo ideaMinStats = flowInfo.getIdeaMinStats();
                AdxIndexStatsDo ideaMinComputes = AdxStatData.adxIndexCompute(ideaMinStats);
                AdxIndexStatDo ideaDayStats = flowInfo.getIdeaDayStats();
                AdxIndexStatsDo ideaDayComputes = AdxStatData.adxIndexCompute(ideaDayStats);
                isConfident = ideaMinComputes.getConfident() && ideaDayComputes.getConfident();


                //2.获取历史过滤比例
                Map<Integer, AdxFilterCntDo> ideafilterDo = flowFilterRequestDo.getIdeafilterCntDo();
                Double minfilterRate = getFilterRate(ideafilterDo, "20min");
                Double dayfilterRate = getFilterRate(ideafilterDo, "1day");
                Double filterRate = Math.max(minfilterRate, dayfilterRate);


                //3.判断流量过滤（前提：数据置信）
                Double statLaunchPv = AdxStatData.nullToDefault(ideaMinComputes.getLaunchPv(), ideaDayComputes.getLaunchPv());
                Double conLaunchPv = AdxStatData.getConValue(preLaunchPv, statLaunchPv, preWeight, flowerLimit, fUpperLimit, defaultValue);
                Double launchPvLimit = AdxStatData.nullToMinDefault(flowFilterRequestDo.getLaunchPvLimit(), 0.0);

                double random = Math.random();
                if (isConfident) {

                    if (conLaunchPv < launchPvLimit * 0.5) {
                        flowFilterLabel = random < 0.99 ? 1 : 0;
                    } else if (conLaunchPv < launchPvLimit * 0.7) {
                        flowFilterLabel = random < 0.90 ? 1 : 0;
                    } else if (conLaunchPv < launchPvLimit * 0.9) {
                        flowFilterLabel = random < 0.80 ? 1 : 0;
                    }
                }

                //4.控制整体放弃比例
                double random2 = Math.random();
                if (flowFilterLabel.equals(1)){

                    if (filterRate > 0.8) {
                        flowFilterLabel = random < 0.2 ? 1 : 0;
                    } else if (filterRate > 0.5) {
                        flowFilterLabel = random2 < 0.5 ? 1 : 0;
                    }
                }

            }

            //算法过滤原因(0-不过滤，1-流量预过滤，2-每pv发券过滤)
            filterType = flowFilterLabel == 1 ? 2 : 0;

            ret.setFlowFilterLabel(flowFilterLabel);
            ret.setFilterType(filterType);
            ret.setPreLaunchPv(preLaunchPv);


        } catch (Exception e) {
            logger.error("LaunchPvFiltering.getLaunchPvFiltering error " + e);
        }

        return ret;
    }


    /**
     * 校验基础信息是否合法
     *
     * @param flowFilterRequestDo
     * @return
     */
    private static Boolean valid(FlowFilterRequestDo flowFilterRequestDo) {

        Boolean ret = true;

        if (flowFilterRequestDo == null) {
            logger.error("LaunchPvFiltering.getLaunchPvFiltering() input params valid , params flowFilterRequestDo is null");
            return false;
        }

        if (flowFilterRequestDo.getLaunchPvLtfModel() == null) {
            logger.error("LaunchPvFiltering.getLaunchPvFiltering() input params valid , params launchPvLtfModel is null");
            return false;
        }

        if (flowFilterRequestDo.getLaunchPvCoderModel() == null) {
            logger.error("LaunchPvFiltering.getLaunchPvFiltering() input params valid , params launchPvCoderModel is null");
            return false;
        }

        if (flowFilterRequestDo.getAdxFeatureDo() == null) {
            logger.error("LaunchPvFiltering.getLaunchPvFiltering() input params valid ,params adxFeatureDo is null");
            return false;
        }

        if (flowFilterRequestDo.getFlowInfo() == null) {
            logger.error("LaunchPvFiltering.getLaunchPvFiltering() input params valid ,params flowInfo is null");
            return false;
        }

        return ret;
    }


    /**
     * 计算过滤比例
     *
     * @param ideafilterDo
     * @return
     */
    private static Double getFilterRate(Map<Integer, AdxFilterCntDo> ideafilterDo, String timeIndex) {

        Double ret = 0.0;

        if (AssertUtil.isNotEmpty(ideafilterDo)) {

            AdxFilterCntDo filterCntDo1 = Optional.ofNullable(ideafilterDo.get(0)).orElse(new AdxFilterCntDo());
            AdxFilterCntDo filterCntDo2 = Optional.ofNullable(ideafilterDo.get(1)).orElse(new AdxFilterCntDo());
            AdxFilterCntDo filterCntDo3 = Optional.ofNullable(ideafilterDo.get(2)).orElse(new AdxFilterCntDo());

            Long cnt1 = 0L, cnt2 = 0L, cnt3 = 0L;
            if (timeIndex.equals("20min")) {
                cnt1 = AdxStatData.nullToMinDefault(filterCntDo1.getLast20MinCnt(),0L);
                cnt2 = AdxStatData.nullToMinDefault(filterCntDo2.getLast20MinCnt(),0L);
                cnt3 = AdxStatData.nullToMinDefault(filterCntDo3.getLast20MinCnt(),0L);
            } else if (timeIndex.equals("1day")) {
                cnt1 = AdxStatData.nullToMinDefault(filterCntDo1.getLast1DayCnt(),0L);
                cnt2 = AdxStatData.nullToMinDefault(filterCntDo2.getLast1DayCnt(),0L);
                cnt3 = AdxStatData.nullToMinDefault(filterCntDo3.getLast1DayCnt(),0L);
            }

            // 计算每PV发券过滤比例
            if (AdxStatData.isLarger(cnt3, 10L) && AdxStatData.isLarger((cnt1+cnt2+cnt3), 30L)) {
                ret = DataUtil.division(cnt3, (cnt1+cnt2+cnt3), 5);
            }

        }

        return AdxStatData.getNormalValue(ret, 0.0, 0.0, 1.0);
    }
}
