package cn.com.duiba.nezha.alg.api.model;

import cn.com.duiba.nezha.alg.api.model.util.WuWuPredResult;
import cn.com.duiba.nezha.alg.api.model.util.WuWuTFUtils;
import com.alibaba.fastjson.JSON;
import jdk.nashorn.internal.ir.debug.ObjectSizeCalculator;
import org.slf4j.LoggerFactory;
import org.tensorflow.SavedModelBundle;
import org.tensorflow.Session;
import org.tensorflow.Tensor;
import org.tensorflow.framework.DataType;
import org.tensorflow.framework.MetaGraphDef;
import org.tensorflow.framework.SignatureDef;
import org.tensorflow.framework.TensorInfo;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


/**
 * 每个特征的输入shape均为(None,)
 * 每个输出的shape均为(None,k)，k为任意常数
 * 输入与输出都只支持2个类型，Float与String
 * TODO 由于输入输出的类型和形状均来自于模型文件，导致有许多对于类型和形状的判断，还有Object数组的使用，需要问问专业的java开发看看怎么办比较好
 * TODO keySet的遍历要改成entrySet
 * */
public class WuWuLocalTFModel {

    private static final org.slf4j.Logger logger = LoggerFactory.getLogger(WuWuLocalTFModel.class);
    //默认导出pb时候的签名，基本不变
    private static String SIGNATURE_DEF = "serving_default";
    //java跟tf交互的入口，取输入输出建session全靠它（在load模型的时候获取到的）
    private SavedModelBundle bundle;
    //模型的路径
    private String path;
    //模型创建的时间戳
    private Long version;
    //输入的名字跟对应的TensorInfo，TensorInfo里面存了形状与类型
    private Map<String, TensorInfo> inputSpecMap;
    //输出的名字跟对应的TensorInfo，TensorInfo里面存了形状与类型
    private Map<String, TensorInfo> outputSpecMap;

    //从路径加载最新模型
    public void loadLatestModel(String path) throws Exception {
        Long lastVersion = WuWuTFUtils.getLastVersion(path);
        SavedModelBundle bundle = WuWuTFUtils.loadModel(path, lastVersion + "");
        logger.info("loadModel ,path=" + path + "with version=" + lastVersion);
        setModel(bundle);
        this.path = path;
        this.version = lastVersion;
    }

    //关闭模型的方法，不然会一直占用内存
    public void close() throws Exception {
        if (bundle != null) {
            logger.info("close model  ,path=" + path + "with version=" + version);
            logger.info("bundle.size before close :" + ObjectSizeCalculator.getObjectSize(bundle));
            bundle.close();
            bundle = null;
        }
    }

    //把仨核心变量配置上
    private void setModel(SavedModelBundle bundle) throws Exception {
        this.bundle = bundle;
        this.inputSpecMap = getInputSpecMap(bundle, SIGNATURE_DEF);
        this.outputSpecMap = getOutputSpecMap(bundle, SIGNATURE_DEF);
    }

    //获取输入的名字与形状与类型
    private Map<String, TensorInfo> getInputSpecMap(SavedModelBundle bundle, String SIGNATURE_DEF) throws Exception {
        SignatureDef sig = MetaGraphDef.parseFrom(bundle.metaGraphDef()).getSignatureDefOrThrow(SIGNATURE_DEF);
        return sig.getInputsMap();
    }

    //获取输出的名字与形状与类型
    private Map<String, TensorInfo> getOutputSpecMap(SavedModelBundle bundle, String SIGNATURE_DEF) throws Exception {
        SignatureDef sig = MetaGraphDef.parseFrom(bundle.metaGraphDef()).getSignatureDefOrThrow(SIGNATURE_DEF);
        return sig.getOutputsMap();
    }


    /***
     *
     * @param sampleKeyList, 样本key的列表，用来决定顺序
     * @param sampleWithFeatureMap 样本key与该样本的featureMap组成的Map, 示例：
     *                             {“样本1”:{"f1":"wowowo","f2":"tututu"},
     *                             "样本2":{"f1":"ninini","f2":"hahaha"}}
     * @param <T>
     * @return 返回Map的key为输入特征的tensorflow operation节点名（不同于特征名），值为该特征的Tensor，顺序与sampleKeyList参数相同
     * 用来把行格式的featureMap，转变为列格式的Tensor，方便输入tensorflow的session，示例：
     * {"f1":Tensor<String>("wowowo","ninini"),
     * "f2":Tensor<String>("tututu","hahaha")}
     */
    private <T> Map<String,Tensor<?>> preprocessInput(List<T> sampleKeyList, Map<T,Map<String,String>> sampleWithFeatureMap) {
        Map<String,Tensor<?>> formattedInput=new HashMap<>();
        for (String featureKey : this.inputSpecMap.keySet()) {
            TensorInfo inputInfo = this.inputSpecMap.get(featureKey);
            String tfInputKey = inputInfo.getName();
            List<String> oneFeatureList = new ArrayList<>();
            for (T sampleKey : sampleKeyList) {
                Map<String, String> featureMap = sampleWithFeatureMap.get(sampleKey);
                oneFeatureList.add(featureMap.getOrDefault(featureKey,null));
            }
            Tensor<?> oneFeatureTensor=WuWuTFUtils.listToTensor(oneFeatureList,inputInfo);
            formattedInput.put(tfInputKey,oneFeatureTensor);
        }
        return formattedInput;
    }

    /***
     *
     * @param formattedInput 已经格式化好，按照列式存储的输入，顺序与sampleKeyList相同,示例：
     * {"f1":Tensor<String>("wowowo","ninini"),
     * "f2":Tensor<String>("tututu","hahaha")}
     * @return 返回的Map，key为导出pb时指定的字符串（不同于tensorflow operation的名称）,例如：
     * @tf.function
     * def serving_function(data):
     *     result={'prob':model_1(data)}
     *     return result
     *
     * output_func=serving_function.get_concrete_function(input_spec)
     * 如果把output_func作为saved model导出的signatures的话
     * 那么返回Map就只有1个key，名字为prob
     * 一个示例为：
     * {“ctr”:[[0.1],[0.3]]
     * "cvr":[[0.03],[0.09]]}
     * 此处输出使用二维数组，为了兼容多分类或者召回的情况
     */
    private Map<String, Object[][]> executeTfGraph(Map<String, Tensor<?>> formattedInput) {

        List<String> outputKeyList = new ArrayList<>();
        List<String> outputTfKeyList = new ArrayList<>();
        for (String outputKey : this.outputSpecMap.keySet()) {
            outputKeyList.add(outputKey);
            outputTfKeyList.add(this.outputSpecMap.get(outputKey).getName());
        }
        Session.Runner runner = this.bundle.session().runner();
        for (String inputTfKey : formattedInput.keySet()) {
            runner.feed(inputTfKey, formattedInput.get(inputTfKey));
        }
        for (String outputTfKey : outputTfKeyList) {
            runner.fetch(outputTfKey);
        }
        List<Tensor<?>> result = runner.run();
        Map<String, Object[][]> resultMap = new HashMap<>();
        // TODO：在哪里决定的输出是2维数组
        for (int i = 0; i < outputKeyList.size(); i++) {
            Tensor<?> tensorResult = result.get(i);
            resultMap.put(outputKeyList.get(i), WuWuTFUtils.tensorToArray(tensorResult));
        }
        //不close要内存爆炸，tensorflow真特么笨
        for (Tensor<?> tensor : formattedInput.values()) {
            tensor.close();
        }
        for (Tensor<?> tensor : result) {
            tensor.close();
        }
        return resultMap;
    }

    /***
     *
     * @param sampleKeyList 样本key的列表，决定了预测结果的顺序
     * @param rawOutputs 列式存储的预测结果, 顺序与sampleKeyList相同，示例：
     * {“ctr”:[[0.1],[0.3]]
     * "cvr":[[0.03],[0.09]]}
     * @param <T>
     * @return 行式存储的预估结果，示例：
     * {“样本1”:{"ctr":0.1,"cvr":0.3}
     * “样本2”:{"ctr":0.3,"cvr":0.09}}
     */
    private <T> Map<T, Map<String, WuWuPredResult>> formatOutputs(List<T> sampleKeyList, Map<String, Object[][]> rawOutputs) {
        Map<T, Map<String, WuWuPredResult>> resultMap = new HashMap<>();
        List<String> outputKeyList = new ArrayList<>(this.outputSpecMap.keySet());
        for (int i = 0; i < sampleKeyList.size(); i++) {
            Map<String, WuWuPredResult> outputMap = new HashMap<>();
            for (String outputKey : outputKeyList) {
                WuWuPredResult wuWuPredResult = new WuWuPredResult(outputKey);
                Object[] oneOutputs = rawOutputs.get(outputKey)[i];
                DataType dtype = this.outputSpecMap.get(outputKey).getDtype();
                if (oneOutputs.length==1) {
                    wuWuPredResult.setSingleResult(oneOutputs[0],dtype);
                } else {
                    wuWuPredResult.setArrayResult(oneOutputs,dtype);
                }
                outputMap.put(outputKey, wuWuPredResult);
            }
            resultMap.put(sampleKeyList.get(i),outputMap);
        }

        return resultMap;
    }

    /***
     * 预估的主接口
     * @param sampleWithFeatureMap 样本key与该样本的featureMap组成的Map, 示例：
     *                             {“样本1”:{"f1":"wowowo","f2":"tututu"},
     *                             "样本2":{"f1":"ninini","f2":"hahaha"}}
     * @param <T>
     * @return 行式存储的预估结果，示例：
     * {“样本1”:{"ctr":0.1,"cvr":0.3}
     * “样本2”:{"ctr":0.3,"cvr":0.09}}
     */
    public <T> Map<T,Map<String, WuWuPredResult>> predict(Map<T,Map<String,String>> sampleWithFeatureMap) {
        //第一步，把样本的key都取出来
        List<T> sampleKeyList= new ArrayList<>(sampleWithFeatureMap.keySet());
        System.out.println("sampleKeyList: " + sampleKeyList);
        //第二步，把特征从一行一行的变成一列一列的，key是特征名字，value是按照sampleKeyList排序的特征取值tensor
        Map<String,Tensor<?>> formattedInput = preprocessInput(sampleKeyList,sampleWithFeatureMap);
        System.out.println("formattedInput: " + JSON.toJSONString(formattedInput.keySet()));
        //第三步，把上述特征输入tensorflow中，得到按sampleKeyList顺序的预估结果，并关闭所有tensor, key是tf.function中输出的名字
        Map<String,Object[][]> rawOutputs = executeTfGraph(formattedInput);
        System.out.println("rawOutputs: " + JSON.toJSONString(rawOutputs));
        //第四步，把原始输出组装为PredResult对象，里面的key是tf.function中输出的名字
        Map<T,Map<String, WuWuPredResult>> predResults = formatOutputs(sampleKeyList,rawOutputs);
        return predResults;
    }

    public static void main(String[] args) throws Exception {
        Map<String,Map<String,String>> sampleWithFeatureMap=new HashMap<>();
        Map<String,String> featureMap=new HashMap<>();
        featureMap.put("f00001","a");
        featureMap.put("f00002","a");
        featureMap.put("f00003","a");
        featureMap.put("f00004","a");
        featureMap.put("f00005","a");
        featureMap.put("f00006","a");
        featureMap.put("f00007","a");
        featureMap.put("f00008","a");
        featureMap.put("f00009","a");
        featureMap.put("f00010","a");
        featureMap.put("f00011","a");
        featureMap.put("f00012","a");
        featureMap.put("f00013","a");
        featureMap.put("f00014","a");
        featureMap.put("f00015","a");
        featureMap.put("f00016","a");
        featureMap.put("f00017","a");
        featureMap.put("f00018","a");
        featureMap.put("f00019","a");
        featureMap.put("f00020","a");
        featureMap.put("f00021","a");
        featureMap.put("f00022","a");
        featureMap.put("f00023","a");
        sampleWithFeatureMap.put("sample1",featureMap);
        sampleWithFeatureMap.put("sample2",featureMap);
        sampleWithFeatureMap.put("sample3",featureMap);
        WuWuLocalTFModel model=new WuWuLocalTFModel();
        model.loadLatestModel("/Users/pengzhixiong/Downloads/pengzong/full");
        Map<String, Map<String, WuWuPredResult>> result = model.predict(sampleWithFeatureMap);

        System.out.println(result);
    }


}
