package cn.com.duiba.nezha.alg.api.model;

import cn.com.duiba.nezha.alg.api.model.util.E2ETFUtils;
import cn.com.duiba.nezha.alg.api.model.util.TensorflowUtils;
import com.alibaba.fastjson.JSON;
//import com.sun.jmx.mbeanserver.MXBeanMapping;
import jdk.nashorn.internal.ir.debug.ObjectSizeCalculator;
import lombok.Data;
import org.apache.commons.lang.StringUtils;
import org.slf4j.LoggerFactory;
import org.tensorflow.SavedModelBundle;
import org.tensorflow.Session;
import org.tensorflow.Tensor;
import org.tensorflow.Tensors;
import org.tensorflow.framework.DataType;
import org.tensorflow.framework.MetaGraphDef;
import org.tensorflow.framework.SignatureDef;
import org.tensorflow.framework.TensorInfo;

import java.lang.management.ManagementFactory;
import java.lang.management.MemoryMXBean;
import java.lang.management.MemoryUsage;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static java.nio.charset.StandardCharsets.UTF_8;

/**
 * Adx本地PB模型类
 *
 * @author caiyida@duiba.com.cn
 * @date 2022/3/24
 */
@Data
public class AdxLocalTFModel {


    private static final org.slf4j.Logger logger = LoggerFactory.getLogger(E2ELocalTFModel.class);
    //默认导出pb时候的签名，基本不变
    private static final String SIGNATURE_DEF = "serving_default";
    //java跟tf交互的入口，取输入输出建session全靠它（在load模型的时候获取到的）
    private SavedModelBundle bundle;
    //模型的路径
    private String path;
    //模型创建的时间戳
    private Long version;
    //输入的名字跟对应的TensorInfo，TensorInfo里面存了形状与类型
    private Map<String, TensorInfo> inputSpecMap;
    //输出的名字跟对应的TensorInfo，TensorInfo里面存了形状与类型
    private Map<String, TensorInfo> outputSpecMap;

    //从路径加载最新模型
    public void loadLatestModel(String path) throws Exception {
        Long lastVersion = E2ETFUtils.getLastVersion(path);
        SavedModelBundle bundle = E2ETFUtils.loadModel(path, lastVersion + "");
        logger.info("loadModel ,path=" + path + "with version=" + lastVersion);
        setModel(bundle);
        this.path = path;
        this.version = lastVersion;
    }

    //关闭模型的方法，不然会一直占用内存
    public void close() throws Exception {
        if (bundle != null) {
            logger.info("close model  ,path=" + path + "with version=" + version);
            logger.info("bundle.size before close :" + ObjectSizeCalculator.getObjectSize(bundle));
            bundle.close();
            bundle = null;
        }
    }

    //把仨核心变量配置上
    private void setModel(SavedModelBundle bundle) throws Exception {
        this.bundle = bundle;
        this.inputSpecMap = getInputSpecMap(bundle);
        this.outputSpecMap = getOutputSpecMap(bundle);
    }

    //获取输入的名字与形状与类型
    private Map<String, TensorInfo> getInputSpecMap(SavedModelBundle bundle) throws Exception {
        SignatureDef sig = MetaGraphDef.parseFrom(bundle.metaGraphDef()).getSignatureDefOrThrow(AdxLocalTFModel.SIGNATURE_DEF);
        return sig.getInputsMap();
    }

    //获取输出的名字与形状与类型
    private Map<String, TensorInfo> getOutputSpecMap(SavedModelBundle bundle) throws Exception {
        SignatureDef sig = MetaGraphDef.parseFrom(bundle.metaGraphDef()).getSignatureDefOrThrow(AdxLocalTFModel.SIGNATURE_DEF);
        return sig.getOutputsMap();
    }

    /**
     * @param sampleKeyList
     * @param sampleWithFeatureMap
     * @param <T1>
     * @return
     */
    private <T1> Map<String, Tensor<?>> preprocessInput(List<T1> sampleKeyList, Map<T1, Map<String, String>> sampleWithFeatureMap) {
        Map<String, Tensor<?>> formattedInput = new HashMap<>();
        for (String featureKey : this.inputSpecMap.keySet()) {
            TensorInfo inputInfo = this.inputSpecMap.get(featureKey);
            String tfInputKey = inputInfo.getName();
            List<String> oneFeatureList = new ArrayList<>();
            for (T1 sampleKey : sampleKeyList) {
                Map<String, String> featureMap = sampleWithFeatureMap.get(sampleKey);
                oneFeatureList.add(featureMap.getOrDefault(featureKey, null));
            }
            Tensor<?> oneFeatureTensor = AdxLocalTFModel.listToTensor(oneFeatureList, inputInfo);
            formattedInput.put(tfInputKey, oneFeatureTensor);
        }
        return formattedInput;
    }

    /***
     *
     * @param formattedInput 已经格式化好，按照列式存储的输入，顺序与sampleKeyList相同,示例：
     * {"f1":Tensor<String>("wowowo","ninini"),
     * "f2":Tensor<String>("tututu","hahaha")}
     * @return 返回的Map，key为导出pb时指定的字符串（不同于tensorflow operation的名称）,例如：
     * @tf.function
     * def serving_function(data):
     *     result={'prob':model_1(data)}
     *     return result
     *
     * output_func=serving_function.get_concrete_function(input_spec)
     * 如果把output_func作为saved model导出的signatures的话
     * 那么返回Map就只有1个key，名字为prob
     * 一个示例为：
     * {“ctr”:[[0.1],[0.3]]
     * "cvr":[[0.03],[0.09]]}
     * 此处输出使用二维数组，为了兼容多分类或者召回的情况
     */
    private <T> Map<String, T[][]> executeTfGraph(Map<String, Tensor<?>> formattedInput) {

        Map<String, T[][]> resultMap;
        List<Tensor<?>> result = null;
        try {
            List<String> outputKeyList = new ArrayList<>();
            List<String> outputTfKeyList = new ArrayList<>();
            for (String outputKey : this.outputSpecMap.keySet()) {
                outputKeyList.add(outputKey);
                outputTfKeyList.add(this.outputSpecMap.get(outputKey).getName());
            }
            Session.Runner runner = this.bundle.session().runner();
            for (String inputTfKey : formattedInput.keySet()) {
                runner.feed(inputTfKey, formattedInput.get(inputTfKey));
            }
            for (String outputTfKey : outputTfKeyList) {
                runner.fetch(outputTfKey);
            }
            result = runner.run();
            resultMap = new HashMap<>();
            for (int i = 0; i < outputKeyList.size(); i++) {
                Tensor<?> tensorResult = result.get(i);
                resultMap.put(outputKeyList.get(i), tensorToArray(tensorResult));
            }
        } finally {
            TensorflowUtils.closeQuitely(formattedInput.values());
            TensorflowUtils.closeQuitely(result);
        }

        return resultMap;
    }

    /***
     *
     * @param sampleKeyList 样本key的列表，决定了预测结果的顺序
     * @param rawOutputs 列式存储的预测结果, 顺序与sampleKeyList相同，示例：
     * {“ctr”:[[0.1],[0.3]]
     * "cvr":[[0.03],[0.09]]}
     * @return 行式存储的预估结果，示例：
     * {“样本1”:{"ctr":0.1,"cvr":0.3}
     * “样本2”:{"ctr":0.3,"cvr":0.09}}
     */
    private <T1, T2> Map<String, Map<T1, T2[]>> formatOutputs(List<T1> sampleKeyList, Map<String, T2[][]> rawOutputs) {
        Map<String, Map<T1, T2[]>> resultMap = new HashMap<>();
        List<String> outputKeyList = new ArrayList<>(this.outputSpecMap.keySet());
        for (String outputKey : outputKeyList) {
            Map<T1, T2[]> outputMap = new HashMap<>();
            for (int i = 0; i < sampleKeyList.size(); i++) {
                T2[] oneOutputs = rawOutputs.get(outputKey)[i];
                outputMap.put(sampleKeyList.get(i), oneOutputs);
            }
            resultMap.put(outputKey, outputMap);
        }
        return resultMap;
    }

    /**
     * @param sampleWithFeatureMap 带featureMap的样本
     * @param <T1>                 样本key
     * @param <T2>                 模型输出的类型
     * @return <预估值类型，<样本id,预估值>>
     */
    public <T1, T2> Map<String, Map<T1, T2[]>> predict(Map<T1, Map<String, String>> sampleWithFeatureMap) {
        //第一步，把样本的key都取出来
        List<T1> sampleKeyList = new ArrayList<>(sampleWithFeatureMap.keySet());
        //第二步，把特征从一行一行的变成一列一列的，key是特征名字，value是按照sampleKeyList排序的特征取值tensor
        Map<String, Tensor<?>> formattedInput = preprocessInput(sampleKeyList, sampleWithFeatureMap);
        //第三步，把上述特征输入tensorflow中，得到按sampleKeyList顺序的预估结果，并关闭所有tensor, key是tf.function中输出的名字
        Map<String, T2[][]> rawOutputs = executeTfGraph(formattedInput);
        //第四步，把原始输出组装为PredResult对象，里面的key是tf.function中输出的名字
        Map<String, Map<T1, T2[]>> predResults = formatOutputs(sampleKeyList, rawOutputs);
        return predResults;
    }

    /***
     * 把字符串的java list换成Tensor
     * 只支持Float跟String两种类型
     * @param input 输入的字符串List
     * @param inputInfo 包括了需要的输入类型，字符串类型默认值为“”，Float类型默认值为0.0
     * @return 转好的Tensor
     */
    public static Tensor<?> listToTensor(List<String> input, TensorInfo inputInfo) {
        DataType dtype = inputInfo.getDtype();
        int inputSize = input.size();
        if (dtype == DataType.DT_FLOAT) {
            float[] arrayInput = new float[inputSize];
            for (int i = 0; i < input.size(); i++) {
                String s = input.get(i);
                float inputWithType = 0.0f;
                if (s != null) {
                    try {
                        inputWithType = Float.parseFloat(s);
                    } catch (NumberFormatException e) {
                        logger.warn("listToTensor exception", e);
                    }
                }
                arrayInput[i] = inputWithType;
            }
            return Tensors.create(arrayInput);
        } else if (dtype == DataType.DT_STRING) {
            byte[][] arrayInput = new byte[inputSize][];
            for (int i = 0; i < input.size(); i++) {
                String s = input.get(i);
                String inputWithType = "";
                if (s != null) {
                    inputWithType = s;
                }
                arrayInput[i] = inputWithType.getBytes(UTF_8);
            }
            return Tensors.create(arrayInput);
        } else {
            return null;
        }
    }

    /**
     * @param tensorResult
     * @param <T>
     * @return
     */
    private <T> T[][] tensorToArray(Tensor<?> tensorResult) {
        long[] tensorShape = tensorResult.shape();
        if (org.tensorflow.DataType.STRING == tensorResult.dataType()) {
            byte[][][] byteListResult = new byte[(int) tensorShape[0]][(int) tensorShape[1]][];
            tensorResult.copyTo(byteListResult);
            String[][] listResult = new String[(int) tensorShape[0]][(int) tensorShape[1]];
            for (int p = 0; p < byteListResult.length; ++p) {
                for (int q = 0; q < byteListResult[p].length; ++q) {
                    listResult[p][q] = new String(byteListResult[p][q], UTF_8);
                }
            }
            return (T[][]) listResult;
        } else if (org.tensorflow.DataType.FLOAT == tensorResult.dataType()) {
            float[][] listResult = new float[(int) tensorShape[0]][(int) tensorShape[1]];
            tensorResult.copyTo(listResult);
            //由于tensor不支持copyTo一个Float的数组，只能这样了
            Float[][] listResult1 = new Float[(int) tensorShape[0]][(int) tensorShape[1]];
            for (int i = 0; i < listResult1.length; i++) {
                for (int j = 0; j < listResult1[i].length; j++) {
                    listResult1[i][j] = listResult[i][j];
                }
            }
            return (T[][]) listResult1;
        }
        return null;
    }

    public static void main(String[] args) throws Exception {
//        MemoryMXBean memoryMXBean = ManagementFactory.getMemoryMXBean();
//        MemoryUsage heapMemoryUsage = memoryMXBean.getHeapMemoryUsage();
//        MemoryUsage nonHeapMemoryUsage = memoryMXBean.getNonHeapMemoryUsage();
//        System.out.println("heap used "+heapMemoryUsage.getUsed() +" no heap used "+nonHeapMemoryUsage.getUsed());
        AdxLocalTFModel adxLocalTFModel = new AdxLocalTFModel();
        adxLocalTFModel.loadLatestModel("/Users/panhangyi/mid-adx-esmm-ctclk-phy");
//        MemoryUsage heapMemoryUsage1 = memoryMXBean.getHeapMemoryUsage();
//        MemoryUsage nonHeapMemoryUsage1 = memoryMXBean.getNonHeapMemoryUsage();
//        System.out.println("heap used "+heapMemoryUsage1.getUsed() +" no heap used "+nonHeapMemoryUsage1.getUsed());


//        HashMap<String, String> featureMap = new HashMap<>();
//        ctr:0.01, click:0
//        String str = "{\"f03014\":\"2\",\"f03015\":\"0\",\"f03012\":\"36\",\"f03013\":\"8\",\"f03018\":\"0\",\"f03019\":\"0\",\"f03016\":\"0\",\"f03017\":\"0\",\"f309040\":\"0.0\",\"f414004\":\"0\",\"f3010010\":\"33034\",\"f03021\":\"0\",\"f03020\":\"0\",\"f5020101\":\"2\",\"f5020103\":\"1\",\"f309050\":\"0.0\",\"f5020102\":\"1\",\"f5020105\":\"1003\",\"f5020104\":\"1003\",\"f5020106\":\"1003\",\"f413004\":\"0\",\"f2010010\":\"2726\",\"f5020010\":\"3\",\"f03003\":\"2\",\"f03004\":\"1\",\"f03002\":\"4\",\"f03007\":\"30\",\"f03008\":\"8\",\"f03005\":\"0\",\"f1010020\":\"36\",\"f7040050\":\"5\",\"f03009\":\"1\",\"f3016024\":\"103783\",\"f3016025\":\"55326\",\"f3016022\":\"329207\",\"f4010010\":\"\",\"f3016023\":\"393878\",\"f0109\":\"021001\",\"f7040060\":\"0\",\"f409004\":\"0\",\"f3016020\":\"24268\",\"f3010161\":\"9013\",\"f3016021\":\"12727\",\"f3010160\":\"31080\",\"f5020030\":\"0\",\"f0104\":\"020702\",\"f0467\":\"ac0b71be,3xiaxs6semgtwwg\",\"f0103\":\"010502\",\"f0466\":\"1891,2726,2725\",\"f0102\":\"01020302\",\"f0465\":\"159,129\",\"f0101\":\"010101\",\"f0108\":\"02093205\",\"f0107\":\"020101\",\"f0106\":\"02083205\",\"f0469\":\"159\",\"f0468\":\"3xvdvn8e3smxcdw,3xa7n2p2fyg6uug,3xa4qq6rmmnztrk,3xjba9wd4rtenbw,3x65hkewamncpk4,3xarhmrry7b8e5k,3xnbu3s8ix473mi,3xwhijzkaxb69tm,3x6kbcab2mha3wa,3xb64tsmkrku3ns,3x37sjsabn6zvcm,3xfuwtckv2beevc,3x2pzyanrqkstj6,3xp2jqh2r5piv4g,3xq9p5zcy9n3bwq,3xwceykzsepmx6g,15099950031199838574,3xzsj8cjsznqr7i,3xx36ecqqxnz2d4,3xrd3eqghz42ebm,3xf3682ujrdhytw,14341435844123720366,3x6aiixrg6zzeba,3x5335ssy5ffr34,3x4zqqgeaca9usw,3xaqx9egu6a4fcw,3xx526aye4a5am4,3xzscndnsnbp9k4,3xs3j9h28yqh8hm,3x6qjh5dp7etrty,3xkj3zzzcftsbes,3xkpmx8dcyrzchs,3xend87pateq2aq,3x2pkxa8ztif8di,3xpttncaxb3z57k,3x8smqt6eii3jze,3xqhqrnqdgxb2dc,3xh2szrt477z8cw,3xb29awzdsc9a3u,3xgmdaqgjgt5eqy,3xubfdfczun2ub2,3xtir376jatptxs,3xpemqxdigak2r4,3xj2m7eyfz9q5ba,3x4v9suam2xyz9g,3xr4zaqd3fgtevg,3xz6b6n2j3fqpr2,3xkdpmd2x5bk4y6,3xyrzyi4esagdda,3xiq4dywniahs5g,3x9wcv4g6q397b2,3xng5rudytivjiw,3xggvshkghkjhqi,3xnjfhnxhj75nvu,3x9rkv2si2s45s9\",\"f4010121\":\"23\",\"f5010110\":\"4\",\"f2010060\":\"3xqhqrnqdgxb2dc\",\"f410004\":\"0\",\"f2010061\":\"3xiaxs6semgtwwg\",\"f5020020\":\"3\",\"f6010020\":\"\",\"f4010122\":\"7\",\"f4010031\":\"82\",\"f4010030\":\"3205\",\"f4010032\":\"1\",\"f7040080\":\"2\",\"f3010144\":\"7\",\"f3010143\":\"8\",\"f3010142\":\"4\",\"f6010010\":\"\",\"f3010141\":\"5\",\"f5020050\":\"3\",\"f3010149\":\"5331\",\"f3010148\":\"10196\",\"f0001\":\"23\",\"f3010147\":\"379\",\"f3010146\":\"4675\",\"f3010145\":\"4\",\"f0003\":\"1\",\"f0002\":\"7\",\"f4010020\":\"153.37.203.127\",\"f5010135\":\"33002000,23002000,24002000,39002000,29002000,41002000,35002000,61002000,26002000,21002000,31002000,30002000,46002000,45002000,37002000,20002000,40002000,28002000,34002000,38002000\",\"f3016013\":\"1843\",\"f5010134\":\"33002000,23002000,24002000,39002000,29002000,41002000,35002000,61002000,26002000,21002000,31002000,30002000,46002000,45002000,37002000,20002000,40002000,28002000,34002000,38002000\",\"f3016014\":\"16108\",\"f5010133\":\"33002000,23002000,24002000,39002000,29002000,41002000,35002000,61002000,26002000,21002000,31002000,30002000,46002000,45002000,37002000,20002000,40002000,28002000,34002000,38002000\",\"f3016011\":\"10882\",\"f4010021\":\"DYN\",\"f3016012\":\"3585\",\"f3016017\":\"1716\",\"f3016018\":\"103035\",\"f7040090\":\"0\",\"f3016015\":\"11026\",\"f5010136\":\"33002000,23002000,24002000,39002000,29002000,41002000,35002000,61002000,26002000,21002000,31002000,30002000,46002000,45002000,37002000,20002000,40002000,28002000,34002000,38002000\",\"f3016016\":\"3385\",\"f3010155\":\"64382\",\"f3010154\":\"122309\",\"f3010153\":\"4514\",\"f3010152\":\"58545\",\"f3010151\":\"666\",\"f5020040\":\"3\",\"f3010150\":\"1383\",\"f3016010\":\"12002\",\"f0470\":\"2726\",\"f3010159\":\"826\",\"f0474\":\"2726\",\"f3010158\":\"2895\",\"f0473\":\"159\",\"f3010157\":\"9776\",\"f6060011\":\"0\",\"f0472\":\"3xwceykzsepmx6g,3x65hkewamncpk4,3xx36ecqqxnz2d4,3x2pkxa8ztif8di,3x4v9suam2xyz9g,3xr4zaqd3fgtevg,3xz6b6n2j3fqpr2,3xf3682ujrdhytw,3xpttncaxb3z57k,3xkdpmd2x5bk4y6,3xiq4dywniahs5g,3xqhqrnqdgxb2dc,3xfuwtckv2beevc,3xh2szrt477z8cw,3xnjfhnxhj75nvu,3xaqx9egu6a4fcw\",\"f3010156\":\"19348\",\"f6060012\":\"0\",\"f0471\":\"3xiaxs6semgtwwg\",\"f6060013\":\"0\",\"f6060014\":\"1\",\"f6060015\":\"1\",\"f3016019\":\"78201\",\"f0113\":\"021201\",\"f0476\":\"3xr4zaqd3fgtevg\",\"f0112\":\"02113205\",\"f0475\":\"3xiaxs6semgtwwg\",\"f5010010\":\"010101\",\"f0309\":\"1\",\"f0308\":\"3\",\"f0307\":\"5\",\"f4010050\":\"VIVO\",\"f411004\":\"0\",\"f5020070\":\"3\",\"f3030011\":\"\",\"f0302\":\"2\",\"f0301\":\"0\",\"f7020010\":\"com.achievo.vipshop,com.xunmeng.pinduoduo,com.tmri.app.main,com.sankuai.meituan,com.ss.android.ugc.aweme.lite,com.jingdong.app.mall,com.ss.android.article.lite,com.tencent.mm,com.baidu.searchbox.lite,com.autonavi.minimap,com.taobao.litetao,com.eg.android.AlipayGphone,com.jd.jrapp,com.yz.xfyzc2,com.taobao.taobao,cn.gov.tax.its,com.snda.wifilocating,com.kuaishou.nebula\",\"f0306\":\"0\",\"f0305\":\"0\",\"f0304\":\"1\",\"f0303\":\"2\",\"f5010036\":\"3\",\"f5010035\":\"12\",\"f5010034\":\"17\",\"f5010033\":\"20\",\"f5020060\":\"0\",\"f5010031\":\"010502\",\"f5010030\":\"01020302\",\"f4010070\":\"0\",\"f5020090\":\"0\",\"f0320\":\"0\",\"f0441\":\"159\",\"f0440\":\"3xgmdaqgjgt5eqy,3xubfdfczun2ub2,3xjba9wd4rtenbw,3xzscndnsnbp9k4,3x65hkewamncpk4,3x6qjh5dp7etrty,3xarhmrry7b8e5k,3xr4zaqd3fgtevg,3x6aiixrg6zzeba,3x9wcv4g6q397b2,3xqhqrnqdgxb2dc,3x5335ssy5ffr34,3xq9p5zcy9n3bwq,3xaqx9egu6a4fcw\",\"f307040\":\"0.0\",\"f0203\":\"0\",\"f0324\":\"6\",\"f0445\":\"159\",\"f0202\":\"0\",\"f0323\":\"6\",\"f0444\":\"3xr4zaqd3fgtevg,3xqhqrnqdgxb2dc,3x65hkewamncpk4,3xaqx9egu6a4fcw\",\"f0201\":\"0\",\"f0322\":\"0\",\"f0443\":\"3xiaxs6semgtwwg\",\"f0321\":\"0\",\"f0442\":\"2726\",\"f0327\":\"0\",\"f0448\":\"3xr4zaqd3fgtevg\",\"f0326\":\"3\",\"f0447\":\"3xiaxs6semgtwwg\",\"f0204\":\"0\",\"f0325\":\"4\",\"f0446\":\"2726\",\"f0319\":\"0\",\"f0318\":\"0\",\"f0439\":\"3xiaxs6semgtwwg\",\"f4010060\":\"vivoX60\",\"f5020080\":\"3\",\"f3060017\":\"4\",\"f3060016\":\"5\",\"f307050\":\"0.0\",\"f3060013\":\"8\",\"f5010050\":\"acd37714e33962711601044a32fdedef\",\"f0313\":\"3\",\"f3060012\":\"5\",\"f0312\":\"5\",\"f3060015\":\"13\",\"f0311\":\"0\",\"f3060014\":\"13\",\"f0310\":\"0\",\"f0317\":\"0\",\"f0438\":\"2726,2725\",\"f0316\":\"0\",\"f0437\":\"159\",\"f3060011\":\"8\",\"f0315\":\"0\",\"f0314\":\"2\",\"f4010091\":\"2\",\"f1010010\":\"159\",\"f5010060\":\"0\",\"f3020000\":\"0\",\"f5010079\":\"3\",\"f5010078\":\"7\",\"f5010077\":\"0\",\"f5030101\":\"159&1003\",\"f5030102\":\"2726&1003\",\"f5030103\":\"33034&1003\",\"f4010080\":\"2\",\"f5010070\":\"0\",\"f5010076\":\"0\",\"f5010075\":\"0\",\"f5010074\":\"0\",\"f7040110\":\"2\",\"f6010090\":\"6000040\",\"f5010082\":\"0\",\"f5010081\":\"0\",\"f5010080\":\"0\",\"f7040120\":\"0\",\"f412004\":\"0\",\"f5010090\":\"0\"}";
//        String str = "{\"f309040\":\"0.0\",\"f414004\":\"0\",\"f3010010\":\"32545\",\"f5020101\":\"0\",\"f5020103\":\"0\",\"f309050\":\"0.0\",\"f5020102\":\"0\",\"f5020105\":\"1000\",\"f5020104\":\"1000\",\"f5020106\":\"1000\",\"f413004\":\"0\",\"f2010010\":\"2636\",\"f5020010\":\"0\",\"f1010020\":\"44\",\"f7040050\":\"0\",\"f3016024\":\"221290\",\"f3016025\":\"120527\",\"f3016022\":\"1261178\",\"f4010010\":\"Mozilla/5.0 (iPhone; CPU iPhone OS 15_6 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Mobile/15E148 MicroMessenger/8.0.28(0x18001c2c) NetType/4G Language/zh_CN\",\"f3016023\":\"457932\",\"f7040060\":\"0\",\"f409004\":\"0\",\"f3016020\":\"113899\",\"f3010161\":\"113899\",\"f3016021\":\"62498\",\"f3010160\":\"234457\",\"f5020030\":\"0\",\"f4010121\":\"15\",\"f5010110\":\"1\",\"f2010060\":\"411143\",\"f410004\":\"0\",\"f5020020\":\"0\",\"f4010122\":\"7\",\"f4010030\":\"0\",\"f7040080\":\"0\",\"f3010144\":\"9\",\"f3010143\":\"10\",\"f3010142\":\"7\",\"f3010141\":\"8\",\"f5020050\":\"0\",\"f3010149\":\"78286\",\"f3010148\":\"220104\",\"f0001\":\"15\",\"f3010147\":\"7173\",\"f3010146\":\"69150\",\"f3010145\":\"7\",\"f0003\":\"1\",\"f0002\":\"7\",\"f5010135\":\"\",\"f3016013\":\"7778\",\"f5010134\":\"\",\"f3016014\":\"220104\",\"f5010133\":\"\",\"f3016011\":\"25292\",\"f3016012\":\"12710\",\"f3016017\":\"23726\",\"f3016018\":\"666218\",\"f7040090\":\"0\",\"f3016015\":\"78286\",\"f5010136\":\"\",\"f3016016\":\"39700\",\"f3010155\":\"457932\",\"f3010154\":\"1261178\",\"f3010153\":\"62498\",\"f3010152\":\"666218\",\"f3010151\":\"23726\",\"f5020040\":\"0\",\"f3010150\":\"39700\",\"f3016010\":\"75969\",\"f3010159\":\"11655\",\"f3010158\":\"23170\",\"f3010157\":\"120527\",\"f6060011\":\"0\",\"f3010156\":\"221290\",\"f6060012\":\"0\",\"f6060013\":\"0\",\"f6060014\":\"1\",\"f6060015\":\"1\",\"f3016019\":\"234457\",\"f5010010\":\"010105\",\"f0309\":\"0\",\"f0308\":\"0\",\"f0307\":\"0\",\"f411004\":\"0\",\"f5020070\":\"0\",\"f3030011\":\"\",\"f0302\":\"0\",\"f0301\":\"0\",\"f0306\":\"0\",\"f0305\":\"0\",\"f0304\":\"0\",\"f0303\":\"0\",\"f5010036\":\"0\",\"f5010035\":\"0\",\"f5010034\":\"0\",\"f5010033\":\"0\",\"f5020060\":\"0\",\"f6010102\":\"[1]\",\"f4010070\":\"3\",\"f5020090\":\"0\",\"f0320\":\"0\",\"f307040\":\"0.0\",\"f0203\":\"0\",\"f0324\":\"0\",\"f0202\":\"0\",\"f0323\":\"0\",\"f0201\":\"0\",\"f0322\":\"0\",\"f0321\":\"0\",\"f0327\":\"0\",\"f0326\":\"0\",\"f0204\":\"0\",\"f0325\":\"0\",\"f0319\":\"0\",\"f0318\":\"0\",\"f5020080\":\"0\",\"f3060017\":\"1\",\"f3060016\":\"1\",\"f307050\":\"0.0\",\"f3060013\":\"1\",\"f5010050\":\"8e3H3Ny3GBDTLn3T\",\"f0313\":\"0\",\"f3060012\":\"1\",\"f0312\":\"0\",\"f3060015\":\"1\",\"f0311\":\"0\",\"f3060014\":\"1\",\"f0310\":\"0\",\"f0317\":\"0\",\"f0316\":\"0\",\"f3060011\":\"1\",\"f0315\":\"0\",\"f0314\":\"0\",\"f4010091\":\"0\",\"f1010010\":\"190\",\"f5010060\":\"0\",\"f3020000\":\"0\",\"f5010079\":\"0\",\"f5010078\":\"0\",\"f5010077\":\"0\",\"f5030101\":\"190&1000\",\"f5030102\":\"2636&1000\",\"f5030103\":\"32545&1000\",\"f4010080\":\"0\",\"f5010070\":\"0\",\"f5010076\":\"0\",\"f5010075\":\"0\",\"f5010074\":\"0\",\"f6010090\":\"411143\",\"f5010082\":\"0\",\"f5010081\":\"0\",\"f5010080\":\"0\",\"f412004\":\"0\",\"f5010090\":\"0\"}";
//        ctr:0.01, click:0 adxr_id: c3e017b923f8e95a12bb3429e7b389b7-l8mqo40z-1103305889214923958_6000003
//        String str = "{\"f309040\":\"0.0\",\"f414004\":\"0\",\"f3010010\":\"32824\",\"f5020101\":\"0\",\"f5020103\":\"0\",\"f309050\":\"0.0\",\"f5020102\":\"0\",\"f5020105\":\"1000\",\"f5020104\":\"1000\",\"f5020106\":\"1000\",\"f413004\":\"0\",\"f2010010\":\"2724\",\"f5020010\":\"0\",\"f1010020\":\"36\",\"f7040050\":\"0\",\"f3016024\":\"381\",\"f3016025\":\"7304\",\"f3016022\":\"66780\",\"f4010010\":\"\",\"f3016023\":\"1073\",\"f0109\":\"021001\",\"f7040060\":\"0\",\"f409004\":\"0\",\"f3016020\":\"139\",\"f3010161\":\"139\",\"f3016021\":\"66\",\"f3010160\":\"310\",\"f5020030\":\"0\",\"f0104\":\"020701\",\"f0467\":\"3xg6gx9bxpxhe8y\",\"f0466\":\"2724,2722,2726,2725\",\"f0465\":\"159\",\"f0108\":\"02093302\",\"f0107\":\"020101\",\"f0106\":\"02083301\",\"f0469\":\"159\",\"f0468\":\"3xjnd2xpcmkwbbq,3xn85ksbm6c3st9,3xsb7jt4vqtbw8k,3x8p35b8r8z55dm,3xtp97d9qmqzgb6,3x35kh8r9wac9qm,3xdsvaq9gtc9tda,3x3ugenpqmuwfh6,3xyb8chdt64nk7y,3xanbns44ytmqqu,3xyb6swwfxmuvc9,3xwwkxfrweniq3a,3xcny68c5y7arpk,3xnrvwg2khqc682,3xtdptygknysbz2,3xb2v8fgvdwsuu4,3xkyjsp3yc8g6xy,3x5vjghif5wj7m4,3xkp6mhepqztfmq,3xnevs6z2x7m3ja\",\"f4010121\":\"12\",\"f5010110\":\"4\",\"f2010060\":\"3xid3w35m54m5rs\",\"f410004\":\"0\",\"f2010061\":\"3xg6gx9bxpxhe8y\",\"f5020020\":\"0\",\"f6010020\":\"\",\"f4010122\":\"7\",\"f4010031\":\"82\",\"f4010030\":\"3302\",\"f4010032\":\"1\",\"f7040080\":\"0\",\"f3010144\":\"2\",\"f3010143\":\"3\",\"f3010142\":\"1\",\"f6010010\":\"\",\"f3010141\":\"1\",\"f5020050\":\"0\",\"f3010149\":\"61\",\"f3010148\":\"355\",\"f0001\":\"12\",\"f3010147\":\"2\",\"f3010146\":\"96\",\"f3010145\":\"7\",\"f0003\":\"1\",\"f0002\":\"7\",\"f4010020\":\"223.104.161.215\",\"f5010135\":\"\",\"f3016013\":\"2\",\"f5010134\":\"\",\"f3016014\":\"355\",\"f5010133\":\"\",\"f3016011\":\"7\",\"f4010021\":\"MOB\",\"f3016012\":\"4\",\"f3016017\":\"14\",\"f3016018\":\"1867\",\"f7040090\":\"0\",\"f3016015\":\"61\",\"f5010136\":\"\",\"f3016016\":\"30\",\"f3010155\":\"1073\",\"f3010154\":\"7437\",\"f3010153\":\"66\",\"f3010152\":\"1867\",\"f3010151\":\"14\",\"f5020040\":\"0\",\"f3010150\":\"30\",\"f3016010\":\"85\",\"f0470\":\"2724,2722,2726\",\"f3010159\":\"4\",\"f3010158\":\"7\",\"f3010157\":\"189\",\"f6060011\":\"0\",\"f0472\":\"3xjnd2xpcmkwbbq,3xsb7jt4vqtbw8k,3x8p35b8r8z55dm,3xtp97d9qmqzgb6,3x35kh8r9wac9qm,3xdsvaq9gtc9tda,3x3ugenpqmuwfh6,3xanbns44ytmqqu,3xyb8chdt64nk7y,3xyb6swwfxmuvc9,3xwwkxfrweniq3a,3xcny68c5y7arpk,3xtdptygknysbz2,3xb2v8fgvdwsuu4,3xkyjsp3yc8g6xy,3x5vjghif5wj7m4,3xkp6mhepqztfmq,3xnevs6z2x7m3ja\",\"f3010156\":\"381\",\"f6060012\":\"0\",\"f0471\":\"3xg6gx9bxpxhe8y\",\"f6060013\":\"0\",\"f6060014\":\"1\",\"f6060015\":\"1\",\"f3016019\":\"310\",\"f5010010\":\"010105\",\"f0309\":\"0\",\"f0308\":\"0\",\"f0307\":\"0\",\"f4010050\":\"VIVO\",\"f411004\":\"0\",\"f5020070\":\"0\",\"f3030011\":\"\",\"f0302\":\"0\",\"f0301\":\"0\",\"f0306\":\"0\",\"f0305\":\"0\",\"f0304\":\"0\",\"f0303\":\"0\",\"f5010036\":\"0\",\"f5010035\":\"0\",\"f5010034\":\"0\",\"f5010033\":\"0\",\"f5020060\":\"0\",\"f5010030\":\"0\",\"f4010070\":\"0\",\"f5020090\":\"0\",\"f0320\":\"0\",\"f0441\":\"159\",\"f0440\":\"3x8p35b8r8z55dm,3xkyjsp3yc8g6xy,3xtp97d9qmqzgb6,3x35kh8r9wac9qm,3xyb8chdt64nk7y,3xanbns44ytmqqu,3xnevs6z2x7m3ja,3xwwkxfrweniq3a\",\"f307040\":\"0.0\",\"f0203\":\"0\",\"f0324\":\"3\",\"f0202\":\"0\",\"f0323\":\"4\",\"f0444\":\"3x8p35b8r8z55dm,3xtp97d9qmqzgb6,3xkyjsp3yc8g6xy,3x35kh8r9wac9qm,3xanbns44ytmqqu,3xyb8chdt64nk7y,3xnevs6z2x7m3ja,3xwwkxfrweniq3a\",\"f0201\":\"0\",\"f0322\":\"0\",\"f0443\":\"3xg6gx9bxpxhe8y\",\"f0321\":\"0\",\"f0442\":\"2724,2722,2726\",\"f0327\":\"0\",\"f0326\":\"1\",\"f0204\":\"0\",\"f0325\":\"1\",\"f0319\":\"0\",\"f0318\":\"0\",\"f0439\":\"3xg6gx9bxpxhe8y\",\"f4010060\":\"\",\"f5020080\":\"0\",\"f3060017\":\"1\",\"f3060016\":\"1\",\"f307050\":\"0.0\",\"f3060013\":\"1\",\"f5010050\":\"c3e017b923f8e95a12bb3429e7b389b7\",\"f0313\":\"0\",\"f3060012\":\"1\",\"f0312\":\"0\",\"f3060015\":\"1\",\"f0311\":\"0\",\"f3060014\":\"1\",\"f0310\":\"0\",\"f0317\":\"0\",\"f0438\":\"2724,2722,2726\",\"f0316\":\"0\",\"f0437\":\"159\",\"f3060011\":\"1\",\"f0315\":\"0\",\"f0314\":\"0\",\"f4010091\":\"0\",\"f1010010\":\"159\",\"f5010060\":\"0\",\"f3020000\":\"0\",\"f5010079\":\"0\",\"f5010078\":\"0\",\"f5010077\":\"0\",\"f5030101\":\"159&1000\",\"f5030102\":\"2724&1000\",\"f5030103\":\"32824&1000\",\"f4010080\":\"1\",\"f5010070\":\"0\",\"f5010076\":\"0\",\"f5010075\":\"0\",\"f5010074\":\"0\",\"f6010090\":\"6000003\",\"f5010082\":\"0\",\"f5010081\":\"0\",\"f5010080\":\"0\",\"f412004\":\"0\",\"f5010090\":\"0\"}";

//        ctr:0.01, click:0, adxr_id:9eafc192f0847acc83cf779fcd1ac715-l8mq7tp9-679920280222706050_30033003330
        String str = "{\"f6010070\":\"\",\"f309040\":\"0.0\",\"f414004\":\"0\",\"f3010010\":\"33366\",\"f5020101\":\"0\",\"f5020103\":\"0\",\"f309050\":\"0.0\",\"f5020102\":\"0\",\"f5020105\":\"1000\",\"f5020104\":\"1000\",\"f5020106\":\"1000\",\"f413004\":\"0\",\"f2010010\":\"3011\",\"f5020010\":\"0\",\"f1010020\":\"19\",\"f7040050\":\"0\",\"f3016024\":\"7734\",\"f3016025\":\"5592\",\"f3016022\":\"5331411\",\"f4010010\":\"Mozilla/5.0 (Linux; Android 10; V1911A Build/QP1A.190711.020; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/83.0.4103.106 Mobile Safari/537.36 Mobads\",\"f3016023\":\"25410\",\"f0109\":\"021003\",\"f7040060\":\"0\",\"f409004\":\"0\",\"f3016020\":\"7047\",\"f3010161\":\"4180\",\"f3016021\":\"5169\",\"f3010160\":\"13613\",\"f5020030\":\"0\",\"f0104\":\"020701\",\"f0467\":\"e67657ba\",\"f0103\":\"010502\",\"f0466\":\"1120,2927\",\"f0102\":\"010202\",\"f0465\":\"129,171\",\"f0101\":\"010101\",\"f0108\":\"02093203\",\"f0107\":\"020103\",\"f0106\":\"02083203\",\"f0469\":\"129\",\"f0468\":\"74441300498,7188856\",\"f4010121\":\"20\",\"f5010110\":\"3\",\"f2010060\":\"7820081\",\"f410004\":\"0\",\"f2010062\":\"3399\",\"f2010061\":\"e67657ba\",\"f5020020\":\"0\",\"f6010020\":\"com.devuni.flashlight\",\"f4010122\":\"7\",\"f4010031\":\"82\",\"f4010030\":\"3201\",\"f4010032\":\"1\",\"f7040080\":\"0\",\"f3010144\":\"6\",\"f3010143\":\"7\",\"f3010142\":\"4\",\"f6010010\":\"e67657ba\",\"f3010141\":\"6\",\"f5020050\":\"0\",\"f3010149\":\"5211\",\"f3010148\":\"663914\",\"f0001\":\"20\",\"f3010147\":\"640\",\"f3010146\":\"669127\",\"f3010145\":\"4\",\"f0003\":\"1\",\"f0002\":\"7\",\"f4010020\":\"49.94.180.83\",\"f5010135\":\"33002000,30002000,24002000,45002000,39002000,28002000,41002000,35002000,54002000,38002000\",\"f3016013\":\"1253\",\"f5010134\":\"33002000,30002000,24002000,45002000,39002000,28002000,41002000,35002000,54002000,38002000\",\"f3016014\":\"1545898\",\"f5010133\":\"33002000,30002000,24002000,45002000,39002000,28002000,41002000,35002000,54002000,38002000\",\"f3016011\":\"6178\",\"f4010021\":\"MOB\",\"f3016012\":\"1656\",\"f3016017\":\"1589\",\"f3016018\":\"4549490\",\"f7040090\":\"0\",\"f3016015\":\"9990\",\"f5010136\":\"33002000,30002000,24002000,45002000,39002000,28002000,41002000,35002000,54002000,38002000\",\"f3016016\":\"2173\",\"f3010155\":\"14633\",\"f3010154\":\"2543637\",\"f3010153\":\"3158\",\"f3010152\":\"2165297\",\"f3010151\":\"910\",\"f5020040\":\"0\",\"f3010150\":\"1217\",\"f3016010\":\"1009831\",\"f0470\":\"1120\",\"f3010159\":\"875\",\"f3010158\":\"3461\",\"f6060010\":\"3011&e67657ba\",\"f3010157\":\"3420\",\"f6060011\":\"10433\",\"f0472\":\"7188856\",\"f3010156\":\"4542\",\"f6060012\":\"56\",\"f0471\":\"e67657ba\",\"f6060013\":\"640\",\"f6060014\":\"4\",\"f6060015\":\"12\",\"f3016019\":\"23623\",\"f5010010\":\"010101\",\"f0309\":\"0\",\"f0308\":\"0\",\"f0307\":\"0\",\"f4010050\":\"vivo\",\"f411004\":\"0\",\"f5020070\":\"0\",\"f3030011\":\"\",\"f0302\":\"0\",\"f0301\":\"0\",\"f0306\":\"0\",\"f0305\":\"0\",\"f0304\":\"0\",\"f0303\":\"0\",\"f5010036\":\"0\",\"f5010035\":\"2\",\"f5010034\":\"10\",\"f5010033\":\"10\",\"f6010100\":\"1\",\"f6010101\":\"1\",\"f5020060\":\"0\",\"f5010031\":\"010502\",\"f5010030\":\"010202\",\"f4010071\":\"10.0.0\",\"f4010070\":\"0\",\"f5020090\":\"0\",\"f0320\":\"0\",\"f307040\":\"0.0\",\"f0203\":\"0\",\"f0324\":\"0\",\"f0202\":\"0\",\"f0323\":\"0\",\"f0201\":\"0\",\"f0322\":\"0\",\"f0321\":\"0\",\"f0327\":\"0\",\"f0326\":\"0\",\"f0204\":\"0\",\"f0325\":\"0\",\"f0319\":\"0\",\"f0318\":\"0\",\"f4010060\":\"V1911A\",\"f5020080\":\"0\",\"f3060017\":\"4\",\"f3060016\":\"5\",\"f307050\":\"0.0\",\"f3060013\":\"8\",\"f5010050\":\"9eafc192f0847acc83cf779fcd1ac715\",\"f0313\":\"0\",\"f3060012\":\"6\",\"f0312\":\"0\",\"f3060015\":\"4\",\"f0311\":\"0\",\"f3060014\":\"4\",\"f0310\":\"0\",\"f0317\":\"0\",\"f0316\":\"0\",\"f3060011\":\"14\",\"f0315\":\"0\",\"f3060010\":\"33366&e67657ba\",\"f0314\":\"0\",\"f4010091\":\"6\",\"f1010010\":\"129\",\"f5010060\":\"2\",\"f3020000\":\"0\",\"f5010079\":\"0\",\"f5010078\":\"0\",\"f5010077\":\"0\",\"f5030101\":\"129&1000\",\"f5030102\":\"3011&1000\",\"f5030103\":\"33366&1000\",\"f4010080\":\"0\",\"f5010070\":\"0\",\"f5010076\":\"0\",\"f5010075\":\"0\",\"f5010074\":\"0\",\"f6010090\":\"30033003330\",\"f5010082\":\"0\",\"f5010081\":\"0\",\"f5010080\":\"2\",\"f412004\":\"0\",\"f5010090\":\"0\"}";
        HashMap<String, String> featureMap = JSON.parseObject(str, HashMap.class);
//        featureMap.put("ft1","aaa");

        Map<Long, Map<String, String>> sample = new HashMap<>();
        sample.put(123L,featureMap);
        long startTIme = System.currentTimeMillis();
        Map<String, Map<Long, Float[]>> predict = adxLocalTFModel.predict(sample);
        long endTime = System.currentTimeMillis();
        System.out.println("耗时 "+(endTime-startTIme));
        System.out.println(predict.toString());

    }
}
