package cn.com.duiba.nezha.alg.api.model;

import cn.com.duiba.nezha.alg.api.model.util.E2EPredResult;
import cn.com.duiba.nezha.alg.api.model.util.E2ETFUtils;
import cn.com.duiba.nezha.alg.api.model.util.TensorflowUtils;
import cn.com.duiba.nezha.alg.api.model.util.WuWuTFUtils;
import jdk.nashorn.internal.ir.debug.ObjectSizeCalculator;
import lombok.Data;
import org.slf4j.LoggerFactory;
import org.tensorflow.SavedModelBundle;
import org.tensorflow.Session;
import org.tensorflow.Tensor;
import org.tensorflow.Tensors;
import org.tensorflow.framework.DataType;
import org.tensorflow.framework.MetaGraphDef;
import org.tensorflow.framework.SignatureDef;
import org.tensorflow.framework.TensorInfo;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static java.nio.charset.StandardCharsets.UTF_8;

/**
 * Adx本地PB模型类
 *
 * @author caiyida@duiba.com.cn
 * @date 2022/3/24
 */
@Data
public class AdxLocalTFModel<T1, T2> {


    private static final org.slf4j.Logger logger = LoggerFactory.getLogger(E2ELocalTFModel.class);
    //默认导出pb时候的签名，基本不变
    private static final String SIGNATURE_DEF = "serving_default";
    //java跟tf交互的入口，取输入输出建session全靠它（在load模型的时候获取到的）
    private SavedModelBundle bundle;
    //模型的路径
    private String path;
    //模型创建的时间戳
    private Long version;
    //输入的名字跟对应的TensorInfo，TensorInfo里面存了形状与类型
    private Map<String, TensorInfo> inputSpecMap;
    //输出的名字跟对应的TensorInfo，TensorInfo里面存了形状与类型
    private Map<String, TensorInfo> outputSpecMap;

    //从路径加载最新模型
    public void loadLatestModel(String path) throws Exception {
        Long lastVersion = E2ETFUtils.getLastVersion(path);
        SavedModelBundle bundle = E2ETFUtils.loadModel(path, lastVersion + "");
        logger.info("loadModel ,path=" + path + "with version=" + lastVersion);
        setModel(bundle);
        this.path = path;
        this.version = lastVersion;
    }

    //关闭模型的方法，不然会一直占用内存
    public void close() throws Exception {
        if (bundle != null) {
            logger.info("close model  ,path=" + path + "with version=" + version);
            logger.info("bundle.size before close :" + ObjectSizeCalculator.getObjectSize(bundle));
            bundle.close();
            bundle = null;
        }
    }

    //把仨核心变量配置上
    private void setModel(SavedModelBundle bundle) throws Exception {
        this.bundle = bundle;
        this.inputSpecMap = getInputSpecMap(bundle);
        this.outputSpecMap = getOutputSpecMap(bundle);
    }

    //获取输入的名字与形状与类型
    private Map<String, TensorInfo> getInputSpecMap(SavedModelBundle bundle) throws Exception {
        SignatureDef sig = MetaGraphDef.parseFrom(bundle.metaGraphDef()).getSignatureDefOrThrow(AdxLocalTFModel.SIGNATURE_DEF);
        return sig.getInputsMap();
    }

    //获取输出的名字与形状与类型
    private Map<String, TensorInfo> getOutputSpecMap(SavedModelBundle bundle) throws Exception {
        SignatureDef sig = MetaGraphDef.parseFrom(bundle.metaGraphDef()).getSignatureDefOrThrow(AdxLocalTFModel.SIGNATURE_DEF);
        return sig.getOutputsMap();
    }

    /**
     *
     * @param sampleKeyList
     * @param sampleWithFeatureMap
     * @param <T1>
     * @param <T2>
     * @return
     */
    private <T1, T2> Map<String, Tensor<?>> preprocessInput(List<T1> sampleKeyList, Map<T1, Map<String, String>> sampleWithFeatureMap) {
        Map<String, Tensor<?>> formattedInput = new HashMap<>();
        for (String featureKey : this.inputSpecMap.keySet()) {
            TensorInfo inputInfo = this.inputSpecMap.get(featureKey);
            String tfInputKey = inputInfo.getName();
            List<String> oneFeatureList = new ArrayList<>();
            for (T1 sampleKey : sampleKeyList) {
                Map<String, String> featureMap = sampleWithFeatureMap.get(sampleKey);
                oneFeatureList.add(featureMap.getOrDefault(featureKey, null));
            }
            Tensor<?> oneFeatureTensor = AdxLocalTFModel.listToTensor(oneFeatureList, inputInfo);
            formattedInput.put(tfInputKey, oneFeatureTensor);
        }
        return formattedInput;
    }

    /***
     *
     * @param formattedInput 已经格式化好，按照列式存储的输入，顺序与sampleKeyList相同,示例：
     * {"f1":Tensor<String>("wowowo","ninini"),
     * "f2":Tensor<String>("tututu","hahaha")}
     * @return 返回的Map，key为导出pb时指定的字符串（不同于tensorflow operation的名称）,例如：
     * @tf.function
     * def serving_function(data):
     *     result={'prob':model_1(data)}
     *     return result
     *
     * output_func=serving_function.get_concrete_function(input_spec)
     * 如果把output_func作为saved model导出的signatures的话
     * 那么返回Map就只有1个key，名字为prob
     * 一个示例为：
     * {“ctr”:[[0.1],[0.3]]
     * "cvr":[[0.03],[0.09]]}
     * 此处输出使用二维数组，为了兼容多分类或者召回的情况
     */
    private <T1,T2> Map<String, T2[][]> executeTfGraph(Map<String, Tensor<?>> formattedInput) {

        Map<String, T2[][]> resultMap;
        List<Tensor<?>> result = null;
        try {
            List<String> outputKeyList = new ArrayList<>();
            List<String> outputTfKeyList = new ArrayList<>();
            for (String outputKey : this.outputSpecMap.keySet()) {
                outputKeyList.add(outputKey);
                outputTfKeyList.add(this.outputSpecMap.get(outputKey).getName());
            }
            Session.Runner runner = this.bundle.session().runner();
            for (String inputTfKey : formattedInput.keySet()) {
                runner.feed(inputTfKey, formattedInput.get(inputTfKey));
            }
            for (String outputTfKey : outputTfKeyList) {
                runner.fetch(outputTfKey);
            }
            result = runner.run();
            resultMap = new HashMap<>();
            for (int i = 0; i < outputKeyList.size(); i++) {
                Tensor<?> tensorResult = result.get(i);
                resultMap.put(outputKeyList.get(i), tensorToArray(tensorResult));
            }
        } finally {
            TensorflowUtils.closeQuitely(formattedInput.values());
            TensorflowUtils.closeQuitely(result);
        }

        return resultMap;
    }

    /***
     *
     * @param sampleKeyList 样本key的列表，决定了预测结果的顺序
     * @param rawOutputs 列式存储的预测结果, 顺序与sampleKeyList相同，示例：
     * {“ctr”:[[0.1],[0.3]]
     * "cvr":[[0.03],[0.09]]}
     * @param <T>
     * @return 行式存储的预估结果，示例：
     * {“样本1”:{"ctr":0.1,"cvr":0.3}
     * “样本2”:{"ctr":0.3,"cvr":0.09}}
     */
    private <T> Map<T, Map<String, E2EPredResult>> formatOutputs(List<T> sampleKeyList, Map<String, Object[][]> rawOutputs) {
        Map<T, Map<String, E2EPredResult>> resultMap = new HashMap<>();
        List<String> outputKeyList = new ArrayList<>(this.outputSpecMap.keySet());
        for (int i = 0; i < sampleKeyList.size(); i++) {
            Map<String, E2EPredResult> outputMap = new HashMap<>();
            Map<T,T> outputMap2 = new HashMap<>(5);
            for (String outputKey : outputKeyList) {
                E2EPredResult predResult = new E2EPredResult(outputKey);
                Object[] oneOutputs = rawOutputs.get(outputKey)[i];
                DataType dtype = this.outputSpecMap.get(outputKey).getDtype();
                if (oneOutputs.length == 1) {
                    predResult.setSingleResult(oneOutputs[0], dtype);
                } else {
                    predResult.setArrayResult(oneOutputs, dtype);
                }
                outputMap.put(outputKey, predResult);
            }
            resultMap.put(sampleKeyList.get(i), outputMap);
        }

        return resultMap;
    }

    private <T1, T2> Map<T1, Map<String, T2[]>> formatOutputs2(List<T1> sampleKeyList, Map<String, T2[][]> rawOutputs) {
        Map<T1, Map<String, T2[]>> resultMap = new HashMap<>();
        List<String> outputKeyList = new ArrayList<>(this.outputSpecMap.keySet());
        for (int i = 0; i < sampleKeyList.size(); i++) {
            Map<String, T2[]> outputMap = new HashMap<>();
            for (String outputKey : outputKeyList) {
                T2[] oneOutputs = rawOutputs.get(outputKey)[i];
                //DataType dtype = this.outputSpecMap.get(outputKey).getDtype();
                outputMap.put( outputKey,  oneOutputs);
            }
            resultMap.put(sampleKeyList.get(i), outputMap);
        }
        return resultMap;
    }

    /**
     *
     * @param sampleWithFeatureMap
     * @param <T1> 样本key
     * @param <T2> 模型输出的类型
     * @return <样本ID，<预估值类型,预估值>>
     */
    public <T1, T2> Map<T1, Map<String, T2[]>> predict(Map<T1, Map<String, String>> sampleWithFeatureMap) {
        //第一步，把样本的key都取出来
        List<T1> sampleKeyList = new ArrayList<>(sampleWithFeatureMap.keySet());

        //第二步，把特征从一行一行的变成一列一列的，key是特征名字，value是按照sampleKeyList排序的特征取值tensor
        Map<String, Tensor<?>> formattedInput = preprocessInput(sampleKeyList, sampleWithFeatureMap);

        //第三步，把上述特征输入tensorflow中，得到按sampleKeyList顺序的预估结果，并关闭所有tensor, key是tf.function中输出的名字
        Map<String, T2[][]> rawOutputs = executeTfGraph(formattedInput);

        //第四步，把原始输出组装为PredResult对象，里面的key是tf.function中输出的名字
        //Map<T, Map<String, E2EPredResult>> predResults = formatOutputs(sampleKeyList, rawOutputs);
        Map<T1, Map<String, T2[]>> predResults = formatOutputs2(sampleKeyList, rawOutputs);

        return predResults;
    }

    public static double formatDouble(float d, int newScala) {
        BigDecimal bg = new BigDecimal(d).setScale(newScala, RoundingMode.UP);
        return bg.doubleValue();
    }

    /**
     *
     * @param input
     * @param inputInfo
     * @return
     */
    public static Tensor<?> listToTensor(List<String> input, TensorInfo inputInfo) {
        DataType dtype = inputInfo.getDtype();
        int inputSize = input.size();
        if (dtype == DataType.DT_FLOAT) {
            float[][] arrayInput = new float[inputSize][1];
            for (int i = 0; i < input.size(); i++) {
                String s = input.get(i);
                float inputWithType = 0.0f;
                if (s != null) {
                    try {
                        inputWithType = Float.parseFloat(s);
                        arrayInput[i][0] = inputWithType;
                    } catch (NumberFormatException e) {
                        logger.warn("listToTensor exception", e);
                    }
                }
            }
            Tensor<Float> tensor = Tensors.create(arrayInput);
            return tensor;
        } else if (dtype == DataType.DT_STRING) {
            byte[][][] arrayInput = new byte[inputSize][1][];
            for (int i = 0; i < input.size(); i++) {
                String s = input.get(i);
                if (s != null) {
                    arrayInput[i][0] = s.getBytes(UTF_8);
                } else {
                    arrayInput[i][0] = "null".getBytes(UTF_8);
                }
            }
            Tensor<String> tensor = Tensors.create(arrayInput);
            return tensor;
        } else {
            return null;
        }
    }

    /**
     *
     * @param tensorResult
     * @param <T1>
     * @param <T2>
     * @return
     */
    public <T1,T2> T2[][] tensorToArray(Tensor<?> tensorResult) {
        long[] tensorShape = tensorResult.shape();
        if (org.tensorflow.DataType.STRING == tensorResult.dataType()) {
            byte[][][] byteListResult = new byte[(int) tensorShape[0]][(int) tensorShape[1]][];
            tensorResult.copyTo(byteListResult);
            String[][] listResult = new String[(int) tensorShape[0]][(int) tensorShape[1]];
            for (int p = 0; p < byteListResult.length; ++p) {
                for (int q = 0; q < byteListResult[p].length; ++q) {
                    listResult[p][q] = new String(byteListResult[p][q], UTF_8);
                }
            }
            return (T2[][]) listResult;
        } else if (org.tensorflow.DataType.FLOAT == tensorResult.dataType()) {
            float[][] listResult = new float[(int) tensorShape[0]][(int) tensorShape[1]];
            tensorResult.copyTo(listResult);
            //由于tensor不支持copyTo一个Float的数组，只能这样了
            Float[][] listResult1 = new Float[(int) tensorShape[0]][(int) tensorShape[1]];
            for (int i = 0; i < listResult1.length; i++) {
                for (int j = 0; j < listResult1[i].length; j++) {
                    listResult1[i][j]=listResult[i][j];
                }
            }
            return (T2[][]) listResult1;
        }
        return null;
    }
}
