package cn.com.duiba.nezha.alg.feature.util;

import cn.com.duiba.nezha.alg.common.enums.DateStyle;
import cn.com.duiba.nezha.alg.common.util.AssertUtil;
import cn.com.duiba.nezha.alg.common.util.DataUtil;
import cn.com.duiba.nezha.alg.common.util.LocalDateUtil;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.JSON;
import lombok.Data;
import org.slf4j.LoggerFactory;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;


@Data
public class DataConverter{
    private static final org.slf4j.Logger logger = LoggerFactory.getLogger(DataConverter.class);

    /**
     * 对map按值累加
     * @param map map数据
     * @return 累计值
     */
    public static <T extends Number> Double accumulateMapValues(Map<?, T> map) {
        if (map == null) {
            return 0.;
        }
        return map.values().stream().mapToDouble(Number::doubleValue).sum();
    }


    /**
     * 将 两个等长序列 转 map
     * @param s1 第一个序列 一般使用 逗号 分割
     * @param s2 第二个序列 一般使用 逗号 分割
     * @param split 分割符
     * @return map
     */
    public static Map<String, String> series2Map(String s1, String s2, String split) {
        Map<String, String> ret = new HashMap<>();
        if (!AssertUtil.isAllNotEmpty(s1, s2, split)){
            return ret;
        }
        String[] s1Arr = s1.split(split);
        String[] s2Arr = s2.split(split);

        if (!(s1Arr.length > 0 && s1Arr.length == s2Arr.length)){
            return ret;
        }

        for (int i = 0; i < s1Arr.length; i++) {
            String item1 = s1Arr[i];
            String item2 = s2Arr[i];
            ret.put(item1, item2);
        }
        return ret;
    }


    /**
     * map 转 str
     * @param map
     */
    public static  String map2Str(Map<?, ?> map) {
        if (map == null) {return null;}
        String ret = "";
        try{
            ret = map.keySet().stream().map(key -> key + "_" + map.get(key)).collect(Collectors.joining(","));
        }catch (Exception  e){
            logger.error("DateConverter.json2Bucket2Str error", e);
        }
        return ret;
    }

    /**
     * json 转str
     * @param amap
     * @return
     */
    public static String json2Str(JSONObject amap) {
        if(amap==null || AssertUtil.isEmpty(amap)) {
            return null;
        }
        String ret = "";
        try{
            ret = amap.keySet().stream().map(key -> key + "_" + amap.get(key)).collect(Collectors.joining(","));
        }catch (Exception  e){
            logger.error("DateConverter.json2Bucket2Str error", e);
        }
        return ret;
    }

    /**
     * 对map value 分箱
     * @param map
     * @param <V>
     * @return
     */
    public static <V extends  Number> String map2Bucket2Str(Map<?, V> map, Number[] bucketList) {
        if (map == null) {
            return null;
        }
        String ret = "";
        try{
            ret = map.keySet().stream().map(key -> key + "_" + bucket(map.get(key), bucketList)).collect(Collectors.joining(","));
        }catch (Exception  e){
            logger.error("DateConverter.json2Bucket2Str error", e);
        }
        return ret;


    }

    public static <V extends  Number> String mapKey2Str(Map<?, V> map) {
        if (map == null) {
            return null;
        }
        String ret = "";

        try{
            ret = map.keySet().stream().map(Object::toString).collect(Collectors.joining(","));
        }catch (Exception  e){
            logger.error("DateConverter.mapKey2Str error", e);
        }
        return ret;


    }

    /**
     * 对json value分箱
     * @param ajson
     * @param bucketList
     * @return
     */
    public static  String json2Bucket2Str(JSONObject ajson, Number[] bucketList) {
        if(ajson==null || AssertUtil.isEmpty(ajson)) {
            return null;
        }
        String ret = "";
        try{
            ret = ajson.keySet().stream().map(key -> key + "_" + bucket(ajson.getDouble(key), bucketList)).collect(Collectors.joining(","));
        }catch (Exception  e){
            logger.error("DateConverter.json2Bucket2Str error", e);
        }
        return ret;
    }



    /**
     * 分箱
     * @param value
     * @param bucketList
     * @param <V1>
     * @param <V2>
     * @return
     */
    public static<V1 extends Number, V2 extends Number> String bucket(V1 value, V2[] bucketList) {
        if (value == null || bucketList == null || bucketList.length == 0) {
            return "";
        }
        int ret = bucketList.length;
        for (int i = 0; i < bucketList.length; i++) {
            Number bound = bucketList[i];
            if (value.doubleValue() <= bound.doubleValue()) {
                ret = i;
                break;
            }
        }
        return DataUtil.Integer2String(ret);
    }


    /**
     * set 转 str
     * @param set
     * @param <T>
     * @return
     */
    public static <T> String set2Str(Set<T> set) {
        if (AssertUtil.isEmpty(set)) {
            return null;
        }
        String ret = "";
        try{
            ret = set.stream().map(Object::toString).collect(Collectors.joining(","));
        }catch (Exception e) {
            logger.error("DateConverter.set2Str error", e);
        }
        return ret;
    }

    /**
     * 计算集合差集
     * @param base
     * @param exclude
     * @param <T>
     * @return
     */
    public static <T> Set<T> setDiff(Set<T> base, Set<T> exclude) {
        if (base == null) {
            return null;
        }
        if (exclude==null){
            return base;
        }

        Set<T> res = new HashSet<>(base);// removeAll 是原地操作
        try{
            res.removeAll(exclude);
        } catch (Exception e){
            logger.error("DateConverter.setDiff error", e);
        }

        return res;
    }

    /**
     *
     * @param JointStr 将格式类似 `a_1&b_2&c_3` 的拼接字符串转成map
     * @param s1 split 多个元素
     * @param s2 split key和value
     * @return
     */
    public static Map<String, String> transJointStrToMap(String JointStr, String s1, String s2) {
        Map<String, String> ret = new HashMap<>();
        if (JointStr == null) {
            return ret;
        }
        String[] valueArr = JointStr.split(s1);
        for (String s : valueArr) {
            String[] itemArr = s.split(s2);
            ret.put(itemArr[0], itemArr.length >= 2 ? itemArr[1] : "");
        }
        return ret;
    }

    /**
     * 计算时间差， 按单位返回
     * @param date1 时间1, 日期格式 2019-02-19 11:01:10
     * @param date2 时间2
     * @param unit 年/月/日/周/时/分/秒
     * @return diff
     */
    public static Long getTimeInterval(String date1, String date2, String unit) {
        if (date1 == null || date2==null || unit==null){
            return -1L;
        }
        LocalDateTime start = LocalDateUtil.getLocalDateTime(date1, DateStyle.YYYY_MM_DD_HH_MM_SS.getValue());
        LocalDateTime end = LocalDateUtil.getLocalDateTime(date2, DateStyle.YYYY_MM_DD_HH_MM_SS.getValue());
        Duration diff = Duration.between(start, end);
        switch (unit)
        {
            case "Y":
                return diff.toDays()/365;
            case "M":
                return diff.toDays()/30;
            case "D":
                return diff.toDays();
            case "W":
                return diff.toDays()/7;
            case "H":
                return diff.toHours();
            case "m":
                return diff.toMinutes();
            case "s":
                return diff.toMillis()/1000;
            default :
                return null;
        }


    }

    /**
     * 不同行业ctr/cvr中取出当前行业的ctr/cvr
     *
     * @param actMap
     * @param newTrade
     * @param buckets
     * @return
     */
    public static String getCurTradeLevel(JSONObject actMap, String newTrade, Number[] buckets) {
        if(newTrade == null || actMap == null  ){
            return null;
        }
        Double cateValue = actMap.getDouble(newTrade);
        if (cateValue == null) {
            return "-1";
        }
        return DataConverter.bucket(cateValue, buckets);
    }

    /**
     * 判断前一次与当前次
     * @param s1 前一次
     * @param s2 当前次
     * @return status
     */
    public static String getMatchStatus(String s1, String s2) {
        if (s1 == null) {
            return "-1";
        } else if (s2 == null) {
            return "-2";
        }else if (!s1.equals(s2)) {
            return "0";
        } else {
            return "1";
        }
    }

    public static String dmpGender2AqyGender(String dmpGender) {
        if (dmpGender == null || dmpGender.equals("")) {
            return null ;
        }
        if (dmpGender.equals("010101")) {
            return "119010101";
        }
        if (dmpGender.equals("010102")) {
            return "119010102";
        }
        return null;
    }




    public static void main(String[] args)  {

        Number[] cntBucket = {0, 1, 3, 5, 10, 15, 20};
        String jsonStr = "{\"totalRecords\": \"1293\", \"pageIndex\": \"1e\",   \"pageSize\": 200}";
        JSONObject jsonboj = JSONObject.parseObject(jsonStr);
        System.out.println(jsonboj.getDouble("totalRecords"));
        System.out.println(jsonboj.getDouble("pageIndex"));
    }

}

