package cn.com.duiba.nezha.alg.feature.parse;

import cn.com.duiba.nezha.alg.common.util.AssertUtil;
import cn.com.duiba.nezha.alg.common.util.DataUtil;
import cn.com.duiba.nezha.alg.common.util.LocalDateUtil;
import cn.com.duiba.nezha.alg.feature.vo.*;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.omg.PortableInterceptor.INACTIVE;
import org.slf4j.LoggerFactory;

import java.time.LocalDateTime;
import java.util.*;

import static cn.com.duiba.nezha.alg.feature.parse.FeatureParse.*;

/**
 * 素材特征解析
 */
public class MaterialFeatureParse {
    private static final org.slf4j.Logger logger = LoggerFactory.getLogger(MaterialFeatureParse.class);

    public static int F_MAX_SIZE = 64;

    //媒体+素材
    public static int[] appSckExpPvBucket = {3, 16, 128, 512, 2048, 8192, 20480, 40960, 81920, 163840}; // 媒体-素材曝光3、7、14天pv
    public static int[] appSckExpUvBucket = {2, 8, 32, 256, 1024, 4096, 16384, 32768, 65536, 98304}; // 媒体-素材曝光3、7、14天uv
    public static int[] appSckClkBucket = {0, 2, 8, 32, 128, 256, 512, 1024, 4096, 8192}; // 媒体-素材点击3、7、14天pv、uv

    //广告位+素材
    public static int[] sexpWeekBucket = {0, 5, 17, 61, 170, 286, 650, 2446, 14211, 277794};
    public static int[] sexpDayBucket = {0, 3, 10, 24, 40, 75, 161, 795, 27479, 120623};

    public static int[] sclickWeekBucket = {0, 1, 2, 4, 9, 19, 61, 164, 1165, 8893};
    public static int[] sclickDayBucket = {0, 1, 1, 2, 3, 9, 24, 140, 1231, 2740};

    public static int[] slotSckExpUvBucket = {2, 8, 32, 256, 512, 1024, 4096, 16384, 65536}; // 广告位-素材曝光3、7、14天uv
    public static int[] slotSckClkUvBucket = {0, 2, 8, 32, 128, 256, 512, 1024, 2048, 3072}; // 广告位-素材点击3、7、14天uv


    //素材：
    public static int[] expWeekBucket = {0, 3, 10, 46, 156, 369, 875, 4103, 36319, 431986};
    public static int[] expDayBucket = {0, 3, 9, 24, 43, 87, 201, 1626, 47910, 126840};
    public static int[] clickWeekBucket = {0, 1, 3, 7, 14, 44, 123, 379, 3391, 14571};
    public static int[] clickDayBucket = {0, 1, 2, 3, 4, 10, 35, 199, 1501, 4698};

    public static int[] convertCntBucket = {0, 1, 2, 5, 10, 20, 50, 100, 500, 1000, 2000};
    public static int[] bconvertCntBucket = {0, 1, 2, 5, 10, 20, 50, 100, 500, 1000, 2000};

    public static double[] userCtrBucket = {0.01, 0.02, 0.03, 0.04, 0.05, 0.06, 0.07, 0.08, 0.09, 0.10, 0.15, 0.20, 0.30, 0.40, 0.50, 1.0};
    public static double[] slotCtrBucket = {0.01, 0.02, 0.03, 0.04, 0.05, 0.06, 0.07, 0.08, 0.09, 0.10, 0.15, 0.20, 0.30, 0.40, 0.50, 1.0};

    public static int[] sckExpUvBucket = {3, 16, 128, 512, 2048, 8192, 20480, 40960, 81920, 163840}; // 素材曝光3、7、14天uv
    public static int[] sckClkUvBucket = {0, 4, 16, 64, 256, 1024, 4096, 8192, 16384, 32768}; // 素材点击3、7、14天uv

    // 用户-XX  素材标签/活动组件（当天/7天/历史）
    //  根据实际数据调整分桶阈值
    public static int[] userBehaviorBucket = {0, 1, 3, 5, 10, 15, 35, 50, 80};

    // 广告位/媒体 相关统计值
    //  根据实际数据调整分桶阈值
    public static int[] slotMaterialTagExposeDayBucket = {3, 6, 10, 20, 50, 100, 500, 2000, 10000, 750000};
    public static int[] slotMaterialTagExposeSevenBucket = {3, 9, 23, 57, 138, 358, 1086, 4313, 23877, 1547838};
    public static int[] slotMaterialTagExposeHisBucket = {3, 12, 37, 103, 282, 843, 2612, 10716, 65838, 4978025};
    public static int[] slotMaterialTagClickDayBucket = {1, 2, 3, 5, 10, 20, 60, 180, 900, 3600};
    public static int[] slotMaterialTagClickSevenBucket = {1, 3, 6, 11, 24, 58, 153, 539, 2977, 135801};
    public static int[] slotMaterialTagClickHisBucket = {3, 12, 37, 103, 282, 843, 2612, 10716, 65838, 4978025};

    public static int[] slotSkinTypeExposeDayBucket = {1, 2, 5, 20, 80, 300, 1000, 2000, 8000, 30000};
    public static int[] slotSkinTypeExposeSevenBucket = {1, 4, 15, 63, 219, 770, 2293, 7241, 32396, 1026962};
    public static int[] slotSkinTypeExposeHisBucket = {1, 2, 4, 18, 111, 547, 2695, 11694, 57077, 3085582};
    public static int[] slotSkinTypeClickDayBucket = {1, 2, 3, 9, 28, 88, 180, 400, 1500, 5000};
    public static int[] slotSkinTypeClickSevenBucket = {2, 8, 24, 80, 272, 717, 1911, 5349, 22938, 864145};
    public static int[] slotSkinTypeClickHisBucket = {1, 4, 14, 54, 217, 896, 3155, 10601, 44939, 2458473};

    public static int[] slotRewardTagExposeDayBucket = {1, 2, 4, 8, 15, 30, 75, 220, 1200, 6000};
    public static int[] slotRewardTagExposeSevenBucket = {2, 4, 7, 14, 27, 58, 146, 507, 3001, 153710};
    public static int[] slotRewardTagExposeHisBucket = {2, 4, 9, 19, 39, 87, 226, 799, 6388, 448837};
    public static int[] slotRewardTagClickDayBucket = {1, 2, 3, 5, 10, 23, 60, 150, 800, 4800};
    public static int[] slotRewardTagClickSevenBucket = {2, 3, 7, 12, 23, 55, 132, 387, 2239, 126967};
    public static int[] slotRewardTagClickHisBucket = {2, 4, 9, 17, 35, 79, 199, 637, 4440, 220232};


    public static int[] appMaterialTagExposeDayBucket = {2, 7, 18, 40, 110, 300, 800, 3000, 18000, 120000};
    public static int[] appMaterialTagExposeSevenBucket = {3, 12, 33, 87, 231, 622, 1841, 6897, 40033, 2893258};
    public static int[] appMaterialTagExposeHisBucket = {4, 15, 54, 170, 532, 1568, 4889, 18573, 108078, 8423830};
    public static int[] appMaterialTagClickDayBucket = {1, 2, 4, 9, 20, 45, 120, 380, 1800, 9000};
    public static int[] appMaterialTagClickSevenBucket = {1, 3, 7, 15, 33, 84, 237, 847, 4832, 244629};
    public static int[] appMaterialTagClickHisBucket = {2, 5, 12, 30, 71, 187, 556, 2013, 12229, 827318};

    public static int[] appSkinTypeExposeDayBucket = {1, 2, 9, 35, 120, 400, 1200, 5000, 22000, 65000};
    public static int[] appSkinTypeExposeSevenBucket = {1, 3, 17, 83, 335, 1263, 3725, 12053, 59810, 1666682};
    public static int[] appSkinTypeExposeHisBucket = {1, 3, 17, 156, 823, 2025, 4229, 17325, 102025, 3854952};
    public static int[] appSkinTypeClickDayBucket = {1, 3, 9, 30, 100, 350, 700, 2100, 8500, 20000};
    public static int[] appSkinTypeClickSevenBucket = {2, 10, 34, 128, 408, 1058, 3607, 9581, 39887, 1055972};
    public static int[] appSkinTypeClickHisBucket = {1, 4, 18, 85, 356, 1317, 4375, 18299, 83998, 3017636};

    public static int[] appRewardTagExposeDayBucket = {1, 2, 5, 8, 15, 33, 120, 450, 2500, 100000};
    public static int[] appRewardTagExposeSevenBucket = {2, 4, 9, 17, 35, 82, 223, 837, 5324, 289787};
    public static int[] appRewardTagExposeHisBucket = {2, 4, 9, 23, 49, 119, 340, 1274, 11159, 682999};
    public static int[] appRewardTagClickDayBucket = {1, 2, 3, 5, 10, 23, 60, 150, 1000, 48000};
    public static int[] appRewardTagClickSevenBucket = {2, 4, 8, 15, 29, 70, 187, 599, 3560, 177802};
    public static int[] appRewardTagClickHisBucket = {2, 5, 10, 20, 42, 100, 303, 1025, 7421, 399933};

    public static int[] bigCntBucket = {0, 500, 1000, 2000, 5000, 10000, 20000, 50000, 100000, 500000, 1000000};


    //时间间隔
    public static int[] timeBucket = {0, 1, 3, 5, 10, 15, 20, 50};

    // 素材-广告行业类目 编码
    public static Map<String, String> AdvertIndustryCodes = new HashMap<String, String>(){
        {
            put("28548","02.15.0007");
            put("28549","02.15.0007");
            put("28550","02.15.0007");
            put("28551","02.15.0007");
            put("28552","02.15.0007");
            put("28553","02.15.0007");
            put("28562","02.10.0005");
            put("28563","02.10.0005");
            put("28564","02.10.0005");
            put("28565","02.10.0005");
            put("28566","02.10.0005");
            put("28567","02.10.0005");
            put("28966","02.24.0003");
            put("28967","02.24.0003");
            put("28968","02.24.0003");
            put("28969","02.24.0003");
            put("28970","02.24.0003");
            put("28971","02.24.0003");
            put("28574","02.22.0001");
            put("28575","02.22.0001");
            put("28576","02.22.0001");
            put("28577","02.22.0001");
            put("28578","02.22.0001");
            put("28579","02.22.0001");
            put("28985","02.05.0002");
            put("28986","02.05.0002");
            put("28987","02.05.0002");
            put("28989","02.05.0002");
            put("28990","02.05.0002");
            put("28992","02.05.0002");
            put("28614","02.06.0004");
            put("28615","02.06.0004");
            put("28616","02.06.0004");
            put("28617","02.06.0004");
            put("28618","02.06.0004");
            put("28619","02.06.0004");
            put("28622","02.06.0003");
            put("28623","02.06.0003");
            put("28624","02.06.0003");
            put("28625","02.06.0003");
            put("28626","02.06.0003");
            put("28627","02.06.0003");
            put("28654","02.03.0007");
            put("28655","02.03.0007");
            put("28656","02.03.0007");
            put("28657","02.03.0007");
            put("28658","02.03.0007");
            put("28659","02.03.0007");
            put("28661","02.19.0001");
            put("28662","02.19.0001");
            put("28663","02.19.0001");
            put("28664","02.19.0001");
            put("28665","02.19.0001");
            put("28666","02.19.0001");
            put("28874","02.23.0001");
            put("28875","02.23.0001");
            put("28876","02.23.0001");
            put("28877","02.23.0001");
            put("28878","02.23.0001");
            put("28879","02.23.0001");
            put("28880","02.01.0002");
            put("28881","02.01.0002");
            put("28882","02.01.0002");
            put("28883","02.01.0002");
            put("28884","02.01.0002");
            put("28885","02.01.0002");
            put("28892","02.01.0007");
            put("28893","02.01.0007");
            put("28894","02.01.0007");
            put("28895","02.01.0007");
            put("28914","02.01.0007");
            put("28982","02.01.0007");
            put("28896","02.16.0002");
            put("28897","02.16.0002");
            put("28898","02.16.0002");
            put("28899","02.16.0002");
            put("28900","02.16.0002");
            put("28901","02.16.0002");
            put("28902","02.24.0001");
            put("28903","02.24.0001");
            put("28904","02.24.0001");
            put("28905","02.24.0001");
            put("28906","02.24.0001");
            put("28907","02.24.0001");
            put("28960","02.05.0001");
            put("28961","02.05.0001");
            put("28962","02.05.0001");
            put("28963","02.05.0001");
            put("28964","02.05.0001");
            put("28965","02.05.0001");
            put("28921","02.10.0002");
            put("28922","02.10.0002");
            put("28923","02.10.0002");
            put("28924","02.10.0002");
            put("28925","02.10.0002");
            put("28950","02.10.0002");
            put("28983","02.09.0001");
            put("28984","02.09.0001");
            put("28988","02.09.0001");
            put("28991","02.09.0001");
            put("28993","02.09.0001");
            put("28994","02.09.0001");
            put("28932","02.17.0003");
            put("28933","02.17.0003");
            put("28934","02.17.0003");
            put("28935","02.17.0003");
            put("28936","02.17.0003");
            put("28937","02.17.0003");
            put("28938","02.04.0001");
            put("28939","02.04.0001");
            put("28940","02.04.0001");
            put("28941","02.04.0001");
            put("28942","02.04.0001");
            put("28943","02.04.0001");
            put("28954","02.21.0009");
            put("28955","02.21.0009");
            put("28956","02.21.0009");
            put("28957","02.21.0009");
            put("28958","02.21.0009");
            put("28959","02.21.0009");
        }
    };

    // 活动奖品标签编码
    public static Map<String, Integer> actRewardCodes = new HashMap<String, Integer>() {
        {
            put("29", 0);
            put("30", 1);
            put("31", 2);
            put("32", 3);
            put("33", 4);
            put("34", 5);
            put("35", 6);
            put("36", 7);
            put("37", 8);
            put("38", 9);
            put("39", 10);
            put("40", 11);
            put("41", 12);
            put("42", 13);
            put("43", 14);
        }
    };

    // 活动皮肤id编码
    public static Map<String, Integer> actSkinTypeCodes = new HashMap<String, Integer>() {
        {
            put("1", 0);
            put("2", 1);
            put("3", 2);
            put("4", 3);
            put("5", 4);
            put("6", 5);
            put("7", 6);
            put("8", 7);
            put("9", 8);
            put("10", 9);
            put("11", 10);
            put("12", 11);
            put("13", 12);
            put("14", 13);
            put("15", 14);
            put("16", 15);
            put("17", 16);
            put("18", 17);
            put("19", 18);
            put("20", 19);
            put("21", 20);
            put("22", 21);
            put("23", 22);
            put("24", 23);
            put("25", 24);
            put("26", 25);
            put("27", 26);
            put("28", 27);
            put("29", 28);
            put("30", 29);
            put("31", 30);
            put("32", 31);
            put("33", 32);
        }
    };

    // 素材标签编码
    public static Map<String, Integer> materialTagCodes = new HashMap<String, Integer>() {
        {
            put("1407", 0);
            put("1408", 1);
            put("1409", 2);
            put("1410", 3);
            put("1411", 4);
            put("1413", 5);
            put("1414", 6);
            put("1415", 7);
            put("1416", 8);
            put("1417", 9);
            put("1419", 10);
            put("1420", 11);
            put("1421", 12);
            put("1422", 13);
            put("1423", 14);
            put("1424", 15);
            put("1425", 16);
            put("1426", 17);
            put("1427", 18);
            put("1429", 19);
            put("1432", 20);
            put("1442", 21);
            put("1443", 22);
            put("1445", 23);
            put("1446", 24);
            put("1498", 25);
            put("1499", 26);
            put("1500", 27);
            put("1501", 28);
            put("1502", 29);
            put("1506", 30);
            put("1507", 31);
            put("1508", 32);
            put("1509", 33);
            put("1535", 34);
            put("1537", 35);
            put("1538", 36);
            put("1539", 37);
            put("1540", 38);
            put("1541", 39);
            put("1542", 40);
            put("1543", 41);
            put("1547", 42);
            put("1549", 43);
            put("1550", 44);
            put("1551", 45);
            put("1555", 46);
            put("1556", 47);
            put("1564", 48);
            put("1567", 49);
            put("1568", 50);
            put("1868", 51);
            put("1869", 52);
            put("1870", 53);
            put("1871", 54);
            put("1872", 55);
            put("1873", 56);
            put("1874", 57);
            put("1875", 58);
            put("1876", 59);
            put("1877", 60);
            put("1878", 61);
            put("1879", 62);
            put("1880", 63);
            put("1881", 64);
            put("1882", 65);
            put("1883", 66);
            put("1884", 67);
            put("1894", 68);
            put("1895", 69);
            put("1896", 70);
            put("1897", 71);
            put("1898", 72);
            put("1899", 73);
            put("1900", 74);
            put("1901", 75);
            put("1903", 76);
            put("1904", 77);
            put("1906", 78);
            put("1907", 79);
            put("1908", 80);
            put("1909", 81);
            put("1910", 82);
            put("1911", 83);
            put("1912", 84);
            put("1913", 85);
            put("1914", 86);
            put("1915", 87);
            put("1975", 88);
            put("1985", 89);
            put("1986", 90);
            put("1987", 91);
            put("1990", 92);
            put("1991", 93);
            put("1995", 94);
            put("1996", 95);
            put("1999", 96);
            put("2009", 97);
            put("2023", 98);
            put("2024", 99);
            put("2025", 100);
            put("2029", 101);
            put("2030", 102);
            put("2033", 103);
            put("2036", 104);
            put("2042", 105);
            put("2049", 106);
        }
    };


    public static Map<String, String> generateFeatureMapStatic(MaterialFeatureDo cf) {
        Map<String, String> retMap = new HashMap<>(F_MAX_SIZE);
        try {
            if (cf != null) {

                /**
                 * 用户特征
                 */
                // 用户画像特征解析
                UserFeature userFeature = Optional.ofNullable(cf.getUserFeature()).orElse(new UserFeature());

                // 基本属性
                retMap.put("f451001", userFeature.getSex());
                retMap.put("f451002", userFeature.getAge());
                retMap.put("f451003", userFeature.getMarry());
                retMap.put("f451004", userFeature.getConsumeLevel());
                retMap.put("f451005", userFeature.getPermanentProvince());
                retMap.put("f451006", userFeature.getPermanentCity());

                // 用户画像特征解析
                retMap.put("f9902", userFeature.getAppList());
                retMap.put("f9801", userFeature.getSigList());

                //截取30个
                if (userFeature.getExpDayMaterials() != null) {
                    List<String> expDayMaterials = Arrays.asList(userFeature.getExpDayMaterials().split(","));
                    retMap.put("f390001", subList(expDayMaterials));
                    // new
                    retMap.put("f391001", FeatureParse.bucket(expDayMaterials.size(), userBehaviorBucket).toString());
                } else {
                    retMap.put("f390001", "null");
                    // new
                    retMap.put("f391001", "null");
                }

                if (userFeature.getClickDayMaterials() != null) {
                    List<String> clickDayMaterials = Arrays.asList(userFeature.getClickDayMaterials().split(","));
                    retMap.put("f390002", subList(clickDayMaterials));
                    // new
                    retMap.put("f391002", FeatureParse.bucket(clickDayMaterials.size(), userBehaviorBucket).toString());
                } else {
                    retMap.put("f390002", "null");
                    // new
                    retMap.put("f391002", "null");
                }

                retMap.put("f390003", sortSubList(userFeature.getExpWeekMaterials()));
                retMap.put("f390004", sortSubList(userFeature.getClickWeekMaterials()));

                //广告位
                SlotFeature slotFeature = Optional.ofNullable(cf.getSlotFeature()).orElse(new SlotFeature());

                retMap.put("f108001", DataUtil.Long2String(slotFeature.getSlotId()));
                retMap.put("f109001", DataUtil.Long2String(slotFeature.getSlotType()));


                AppFeature appFeature = Optional.ofNullable(cf.getAppFeature()).orElse(new AppFeature());
                retMap.put("f201001", DataUtil.Long2String(appFeature.getAppId()));
                // 媒体标签
                retMap.put("f205001", appFeature.getAppIndustryTagPid());
                retMap.put("f205002", appFeature.getAppIndustryTagId());

                //上下文
                MaterialContextFeature contextFeature = Optional.ofNullable(cf.getMaterialContextFeature()).orElse(new MaterialContextFeature());


                retMap.put("f501001", contextFeature.getUa());
                retMap.put("f503001", DataUtil.Long2String(contextFeature.getCityId()));

                retMap.put("f503003", contextFeature.getProvinceCode());
                // 设备型号
                retMap.put("f504001", contextFeature.getModel());
                // 设备价格区间
//                retMap.put("f505001", cf.getPriceSection());
                retMap.put("f390017", contextFeature.getExposeHour());
                retMap.put("f390018", contextFeature.getExposeWeekDay());
                retMap.put("f390019", contextFeature.getExposeHolidays());

                // 券利益点素材 http://cf.dui88.com/pages/viewpage.action?pageId=76658070
                // 券利益-新增-上下文
                putMapNotNull(retMap, "f508006", contextFeature.getDeviceTrademark());
                putMapNotNull(retMap, "f507001", contextFeature.getOperatorType());
                putMapNotNull(retMap, "f507003", contextFeature.getOsVersion());
                putMapNotNull(retMap, "f506001", contextFeature.getConnectionType());

                // 券利益-新增-广告位
                // ??F109002
                JSONObject weekCvrTradeMap = DataUtil.stringToJson(slotFeature.getWeekCvrTrade());
                putMapNotNull(retMap, "f214003", getUserCategoryActLevelByBucket(weekCvrTradeMap, tradeCodes, statCvrBucket));
                JSONObject weekCvrCateMap = DataUtil.stringToJson(slotFeature.getWeekCvrCate());
                putMapNotNull(retMap, "f222004", getUserCategoryActLevelByBucket(weekCvrCateMap, resourceCodes, statCvrBucket));

                // 券利益-新增-用户
                Map<String, Integer> appCatePrefMap = DataUtil.stringToMap(userFeature.getAppCatePref(), 0);
                putMapNotNull(retMap, "f461001", getUserCategoryActLevelByBucket(appCatePrefMap, appCodes, cntBucket));
                Map<String, Integer> textCatePrefMap = DataUtil.stringToMap(userFeature.getTextCatePref(), 0);
                putMapNotNull(retMap, "f461003", getUserCategoryActLevelByBucket(textCatePrefMap, msgSigCodes, cntBucket));
                putMapNotNull(retMap, "f9701", userFeature.getUserBasicTags());
                putMapNotNull(retMap, "f9802", FeatureBaseParse.getLengthLevel(userFeature.getSigList(), 15, 1));
                putMapNotNull(retMap, "f9803", userFeature.getMsgCategory());
                putMapNotNull(retMap, "f9903", FeatureBaseParse.getLengthLevel(userFeature.getAppList(), 5, 100, 5));
                // note ： 9935与9902重复，不能同时使用
                putMapNotNull(retMap, "f9935", userFeature.getAppList());
                String appCategoryLevel = FeatureBaseParse.getCategoryLevel(userFeature.getAppCategory(), 10, 1);
                putMapNotNull(retMap, "f9940", appCategoryLevel);
                putMapNotNull(retMap, "f9944", userFeature.getTopAppList3());

                // 券利益-新增-统计类特征
                putMapNotNull(retMap,"f601002", DataUtil.Long2String(bucket(userFeature.getDayOrderRank(), dayOrderRankBucket)));
                putMapNotNull(retMap,"f602002", DataUtil.Long2String(bucket(userFeature.getOrderRank(), orderRankBucket)));
                Long lastChargeStatus = getChargeStatus(userFeature.getLastChargeNums());
                putMapNotNull(retMap,"f608001", DataUtil.Long2String(lastChargeStatus));
                putMapNotNull(retMap,"f413007", userFeature.getLastLaunchTrade());
                putMapNotNull(retMap,"f413008", DataUtil.Integer2String(userFeature.getClickLaunchTrade()));
                putMapNotNull(retMap, "f414008", DataUtil.Integer2String(userFeature.getEffectClickTrade()));
                putMapNotNull(retMap, "f434001", bucket(userFeature.getVouchFailed(), cntBucket).toString());
                putMapNotNull(retMap, "f434004", bucket(userFeature.getReject(), cntBucket).toString());

                Map<String, Map<String, Long>> ubpMap = getUserBehavioralPreference(
                        userFeature.getUIIds(), userFeature.getUILaunchPV(), userFeature.getUIClickPv(), userFeature.getUIEffectPv());
                putMapNotNull(retMap, "f808001", getUserBehavioralPreference(2, "ctr", ubpMap, null));
                putMapNotNull(retMap, "f809001", getUserBehavioralPreference(0, "ctr", ubpMap, null));
                putMapNotNull(retMap, "f808002", getUserBehavioralPreference(2, "cvr", ubpMap, null));
                putMapNotNull(retMap, "f809002", getUserBehavioralPreference(0, "cvr", ubpMap, null));
                putMapNotNull(retMap, "f830001", userFeature.getClickAdIds());
                putMapNotNull(retMap, "f830004", userFeature.getClickTrIds());
                putMapNotNull(retMap, "f831001", userFeature.getConvertAdverts());
                putMapNotNull(retMap, "f831004", userFeature.getConvertTrades());
                putMapNotNull(retMap, "f832001", userFeature.getBConvertAdverts());
                putMapNotNull(retMap, "f832004", userFeature.getBConvertTrades());


                // 新增后向行为特征
                // 活动访问
                retMap.put("f411001", FeatureParse.bucket(userFeature.getActivityRequest(), cntBucket).toString());
                retMap.put("f411002", FeatureParse.bucket(userFeature.getWeekActivityRequest(), cntBucket).toString());
                retMap.put("f411004", FeatureParse.bucket(userFeature.getIntervalActivityRequest(), cntBucket).toString());
                retMap.put("f411005", userFeature.getPeriodActivityRequest());

                // 活动参与412005
                retMap.put("f412001", FeatureParse.bucket(userFeature.getActivityJoin(), cntBucket).toString());
                retMap.put("f412002", FeatureParse.bucket(userFeature.getWeekActivityJoin(), cntBucket).toString());
                retMap.put("f412004", FeatureParse.bucket(userFeature.getIntervalActivityJoin(), cntBucket).toString());
                retMap.put("f412005", userFeature.getPeriodActivityJoin());

                // 点击
                Map<String, Integer> clickTraceActMap = DataUtil.stringToMap(userFeature.getClickTrade(), 0);
                retMap.put("f414001", getUserCategoryActLevelByBucket(clickTraceActMap, tradeCodes, cntBucket));
                Map<String, Integer> weekClickTradeActMap = DataUtil.stringToMap(userFeature.getWeekClickTrade(), 0);
                retMap.put("f414002", getUserCategoryActLevelByBucket(weekClickTradeActMap, tradeCodes, cntBucket));
                retMap.put("f414004", FeatureParse.bucket(userFeature.getIntervalClickTrade(), cntBucket).toString());
                retMap.put("f414005", userFeature.getPeriodClickTrade());
                retMap.put("f414007", userFeature.getLastClickTrade());

                // 转化
                Map<String, Integer> effectTradeActMap = DataUtil.stringToMap(userFeature.getEffectTrade(), 0);
                retMap.put("f415001", getUserCategoryActLevelByBucket(effectTradeActMap, tradeCodes, cntBucket));
                Map<String, Integer> weekEffectTradeActMap = DataUtil.stringToMap(userFeature.getWeekEffectTrade(), 0);
                retMap.put("f415002", getUserCategoryActLevelByBucket(weekEffectTradeActMap, tradeCodes, cntBucket));
                retMap.put("f415004", FeatureParse.bucket(userFeature.getIntervalEffectTrade(), cntBucket).toString());
                retMap.put("f415005", userFeature.getPeriodEffectTrade());
                retMap.put("f415007", userFeature.getLastEffectTrade());

                // 二期特征
                // 用户活跃度
                retMap.put("f391003", FeatureParse.bucket(userFeature.getExpWeekCnt(), userBehaviorBucket).toString());
                retMap.put("f391004", FeatureParse.bucket(userFeature.getClickWeekCnt(), userBehaviorBucket).toString());
                retMap.put("f391005", FeatureParse.bucket(userFeature.getExpHisCnt(), userBehaviorBucket).toString());
                retMap.put("f391006", FeatureParse.bucket(userFeature.getClickHisCnt(), userBehaviorBucket).toString());

                // 用户-素材标签 (多值特征) (会添加较多的模型训练参数，素材标签大小：107)
//                Map<String, ComponentInfo> userToMatTagMap = userFeature.getUserToMatTagMap();
//                if (!AssertUtil.isEmpty(userToMatTagMap)) {
//                    Map<Integer, Map<String, Integer>> splitTimeMap = getSplitTimeMap(userToMatTagMap);
//                    retMap.put("f391007", getUserCategoryActLevelByBucket(splitTimeMap.get(0), materialTagCodes, userBehaviorBucket));
//                    retMap.put("f391008", getUserCategoryActLevelByBucket(splitTimeMap.get(1), materialTagCodes, userBehaviorBucket));
//                    retMap.put("f391009", getUserCategoryActLevelByBucket(splitTimeMap.get(2), materialTagCodes, userBehaviorBucket));
//                    retMap.put("f391010", getUserCategoryActLevelByBucket(splitTimeMap.get(3), materialTagCodes, userBehaviorBucket));
//                    retMap.put("f391011", getUserCategoryActLevelByBucket(splitTimeMap.get(4), materialTagCodes, userBehaviorBucket));
//                    retMap.put("f391012", getUserCategoryActLevelByBucket(splitTimeMap.get(5), materialTagCodes, userBehaviorBucket));
//                }

                // 用户-活动组件-皮肤类型
                Map<String, ComponentInfo> userToActSkinMap = userFeature.getUserToActSkinMap();
                if (!AssertUtil.isEmpty(userToActSkinMap)) {
                    Map<Integer, Map<String, Integer>> splitTimeMap = getSplitTimeMap(userToActSkinMap);
                    retMap.put("f391013", getUserCategoryActLevelByBucket(splitTimeMap.get(0), actSkinTypeCodes, userBehaviorBucket));
                    retMap.put("f391014", getUserCategoryActLevelByBucket(splitTimeMap.get(1), actSkinTypeCodes, userBehaviorBucket));
                    retMap.put("f391015", getUserCategoryActLevelByBucket(splitTimeMap.get(2), actSkinTypeCodes, userBehaviorBucket));
                    retMap.put("f391016", getUserCategoryActLevelByBucket(splitTimeMap.get(3), actSkinTypeCodes, userBehaviorBucket));
                    retMap.put("f391017", getUserCategoryActLevelByBucket(splitTimeMap.get(4), actSkinTypeCodes, userBehaviorBucket));
                    retMap.put("f391018", getUserCategoryActLevelByBucket(splitTimeMap.get(5), actSkinTypeCodes, userBehaviorBucket));
                }

                // 用户-活动组件-奖品标签
                Map<String, ComponentInfo> userToRewardTagMap = userFeature.getUserToRewardTagMap();
                if (!AssertUtil.isEmpty(userToRewardTagMap)) {
                    Map<Integer, Map<String, Integer>> splitTimeMap = getSplitTimeMap(userToRewardTagMap);
                    retMap.put("f391019", getUserCategoryActLevelByBucket(splitTimeMap.get(0), actRewardCodes, userBehaviorBucket));
                    retMap.put("f391020", getUserCategoryActLevelByBucket(splitTimeMap.get(1), actRewardCodes, userBehaviorBucket));
                    retMap.put("f391021", getUserCategoryActLevelByBucket(splitTimeMap.get(2), actRewardCodes, userBehaviorBucket));
                    retMap.put("f391022", getUserCategoryActLevelByBucket(splitTimeMap.get(3), actRewardCodes, userBehaviorBucket));
                    retMap.put("f391023", getUserCategoryActLevelByBucket(splitTimeMap.get(4), actRewardCodes, userBehaviorBucket));
                    retMap.put("f391024", getUserCategoryActLevelByBucket(splitTimeMap.get(5), actRewardCodes, userBehaviorBucket));
                }


                // 广告位二期特征
                // 素材标签
//                Map<String, ComponentInfo> slotToMatTagMap = slotFeature.getSlotToMatTagMap();
//                if (!AssertUtil.isEmpty(slotToMatTagMap)) {
//                    Map<Integer, Map<String, Integer>> splitTimeMap = getSplitTimeMap(slotToMatTagMap);
//                    retMap.put("f391025", getUserCategoryActLevelByBucket(splitTimeMap.get(0), materialTagCodes, slotMaterialTagExposeDayBucket));
//                    retMap.put("f391026", getUserCategoryActLevelByBucket(splitTimeMap.get(1), materialTagCodes, slotMaterialTagExposeSevenBucket));
//                    retMap.put("f391027", getUserCategoryActLevelByBucket(splitTimeMap.get(2), materialTagCodes, slotMaterialTagExposeHisBucket));
//                    retMap.put("f391028", getUserCategoryActLevelByBucket(splitTimeMap.get(3), materialTagCodes, slotMaterialTagClickDayBucket));
//                    retMap.put("f391029", getUserCategoryActLevelByBucket(splitTimeMap.get(4), materialTagCodes, slotMaterialTagClickSevenBucket));
//                    retMap.put("f391030", getUserCategoryActLevelByBucket(splitTimeMap.get(5), materialTagCodes, slotMaterialTagClickHisBucket));
//                }

                // 活动皮肤
                Map<String, ComponentInfo> slotToActSkinMap = slotFeature.getSlotToActSkinMap();
                if (!AssertUtil.isEmpty(slotToActSkinMap)) {
                    Map<Integer, Map<String, Integer>> splitTimeMap = getSplitTimeMap(slotToActSkinMap);
                    retMap.put("f391031", getUserCategoryActLevelByBucket(splitTimeMap.get(0), actSkinTypeCodes, slotSkinTypeExposeDayBucket));
                    retMap.put("f391032", getUserCategoryActLevelByBucket(splitTimeMap.get(1), actSkinTypeCodes, slotSkinTypeExposeSevenBucket));
                    retMap.put("f391033", getUserCategoryActLevelByBucket(splitTimeMap.get(2), actSkinTypeCodes, slotSkinTypeExposeHisBucket));
                    retMap.put("f391034", getUserCategoryActLevelByBucket(splitTimeMap.get(3), actSkinTypeCodes, slotSkinTypeClickDayBucket));
                    retMap.put("f391035", getUserCategoryActLevelByBucket(splitTimeMap.get(4), actSkinTypeCodes, slotSkinTypeClickSevenBucket));
                    retMap.put("f391036", getUserCategoryActLevelByBucket(splitTimeMap.get(5), actSkinTypeCodes, slotSkinTypeClickHisBucket));
                }

                // 活动奖品标签
                Map<String, ComponentInfo> slotToRewardTagMap = slotFeature.getSlotToRewardTagMap();
                if (!AssertUtil.isEmpty(slotToRewardTagMap)) {
                    Map<Integer, Map<String, Integer>> splitTimeMap = getSplitTimeMap(slotToRewardTagMap);
                    retMap.put("f391037", getUserCategoryActLevelByBucket(splitTimeMap.get(0), actRewardCodes, slotRewardTagExposeDayBucket));
                    retMap.put("f391038", getUserCategoryActLevelByBucket(splitTimeMap.get(1), actRewardCodes, slotRewardTagExposeSevenBucket));
                    retMap.put("f391039", getUserCategoryActLevelByBucket(splitTimeMap.get(2), actRewardCodes, slotRewardTagExposeHisBucket));
                    retMap.put("f391040", getUserCategoryActLevelByBucket(splitTimeMap.get(3), actRewardCodes, slotRewardTagClickDayBucket));
                    retMap.put("f391041", getUserCategoryActLevelByBucket(splitTimeMap.get(4), actRewardCodes, slotRewardTagClickSevenBucket));
                    retMap.put("f391042", getUserCategoryActLevelByBucket(splitTimeMap.get(5), actRewardCodes, slotRewardTagClickHisBucket));
                }

                // 用户-广告位-素材标签
                Map<String, ComponentInfo> userToSlotMatTagMap = slotFeature.getUserToSlotMatTagMap();
                if (!AssertUtil.isEmpty(userToSlotMatTagMap)) {
                    Map<Integer, Map<String, Integer>> splitTimeMap = getSplitTimeMap(userToSlotMatTagMap);
//                    retMap.put("f391061", getUserCategoryActLevelByBucket(splitTimeMap.get(1), materialTagCodes, userBehaviorBucket));
//                    retMap.put("f391062", getUserCategoryActLevelByBucket(splitTimeMap.get(2), materialTagCodes, userBehaviorBucket));
//                    retMap.put("f391063", getUserCategoryActLevelByBucket(splitTimeMap.get(4), materialTagCodes, userBehaviorBucket));
                    retMap.put("f391064", getUserCategoryActLevelByBucket(splitTimeMap.get(5), materialTagCodes, userBehaviorBucket));
                }
                // 用户-广告位-活动组件
                // 活动皮肤
                Map<String, ComponentInfo> userToSlotActSkinMap = slotFeature.getUserToSlotActSkinMap();
                if (!AssertUtil.isEmpty(userToSlotActSkinMap)) {
                    Map<Integer, Map<String, Integer>> splitTimeMap = getSplitTimeMap(userToSlotActSkinMap);
                    retMap.put("f391065", getUserCategoryActLevelByBucket(splitTimeMap.get(1), actSkinTypeCodes, userBehaviorBucket));
                    retMap.put("f391066", getUserCategoryActLevelByBucket(splitTimeMap.get(2), actSkinTypeCodes, userBehaviorBucket));
                    retMap.put("f391067", getUserCategoryActLevelByBucket(splitTimeMap.get(4), actSkinTypeCodes, userBehaviorBucket));
                    retMap.put("f391068", getUserCategoryActLevelByBucket(splitTimeMap.get(5), actSkinTypeCodes, userBehaviorBucket));
                }
                // 活动奖品标签
                Map<String, ComponentInfo> userToSlotRewardTagMap = slotFeature.getUserToSlotRewardTagMap();
                if (!AssertUtil.isEmpty(userToSlotRewardTagMap)) {
                    Map<Integer, Map<String, Integer>> splitTimeMap = getSplitTimeMap(userToSlotRewardTagMap);
                    retMap.put("f391069", getUserCategoryActLevelByBucket(splitTimeMap.get(1), actRewardCodes, userBehaviorBucket));
                    retMap.put("f391070", getUserCategoryActLevelByBucket(splitTimeMap.get(2), actRewardCodes, userBehaviorBucket));
                    retMap.put("f391071", getUserCategoryActLevelByBucket(splitTimeMap.get(4), actRewardCodes, userBehaviorBucket));
                    retMap.put("f391072", getUserCategoryActLevelByBucket(splitTimeMap.get(5), actRewardCodes, userBehaviorBucket));
                }

                // 媒体二期特征
                // 素材标签
//                Map<String, ComponentInfo> appToMatTagMap = appFeature.getAppToMatTagMap();
//                if (!AssertUtil.isEmpty(appToMatTagMap)) {
//                    Map<Integer, Map<String, Integer>> splitTimeMap = getSplitTimeMap(appToMatTagMap);
//                    retMap.put("f391043", getUserCategoryActLevelByBucket(splitTimeMap.get(0), materialTagCodes, appMaterialTagExposeDayBucket));
//                    retMap.put("f391044", getUserCategoryActLevelByBucket(splitTimeMap.get(1), materialTagCodes, appMaterialTagExposeSevenBucket));
//                    retMap.put("f391045", getUserCategoryActLevelByBucket(splitTimeMap.get(2), materialTagCodes, appMaterialTagExposeHisBucket));
//                    retMap.put("f391046", getUserCategoryActLevelByBucket(splitTimeMap.get(3), materialTagCodes, appMaterialTagClickDayBucket));
//                    retMap.put("f391047", getUserCategoryActLevelByBucket(splitTimeMap.get(4), materialTagCodes, appMaterialTagClickSevenBucket));
//                    retMap.put("f391048", getUserCategoryActLevelByBucket(splitTimeMap.get(5), materialTagCodes, appMaterialTagClickHisBucket));
//                }

                // 活动皮肤
                Map<String, ComponentInfo> appToActSkinMap = appFeature.getAppToActSkinMap();
                if (!AssertUtil.isEmpty(appToActSkinMap)) {
                    Map<Integer, Map<String, Integer>> splitTimeMap = getSplitTimeMap(appToActSkinMap);
                    retMap.put("f391049", getUserCategoryActLevelByBucket(splitTimeMap.get(0), actSkinTypeCodes, appSkinTypeExposeDayBucket));
                    retMap.put("f391050", getUserCategoryActLevelByBucket(splitTimeMap.get(1), actSkinTypeCodes, appSkinTypeExposeSevenBucket));
                    retMap.put("f391051", getUserCategoryActLevelByBucket(splitTimeMap.get(2), actSkinTypeCodes, appSkinTypeExposeHisBucket));
                    retMap.put("f391052", getUserCategoryActLevelByBucket(splitTimeMap.get(3), actSkinTypeCodes, appSkinTypeClickDayBucket));
                    retMap.put("f391053", getUserCategoryActLevelByBucket(splitTimeMap.get(4), actSkinTypeCodes, appSkinTypeClickSevenBucket));
                    retMap.put("f391054", getUserCategoryActLevelByBucket(splitTimeMap.get(5), actSkinTypeCodes, appSkinTypeClickHisBucket));
                }

                // 活动奖品标签
                Map<String, ComponentInfo> appToRewardTagMap = appFeature.getAppToRewardTagMap();
                if (!AssertUtil.isEmpty(appToRewardTagMap)) {
                    Map<Integer, Map<String, Integer>> splitTimeMap = getSplitTimeMap(appToRewardTagMap);
                    retMap.put("f391055", getUserCategoryActLevelByBucket(splitTimeMap.get(0), actRewardCodes, appRewardTagExposeDayBucket));
                    retMap.put("f391056", getUserCategoryActLevelByBucket(splitTimeMap.get(1), actRewardCodes, appRewardTagExposeSevenBucket));
                    retMap.put("f391057", getUserCategoryActLevelByBucket(splitTimeMap.get(2), actRewardCodes, appRewardTagExposeHisBucket));
                    retMap.put("f391058", getUserCategoryActLevelByBucket(splitTimeMap.get(3), actRewardCodes, appRewardTagClickDayBucket));
                    retMap.put("f391059", getUserCategoryActLevelByBucket(splitTimeMap.get(4), actRewardCodes, appRewardTagClickSevenBucket));
                    retMap.put("f391060", getUserCategoryActLevelByBucket(splitTimeMap.get(5), actRewardCodes, appRewardTagClickHisBucket));
                }
            }

        } catch (Exception e) {

            logger.error("MaterialFeatureParse.generateFeatureMapStatic error:", e);
            logger.error(JSONObject.toJSONString(cf));
        }

        return retMap;
    }

    private static Map<Integer, Map<String, Integer>> getSplitTimeMap(Map<String, ComponentInfo> componentInfoMap) {
        Map<Integer, Map<String, Integer>> splitMap = new HashMap<>();
        for (Map.Entry<String, ComponentInfo> entry : componentInfoMap.entrySet()) {
            String key = entry.getKey();
            ComponentInfo componentInfo = entry.getValue();
            Map<String, Integer> firstMap = splitMap.getOrDefault(0, new HashMap<>());
            Map<String, Integer> secMap = splitMap.getOrDefault(1, new HashMap<>());
            Map<String, Integer> thirdMap = splitMap.getOrDefault(2, new HashMap<>());
            Map<String, Integer> fourthMap = splitMap.getOrDefault(3, new HashMap<>());
            Map<String, Integer> fifthMap = splitMap.getOrDefault(4, new HashMap<>());
            Map<String, Integer> sixthMap = splitMap.getOrDefault(5, new HashMap<>());

            firstMap.put(key, getValue(componentInfo.getCurrDayExpCnt()));
            secMap.put(key, getValue(componentInfo.getCurrDayClickCnt()));
            thirdMap.put(key, getValue(componentInfo.getLastSevenDaysExpCnt()));
            fourthMap.put(key, getValue(componentInfo.getLastSevenDaysClickCnt()));
            fifthMap.put(key, getValue(componentInfo.getHisExpCnt()));
            sixthMap.put(key, getValue(componentInfo.getHisClickCnt()));
            splitMap.put(0, firstMap);
            splitMap.put(1, thirdMap);
            splitMap.put(2, fifthMap);
            splitMap.put(3, secMap);
            splitMap.put(4, fourthMap);
            splitMap.put(5, sixthMap);
        }
        return splitMap;
    }

    private static Integer getValue(Integer num) {
        if (num != null) {
            return num;
        }
        return 0;
    }


    public static Map<String, String> generateFeatureMapDynamic(MaterialFeatureDo cf, MaterialFeatureDo staticCf, Long materialId) {
        Map<String, String> retMap = new HashMap<>(F_MAX_SIZE);
        if (cf == null) {
            return retMap;
        }
        try {
            Map<Long, MaterialFeatureInfo> featureInfoMap = cf.getMaterialFeatureInfoMap();
            if (null == featureInfoMap) {
                return retMap;
            }

            if (featureInfoMap.containsKey(materialId)) {
                // 添加广告二级类目id编码
                if (AdvertIndustryCodes.containsKey(String.valueOf(materialId))) {
                    retMap.put("f109002",AdvertIndustryCodes.get(String.valueOf(materialId)));
                }

                MaterialFeatureInfo materialFeatureInfo = featureInfoMap.get(materialId);
                if (null != materialFeatureInfo) {
                    // 素材自身特征
                    MaterialFeature materialFeature = materialFeatureInfo.getMaterialFeature();
                    if (null != materialFeature) {
                        retMap.put("f390000", DataUtil.Long2String(materialId));
                        retMap.put("f390005", materialFeature.getCustomization());
                        retMap.put("f390006", materialFeature.getIndustry());
                        retMap.put("f390007", materialFeature.getSlotNature());
                        retMap.put("f390008", materialFeature.getDesignHue());

                        retMap.put("f390047", materialFeature.getStrategy());
                        retMap.put("f390048", materialFeature.getRewardElement());
                        retMap.put("f390049", materialFeature.getDesignType());

                        // 添加用户当天/过去7天 对目前素材所属标签的曝光次数/点击次数
                        UserFeature userFeature = Optional.ofNullable(cf.getUserFeature()).orElse(new UserFeature());
                        Map<String, ComponentInfo> userToMatTagMap = userFeature.getUserToMatTagMap();
                        if (!AssertUtil.isEmpty(userToMatTagMap)) {
                            Map<Integer, Map<String, Integer>> splitTimeMap = getSplitTimeMap(userToMatTagMap);
                            retMap.put("f391073", getCurrMaterialTagInfo(materialFeature, splitTimeMap.get(0)));
//                            retMap.put("f391074", getCurrMaterialTagInfo(materialFeature, splitTimeMap.get(1)));
//                            retMap.put("f391075", getCurrMaterialTagInfo(materialFeature, splitTimeMap.get(3)));
//                            retMap.put("f391076", getCurrMaterialTagInfo(materialFeature, splitTimeMap.get(4)));
                        }

                        retMap.put("f390009", DataUtil.Long2String(bucket(materialFeature.getExposeWeekCnt(), expWeekBucket)));
                        retMap.put("f390010", DataUtil.Long2String(bucket(materialFeature.getClickWeekCnt(), clickWeekBucket)));

                        retMap.put("f390011", DataUtil.Long2String(bucket(materialFeature.getConvertWeekCnt(), convertCntBucket)));
                        retMap.put("f390012", DataUtil.Long2String(bucket(materialFeature.getBConvertWeekCnt(), bconvertCntBucket)));

                        retMap.put("f390013", DataUtil.Long2String(bucket(materialFeature.getExposeDayCnt(), expDayBucket)));
                        retMap.put("f390014", DataUtil.Long2String(bucket(materialFeature.getClickDayCnt(), clickDayBucket)));

                        Double weekCtr = calWilsonScore(materialFeature.getExposeWeekCnt(), materialFeature.getClickWeekCnt());
                        Double dayCtr = calWilsonScore(materialFeature.getExposeDayCnt(), materialFeature.getClickDayCnt());
                        retMap.put("f390042", DataUtil.Long2String(bucket(weekCtr, userCtrBucket)));
                        retMap.put("f390043", DataUtil.Long2String(bucket(dayCtr, userCtrBucket)));

                        retMap.put("f390050", DataUtil.Long2String(bucket(materialFeature.getSckExpUV14D(), sckExpUvBucket)));
                        retMap.put("f390052", DataUtil.Long2String(bucket(materialFeature.getSckExpUV7D(), sckExpUvBucket)));
                        retMap.put("f390054", DataUtil.Long2String(bucket(materialFeature.getSckExpUV3D(), sckExpUvBucket)));
                        retMap.put("f390051", DataUtil.Long2String(bucket(materialFeature.getSckClkUV14D(), sckClkUvBucket)));
                        retMap.put("f390053", DataUtil.Long2String(bucket(materialFeature.getSckClkUV7D(), sckClkUvBucket)));
                        retMap.put("f390055", DataUtil.Long2String(bucket(materialFeature.getSckClkUV3D(), sckClkUvBucket)));

                    }

                    //用户-素材
                    // todo 用户对素材的统计值是否需要做分桶
                    UserMaterialFeature umFeature = materialFeatureInfo.getUserMaterialFeature();
                    if (null != umFeature) {
                        retMap.put("f390020", DataUtil.Long2String(min(umFeature.getUExposeHistCnt(), 100L)));
                        retMap.put("f390021", DataUtil.Long2String(min(umFeature.getUExposeWeekCnt(), 100L)));

                        retMap.put("f390022", bucket(umFeature.getUExposeInterval(), timeBucket).toString());


                        retMap.put("f390023", DataUtil.Long2String(umFeature.getUExposePeriod()));

                        retMap.put("f390024", DataUtil.Long2String(min(umFeature.getUClickHistCnt(), 100L)));
                        retMap.put("f390025", DataUtil.Long2String(min(umFeature.getUClickWeekCnt(), 100L)));

                        retMap.put("f390026", DataUtil.Long2String(min(umFeature.getUClickDayCnt(), 100L)));
                        retMap.put("f390027", DataUtil.Long2String(min(umFeature.getUExposeDayCnt(), 100L)));

                        retMap.put("f390028", bucket(umFeature.getUClickInterval(), timeBucket).toString());
                        retMap.put("f390029", DataUtil.Long2String(umFeature.getUClickPeriod()));


                        Double uhisCtr = calWilsonScore(umFeature.getUExposeHistCnt(), umFeature.getUClickHistCnt());
                        Double uweekCtr = calWilsonScore(umFeature.getUExposeWeekCnt(), umFeature.getUClickWeekCnt());

                        retMap.put("f390044", DataUtil.Long2String(bucket(uhisCtr, userCtrBucket)));
                    }

                    //广告位-素材
                    SlotMaterialFeature smFeature = materialFeatureInfo.getSlotMaterialFeature();
                    if (null != smFeature) {
                        retMap.put("f390030", DataUtil.Long2String(bucket(smFeature.getSlotClickDayCnt(), sclickDayBucket)));
                        retMap.put("f390031", DataUtil.Long2String(bucket(smFeature.getSlotClickWeekCnt(), sclickWeekBucket)));
                        retMap.put("f390032", DataUtil.Long2String(bucket(smFeature.getSlotExposeWeekCnt(), sexpWeekBucket)));
                        retMap.put("f390033", DataUtil.Long2String(bucket(smFeature.getSlotExposeDayCnt(), sexpDayBucket)));

                        retMap.put("f390034", DataUtil.Long2String(smFeature.getSlotExposePeriod()));
                        retMap.put("f390035", DataUtil.Long2String(smFeature.getSlotClickPeriod()));
                        retMap.put("f390036", DataUtil.Long2String(bucket(smFeature.getSlotConvertHistCnt(), convertCntBucket)));
                        retMap.put("f390037", DataUtil.Long2String(bucket(smFeature.getSlotBConvertHistCnt(), bconvertCntBucket)));
                        retMap.put("f390038", DataUtil.Long2String(bucket(smFeature.getSlotConvertWeekCnt(), convertCntBucket)));
                        retMap.put("f390039", DataUtil.Long2String(bucket(smFeature.getSlotBConvertWeekCnt(), bconvertCntBucket)));

                        Double sweekCtr = calWilsonScore(smFeature.getSlotExposeWeekCnt(), smFeature.getSlotClickWeekCnt());
                        Double sdayCtr = calWilsonScore(smFeature.getSlotExposeDayCnt(), smFeature.getSlotClickDayCnt());

                        retMap.put("f390045", DataUtil.Long2String(bucket(sweekCtr, slotCtrBucket)));
                        retMap.put("f390046", DataUtil.Long2String(bucket(sdayCtr, slotCtrBucket)));

                        retMap.put("f390056", DataUtil.Long2String(bucket(smFeature.getSlotSckExpUV14D(), slotSckExpUvBucket)));
                        retMap.put("f390058", DataUtil.Long2String(bucket(smFeature.getSlotSckExpUV7D(), slotSckExpUvBucket)));
                        retMap.put("f390060", DataUtil.Long2String(bucket(smFeature.getSlotSckExpUV3D(), slotSckExpUvBucket)));
                        retMap.put("f390057", DataUtil.Long2String(bucket(smFeature.getSlotSckClkUV14D(), slotSckClkUvBucket)));
                        retMap.put("f390059", DataUtil.Long2String(bucket(smFeature.getSlotSckClkUV7D(), slotSckClkUvBucket)));
                        retMap.put("f390061", DataUtil.Long2String(bucket(smFeature.getSlotSckClkUV3D(), slotSckClkUvBucket)));
                    }

                    //媒体(app)-素材
                    AppMaterialFeature amFeature = materialFeatureInfo.getAppMaterialFeature();
                    if (null != amFeature) {
                        retMap.put("f215000", DataUtil.Long2String(bucket(amFeature.getAppSckExpPV14D(), appSckExpPvBucket)));
                        retMap.put("f215104", DataUtil.Long2String(bucket(amFeature.getAppSckExpPV7D(), appSckExpPvBucket)));
                        retMap.put("f215108", DataUtil.Long2String(bucket(amFeature.getAppSckExpPV3D(), appSckExpPvBucket)));

                        retMap.put("f215102", DataUtil.Long2String(bucket(amFeature.getAppSckExpUV14D(), appSckExpUvBucket)));
                        retMap.put("f215106", DataUtil.Long2String(bucket(amFeature.getAppSckExpUV7D(), appSckExpUvBucket)));
                        retMap.put("f215110", DataUtil.Long2String(bucket(amFeature.getAppSckExpUV3D(), appSckExpUvBucket)));

                        retMap.put("f215101", DataUtil.Long2String(bucket(amFeature.getAppSckClkPV14D(), appSckClkBucket)));
                        retMap.put("f215103", DataUtil.Long2String(bucket(amFeature.getAppSckClkUV14D(), appSckClkBucket)));
                        retMap.put("f215105", DataUtil.Long2String(bucket(amFeature.getAppSckClkPV7D(), appSckClkBucket)));
                        retMap.put("f215107", DataUtil.Long2String(bucket(amFeature.getAppSckClkUV7D(), appSckClkBucket)));
                        retMap.put("f215109", DataUtil.Long2String(bucket(amFeature.getAppSckClkPV3D(), appSckClkBucket)));
                        retMap.put("f215111", DataUtil.Long2String(bucket(amFeature.getAppSckClkUV3D(), appSckClkBucket)));
                    }
                }
            }

        } catch (Exception e) {
            logger.error("MaterialFeatureParse.generateFeatureMapDynamic error:", e);

        }
        return retMap;
    }

    public static String getCurrMaterialTagInfo(MaterialFeature materialFeature, Map<String, Integer> tagMap) {
        if (AssertUtil.isEmpty(tagMap)) {
            return null;
        }
        Map<String, Integer> retMap = new HashMap<>();
        processTag(materialFeature.getDesignHue(), tagMap, retMap);
        processTag(materialFeature.getDesignType(), tagMap, retMap);
        processTag(materialFeature.getIndustry(), tagMap, retMap);
        processTag(materialFeature.getRewardElement(), tagMap, retMap);
        processTag(materialFeature.getStrategy(), tagMap, retMap);
        processTag(materialFeature.getSlotNature(), tagMap, retMap);
        return getUserCategoryActLevelByBucket(retMap, materialTagCodes, userBehaviorBucket);
    }

    private static void processTag(String tag, Map<String, Integer> tagMap, Map<String, Integer> retMap) {
        if (null != tag) {
            String[] tagList = tag.split(",");
            for (String tmpTag : tagList) {
                retMap.put(tmpTag, tagMap.getOrDefault(tmpTag, 0));
            }
        }
    }


    public static Double getCtrOrCvr(Long firstPv, Long secondPv) {
        Double ret = null;
        if (AssertUtil.isAllNotEmpty(firstPv, secondPv) && firstPv > 0)
            ret = DataUtil.division(secondPv, firstPv);

        return ret;
    }

    private static double calWilsonScore(Long exposeCnt, Long clickCnt) {
        if (AssertUtil.isAnyEmpty(exposeCnt, clickCnt)) {
            return 0.0;
        }
        if (exposeCnt < clickCnt) {
            return 0.0;
        }
        // 计算 wilson 置信分数，取90%置信度
        if (exposeCnt == 0) {
            return 0.0;
        }
        double ratio = clickCnt * 1.0 / exposeCnt;
        double faithLevel = 1.645;
        double faithSquare = faithLevel * faithLevel;
        return (ratio + (faithSquare / (2 * exposeCnt)) - faithLevel * Math.sqrt(4 * exposeCnt * ratio * (1 - ratio) + faithSquare) / (2 * exposeCnt))
                / (1 + faithSquare / exposeCnt);
    }

    public static Long bucket(Long value, int[] bucketList) {
        long ret = 0;
        if (value != null && bucketList != null && bucketList.length > 0) {
            ret = bucketList.length + 1;
            for (int i = 0; i < bucketList.length; i++) {
                int bound = bucketList[i];

                if (value <= bound) {
                    ret = i + 1;
                    break;
                }
            }


        }
        return ret;
    }

    public static Long bucket(Integer value, int[] bucketList) {
        long ret = 0;
        if (value != null && bucketList != null && bucketList.length > 0) {
            ret = bucketList.length + 1;
            for (int i = 0; i < bucketList.length; i++) {
                int bound = bucketList[i];

                if (value <= bound) {
                    ret = i + 1;
                    break;
                }
            }


        }
        return ret;
    }

    public static Long bucket(Double value, double[] bucketList) {
        long ret = 0;
        if (value != null && bucketList != null && bucketList.length > 0) {
            ret = bucketList.length + 1;
            for (int i = 0; i < bucketList.length; i++) {
                double bound = bucketList[i];

                if (value <= bound) {
                    ret = i + 1;
                    break;
                }
            }


        }
        return ret;
    }


    public static Long getLog(Long lvalue) {
        Long ret = null;
        if (lvalue != null && lvalue > 0) {
            ret = Math.round(log(lvalue.doubleValue(), 500));
        }
        return ret;

    }

    public static double log(double value, double base) {
        return Math.log(value) / Math.log(base);
    }

    public static Integer level(Long value, Long base, Integer min, Integer max) {
        Integer ret = null;
        if (value != null && value > 0) {
            ret = Math.round(value / base);
            ret = Math.max(Math.min(ret, max), min);
        }
        return ret;
    }


    public static Long min(Long value, Long base) {
        if (AssertUtil.isEmpty(value)) {
            return 0L;
        }
        if (value >= base) {
            value = base;
        }
        return value;
    }


    public static String subList(List<String> list) {
        if (AssertUtil.isEmpty(list)) {
            return null;
        }
        List<String> subList = list;
        if (list.size() > 30) {
            subList = subList.subList(list.size() - 30, list.size());
            return StringUtils.strip(subList.toString().replace(" ", ""), "[]");
        } else {
            return StringUtils.strip(list.toString().replace(" ", ""), "[]");
        }
    }


    public static String sortSubList(String string) {
        if (AssertUtil.isEmpty(string)) {
            return null;
        }
        List<String> list = Arrays.asList(string.split(","));

        Collections.sort(list, new Comparator<String>() {
            @Override
            public int compare(String o1, String o2) {
                String[] array1 = o1.split("-");
                String[] array2 = o2.split("-");
                return array1[0].compareTo(array2[0]);
            }
        });
        List<String> newArray = new ArrayList<>();
        for (String s : list) {
            if (s.split("-").length >= 2) {
                newArray.add(s.split("-")[1]);
            } else {
                newArray.add(s);
            }
        }
        return subList(newArray);
    }

    public static void putMapNotNull(Map<String, String> map, String key, String value) {

        if (map == null || StringUtils.isBlank(value)) {
            return;
        }

        map.put(key, value);
    }

    public static void main(String[] args) {
        String string0 = "20100724201049-0716,20200727201101-01182,20250723201243-187751,20200723201049-716,20200723201101-1182,20200723201243-18751," +
                "20300723201349-7716,20200723201101-1182,20200723200843-187571,20200723201049-716,20200723201901-1182,20200723201243-18751,20200723201349-716,20200723201101-11782,20200723200843-18751,20200723201049-716,20200723201901-1182,20200723201243-18751,20200723201349-716,20200723201101-1182,20200723200843-18751,20200723201049-716,20200723201901-1182,20200723201243-18751,20200723201349-716,20200723201101-1182,20200723200843-18751,20200723201049-716,20200723201901-1182,20200723111243-18751,20200723201349-716,20200723201101-1182,20200723200843-18751,20200723201049-7146,20200723101901-1182,20200723201243-187251,20200723201349-716,20200723201101-1182,20200723200843-18751,20200723201049-716,20200723201901-11282,20200723201243-18751,20200723201349-716,20200723201101-1182,20200723200843-187251,20200723201049-7126,20200723201901-1182,20200723201243-18751,20200723201349-716,20200723201101-11822,20200723200843-187521,20200723201049-7126,20200723201901-1182,20200723201243-18752";
        String string = "1,2,2,3,2,2,2,2,1,2,3,4,5,6,7,56,6,7,8,89,12,3,4,5,1,2,3,45,89,78,76,56,56,43,53,2,2";
//
        System.out.println(sortSubList(string0));


        String str = "1,2,2,3,2,2,2,2,1,2,3,4,5,6,7,56,6,7,8,89,12,3,4,5,1,2,3,45,89,78,76,56,56,43,53,2,2";
//        String str = "1";
        List<String> list = Arrays.asList(str.split(","));
        System.out.println(subList(list));
//        System.out.println(min(1000L,100L));


    }

}

