package cn.com.duiba.nezha.compute.common.model.activityselect;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.PriorityQueue;
import java.util.Comparator;
import java.util.Queue;


public class ActivitySelector {

    private static final Logger logger = LoggerFactory.getLogger(ActivitySelector.class);

    static class Constant {
        static double MIN_REWARD = 0.1;
        static long DISCOUNT = 2;
        static double DECAY = 0.999;  //1000次以前的观察，无效
    }

    static class RankInfo {
        double grpm;
        double gexp;
        double hrpm;
        double hexp;
        double arpm;
        double aexp;
    }

    static class MatchInfo {
        double score;
        ActivityInfo act;
    }

    public static Comparator<MatchInfo> iComparator = new Comparator<MatchInfo>() {

        @Override
        public int compare(MatchInfo c1, MatchInfo c2) {
            return (int) (c1.score - c2.score);
        }
    };

    //recall
    public static List<ActivityInfo> match(List<ActivityInfo> activityInfos, int topn) {
        List<ActivityInfo> result = new ArrayList<>();

        Queue<MatchInfo> candis = new PriorityQueue<>(activityInfos.size(), iComparator);

        //历史表现好
        for (ActivityInfo activity : activityInfos) {

            //System.out.println(activity.hisRequest.globalVal);
            if (activity.hisRequest.globalVal < 100)        //state: new
            {
                if (activityInfos.size()>100 && Math.random() < 0.01) {
                    result.add(activity);
                }
                else
                {
                    result.add(activity);
                }
            } else                                        //state: old
            {
                double slotRpmLeft = activity.hisSend.slotVal != Double.NaN ? wilsonRoofLeft(activity.hisSend.slotVal / 5, activity.hisRequest.slotVal) : 0;
                double globalRpmLeft = activity.hisSend.globalVal != Double.NaN ? wilsonRoofLeft(activity.hisSend.globalVal / 5, activity.hisRequest.globalVal) : 0;
                double appRpmLeft = activity.hisSend.appVal != Double.NaN ? wilsonRoofLeft(activity.hisSend.appVal / 5, activity.hisRequest.appVal) : 0;

                //1.slot优先 2.slot不优先
                double coef = 0, rpm = 0;
                double confidence = Math.min(activity.hisSend.appVal / 1000, 1);

                coef = confidence * 0.7 * Math.ceil(Math.max(slotRpmLeft, appRpmLeft)) + (1 - confidence) * 0.3 * Math.ceil(globalRpmLeft);
                rpm = coef / (confidence * 0.7 * Math.max(slotRpmLeft, appRpmLeft) + (1 - confidence) * 0.3 * globalRpmLeft);

                MatchInfo info = new MatchInfo();
                info.act = activity;
                info.score = rpm;
                candis.add(info);
                candis.add(info);
            }
        }

        //System.out.print(candis.size());

        for (int i = 0; i < topn && i < candis.size(); i++) {
            result.add(candis.poll().act);
        }

        return result;
    }

    //100 - 10 - 1
    public static ActivityInfo select(List<ActivityInfo> activityInfos) {

        //1、init
        ArrayList<Double> rewards = new ArrayList<>();
        ArrayList<Double> counts = new ArrayList<>();
        ArrayList<Double> alphas = new ArrayList<>();
        ArrayList<Double> betas = new ArrayList<>();

        double decay = Constant.DECAY;

        HashMap<Long, RankInfo> mMap = new HashMap();
        double maxG = 0.1, maxH = 0.1, maxA = 0.1;

        //2、match
        activityInfos = match(activityInfos, 100);

        //3、rank
        //3.1 get info
        for (ActivityInfo act : activityInfos) {

            double request = 0;
            double send = 0;

            //get global data
            request = act.request.globalVal - act.lastRequest.globalVal; //半小时
            send = act.send.globalVal - act.lastSend.globalVal;

            act.hisSend.globalVal = act.hisSend.globalVal > 10000 ? act.hisSend.globalVal / Constant.DISCOUNT : act.hisSend.globalVal + send;
            act.hisRequest.globalVal = act.hisSend.globalVal > 10000 ? act.hisRequest.globalVal / Constant.DISCOUNT : act.hisRequest.globalVal + request;


            RankInfo info = mMap.containsKey(act.activityId) ? mMap.get(act.activityId) : new RankInfo();
            double grpm = act.hisSend.globalVal / act.hisRequest.globalVal;
            info.grpm = grpm;
            info.gexp = act.hisRequest.globalVal;
            maxG = Math.max(grpm, maxG);

            //get app data
            request = act.request.appVal - act.lastRequest.appVal;
            send = act.send.appVal - act.lastSend.appVal;
            act.hisSend.appVal = act.hisSend.appVal > 10000 ? act.hisSend.appVal / Constant.DISCOUNT : act.hisSend.appVal + send;
            act.hisRequest.appVal = act.hisSend.appVal > 10000 ? act.hisRequest.appVal / Constant.DISCOUNT : act.hisRequest.appVal + request;
            double arpm = act.hisSend.appVal / act.hisRequest.appVal;
            info.arpm = arpm;
            info.aexp = act.hisRequest.appVal;
            mMap.put(act.activityId, info);
            if (info.hexp > 50)
                maxA = Math.max(arpm, maxA);

            //get slot data
            request = act.request.slotVal - act.lastRequest.slotVal;
            send = act.send.slotVal - act.lastSend.slotVal;
            act.hisSend.slotVal = act.hisSend.slotVal > 10000 ? act.hisSend.slotVal / Constant.DISCOUNT : act.hisSend.slotVal + send;
            act.hisRequest.slotVal = act.hisSend.slotVal > 10000 ? act.hisRequest.slotVal / Constant.DISCOUNT : act.hisRequest.slotVal + request;
            double hrpm = act.hisSend.slotVal / act.hisRequest.slotVal;
            info.hrpm = hrpm;
            info.hexp = act.hisRequest.slotVal;
            mMap.put(act.activityId, info);
            if (info.hexp > 50)
                maxH = Math.max(hrpm, maxH);
            //System.out.println(hrpm+"\t"+maxH);
        }

        List<ActivityInfo> candiList = new ArrayList<>();

        for (ActivityInfo act : activityInfos) {
            if (act.request.globalVal > 0) {
                double reward = Constant.MIN_REWARD;
                double confidence = act.hisSend.slotVal > 1000 ? 1 : act.hisSend.slotVal / 1000;

                reward = (1 - confidence) * normlize(mMap.get(act.activityId).grpm, maxG, 1)
                        + confidence * normlize(mMap.get(act.activityId).hrpm, maxH, 1);
                reward = Math.max(reward, Constant.MIN_REWARD);

                //init
                if (act.count < 20) {
                    act.count = 20;
                    act.reward = Math.min(mMap.get(act.activityId).grpm *  act.count / maxG, act.count);
                }

                act.reward = act.reward * decay + reward;
                act.count = act.count * decay + 1.0;
                act.alpha = 1.0 + act.reward;
                act.beta = 1.0 + (act.count - act.reward);


            }

            rewards.add(act.reward);
            counts.add(act.count);
            alphas.add(act.alpha);
            betas.add(act.beta);
            candiList.add(act);
        }

        //3、select
        int numMachines = candiList.size();
        //System.out.println(alphas.toString()+" "+betas.toString());
        ActivityInfo result = candiList.get(selectMachine(alphas, betas, numMachines));


        if(System.currentTimeMillis() - result.updateTime > 60 * 1000 && result.request.globalVal - result.lastRequest.globalVal > 100)
        {
            result.isUpdate = true;
        }
        mMap.clear();

        result.lastRequest = result.request;
        result.lastSend = result.send;
        result.lastClick = result.click;

        return result;
    }


    public static double normlize(double val, double max, double limit) {
        double norm = Math.min(val * limit / max, limit);
        return norm;
    }


    public ActivitySelector() { //下周

    }


    private static int selectMachine(List<Double> alphas, List<Double> betas, int numMachines) {
        int selectMachine = 0;
        double maxTheta = 0;
        for (int i = 0; i < numMachines; i++) {
            double theta = BetaDistribution.BetaDist(alphas.get(i), betas.get(i));
            if (theta > maxTheta) {
                maxTheta = theta;
                selectMachine = i;
            }
        }
        return selectMachine;
    }

    private double getCtr(double exp, double clk) {
        return exp > 0 ? clk / exp : 0;
    }

    private double sum(List<Long> list) {
        double sum = 0;
        for (Long val : list) {
            sum += val;
        }
        return sum;
    }

    public static double wilsonRoofLeft(Double ratio, Double num) {
        if (ratio == null || num == null)
            return 0;
        long fenzi = (long) (ratio * num);
        long fenmu = num.longValue();
        WilsonPair pair = WilsonInterval.wilsonCalc(fenzi, fenmu);
        return pair.upperBound;
    }

    public double wilsonRoofRight(Double num, Double ratio) {
        if (ratio == null || num == null)
            return 0;
        if (ratio == 0) {
            ratio = 0.00000001;
        }
        long fenmu = (long) (num / ratio);
        long fenzi = num.longValue();
        WilsonPair pair = WilsonInterval.wilsonCalc(fenzi, fenmu);
        return pair.upperBound;
    }

    public double wilsonBottom(Long fenzi, Long fenmu) {
        if (fenzi == null || fenmu == null)
            return 0;
        WilsonPair pair = WilsonInterval.wilsonCalc(fenzi.longValue(), fenmu.longValue());
        return pair.lowerBound;
    }

    public static class SingletonHolder {
        private static final ActivitySelector instance = new ActivitySelector();
    }

    private static ActivitySelector getInstance() {
        return SingletonHolder.instance;
    }
}