package cn.com.duiba.nezha.engine.biz.bo.advert;

import cn.com.duiba.nezha.alg.alg.alg.BudgetSmoothAlg;
import cn.com.duiba.nezha.alg.alg.vo.BudgetDo;
import cn.com.duiba.nezha.alg.alg.vo.BudgetSmoothDo;
import cn.com.duiba.nezha.alg.alg.vo.SmoothResultDo;
import cn.com.duiba.nezha.engine.biz.domain.advert.OrientationPackage;
import cn.com.duiba.nezha.engine.biz.service.CacheService;
import cn.com.duiba.nezha.engine.common.utils.MapUtils;
import cn.com.duiba.nezha.engine.common.utils.RedisKeyUtil;
import cn.com.duiba.nezha.engine.common.utils.StringRedisHelper;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import com.alibaba.fastjson.JSONObject;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.RateLimiter;
import org.springframework.stereotype.Service;

import java.time.LocalDate;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

@Service
public class AdvertBudgetSmoothService extends CacheService {

    private static final RateLimiter RATE_LIMITER = RateLimiter.create(0.02);

    // 预算对象(实时数据提供)
    private LoadingCache<String, BudgetDo> budgetDoCache = CacheBuilder.newBuilder()
            .expireAfterWrite(30L, TimeUnit.SECONDS)
            .build(new CacheLoader<String, BudgetDo>() {
                @Override
                public BudgetDo load(String key) throws Exception {
                    throw new UnsupportedOperationException("not support single query");
                }

                @Override
                public Map<String, BudgetDo> loadAll(Iterable<? extends String> keys) {
                    return StringRedisHelper.of(nezhaStringRedisTemplate).valueMultiGet
                            (keys, BudgetDo.class, BudgetDo::new);
                }
            });

    // 上次时段预算平滑对象
    private LoadingCache<BudgetQuery, Optional<BudgetSmoothDo>> lastBudgetSmoothDoCache = CacheBuilder.newBuilder()
            .expireAfterWrite(60L, TimeUnit.MINUTES)
            .build(new CacheLoader<BudgetQuery, Optional<BudgetSmoothDo>>() {
                @Override
                public Optional<BudgetSmoothDo> load(BudgetQuery key) throws Exception {
                    return Optional.empty();
                }

                @Override
                public Map<BudgetQuery, Optional<BudgetSmoothDo>> loadAll(Iterable<? extends BudgetQuery> keys) {
                    List<BudgetQuery> ks = Lists.newArrayList(keys);
                    Map<BudgetQuery, Optional<BudgetSmoothDo>> map = new HashMap<>(ks.size());
                    ks.forEach(k -> map.put(k, Optional.empty()));
                    return map;
                }
            });


    // 广告/媒体的平滑预算
    private LoadingCache<BudgetQuery, BudgetSmoothDo> budgetSmoothDoCache = CacheBuilder.newBuilder()
            .expireAfterWrite(30L, TimeUnit.SECONDS)
            .build(new CacheLoader<BudgetQuery, BudgetSmoothDo>() {
                @Override
                public BudgetSmoothDo load(BudgetQuery key) throws Exception {
                    throw new UnsupportedOperationException("not support single query");
                }

                @Override
                public Map<BudgetQuery, BudgetSmoothDo> loadAll(Iterable<? extends BudgetQuery> keys) {
                    List<BudgetQuery> newQueries = Lists.newArrayList(keys);
                    return getBudgetSmoothDos(newQueries);
                }

            });


    public void getNewBudgetSmooth(Long appId, Set<OrientationPackage> orientationPackages) {
        try {
            DBTimeProfile.enter("getNewBudgetSmooth");

            // 过滤出需要扶持或者平滑预算的配置
            Set<OrientationPackage> needSupportOrSmoothBudgetPackages = orientationPackages.stream()
                    .filter(orientationPackage -> orientationPackage.getNeedSupportWeight() || orientationPackage.enableBudgetSmooth())
                    .collect(Collectors.toSet());

            if (needSupportOrSmoothBudgetPackages.isEmpty()) {
                return;
            }


            List<BudgetQuery> allKeys = new ArrayList<>(needSupportOrSmoothBudgetPackages.size() * 2);
            Map<OrientationPackage, BudgetQuery> advertQueries = new HashMap<>(needSupportOrSmoothBudgetPackages.size());
            Map<OrientationPackage, BudgetQuery> advertAppQueries = new HashMap<>(needSupportOrSmoothBudgetPackages.size());

            needSupportOrSmoothBudgetPackages.forEach(orientationPackage -> {
                Long advertId = orientationPackage.getAdvertId();
                Long packageId = orientationPackage.getId();

                BudgetQuery advertQuery = new BudgetQuery();
                advertQuery.setAdvertId(advertId);
                advertQuery.setPackageId(packageId);

                BudgetQuery advertAppQuery = new BudgetQuery();
                advertAppQuery.setAdvertId(advertId);
                advertAppQuery.setPackageId(packageId);
                advertAppQuery.setAppId(appId);

                advertQueries.put(orientationPackage, advertQuery);
                advertAppQueries.put(orientationPackage, advertAppQuery);
                allKeys.add(advertQuery);
                allKeys.add(advertAppQuery);
            });


            // 获取本次平滑对象
            Map<BudgetQuery, BudgetSmoothDo> queryBudgetSmoothDoMap = budgetSmoothDoCache.getAll(allKeys);

            needSupportOrSmoothBudgetPackages.forEach(orientationPackage -> {
                BudgetQuery advertQueryKey = advertQueries.get(orientationPackage);
                BudgetSmoothDo advertBudgetSmoothDo = queryBudgetSmoothDoMap.get(advertQueryKey);

                // 提供给扶持广告用
                if (orientationPackage.getNeedSupportWeight()) {
                    orientationPackage.setBudgetSmoothDo(advertBudgetSmoothDo);
                }

                // 需要匀速的对象
                if (orientationPackage.enableBudgetSmooth()) {
                    // 获取广告+媒体纬度的平滑因子对象
                    BudgetQuery advertAppQueryKey = advertAppQueries.get(orientationPackage);
                    BudgetSmoothDo advertAppBudgetSmoothDo = queryBudgetSmoothDoMap.get(advertAppQueryKey);
                    SmoothResultDo smoothResultDo = BudgetSmoothAlg.getSmoothRatio(advertBudgetSmoothDo, advertAppBudgetSmoothDo);
                    orientationPackage.setSmoothResultDo(smoothResultDo);
                }
            });


            // 打印日志，间隔一分钟
            if (RATE_LIMITER.tryAcquire()) {
                needSupportOrSmoothBudgetPackages
                        .stream()
                        .filter(OrientationPackage::enableBudgetSmooth)
                        .forEach(orientationPackage -> logger.info("广告配置的平滑预算对象: " + JSONObject.toJSONString(orientationPackage.getSmoothResultDo())));
            }


            List<OrientationPackage> giveUpOrientationPackages = needSupportOrSmoothBudgetPackages.stream()
                    .filter(OrientationPackage::enableBudgetSmooth)
                    .filter(orientationPackage -> orientationPackage.getSmoothResultDo() != null && orientationPackage.getSmoothResultDo().getGiveUp())
                    .collect(Collectors.toList());
            orientationPackages.removeAll(giveUpOrientationPackages);

        } catch (Exception e) {
            logger.error("getNewBudgetSmooth happened error:{}", e);
        } finally {
            DBTimeProfile.release();
        }
    }


    private Map<BudgetQuery, BudgetSmoothDo> getBudgetSmoothDos(List<BudgetQuery> budgetQueries) {
        try {
            DBTimeProfile.enter("getBudgetSmoothDos");
            String dayFormat = LocalDate.now().format(DAY_FORMATTER);

            Map<BudgetQuery, String> queryKeyMap = new HashMap<>();
            budgetQueries.forEach(newBudgetQuery -> {
                Long advertId = newBudgetQuery.getAdvertId();
                Long packageId = newBudgetQuery.getPackageId();
                Long appId = newBudgetQuery.getAppId();
                queryKeyMap.put(newBudgetQuery, RedisKeyUtil.getBudgetQueryKey(appId, advertId, packageId, dayFormat));
            });


            // 获取大数据提供的实时数据
            Map<String, BudgetDo> keyBudgetMap = budgetDoCache.getAll(queryKeyMap.values());
            Map<BudgetQuery, BudgetDo> queryBudgetDoMap = MapUtils.translate(queryKeyMap, keyBudgetMap);

            // 获取上一次的预算平滑对象
            Map<BudgetQuery, Optional<BudgetSmoothDo>> lastQueryBudgetSmoothDoMap = lastBudgetSmoothDoCache.getAll(budgetQueries);


            Map<BudgetQuery, BudgetSmoothDo> resultMap = new HashMap<>(budgetQueries.size());

            for (BudgetQuery query : budgetQueries) {
                BudgetDo budgetDo = queryBudgetDoMap.get(query);
                if(budgetDo.getAdvertId() == null){
                    budgetDo = null;
                }

                BudgetSmoothDo lastBudgetSmoothDo = lastQueryBudgetSmoothDoMap.get(query).orElse(null);

                BudgetSmoothDo newBudgetSmoothDo = BudgetSmoothAlg.getBudgetRatio(budgetDo, lastBudgetSmoothDo);
                lastBudgetSmoothDoCache.put(query, Optional.of(newBudgetSmoothDo));
                resultMap.put(query, newBudgetSmoothDo);

            }
            return resultMap;

        } catch (Exception e) {
            logger.error("getAllBudgetSmoothDos happened error:{}", e);
            return new HashMap<>();
        } finally {
            DBTimeProfile.release();
        }
    }

    class BudgetQuery {
        private Long advertId;
        private Long packageId;
        private Long appId;

        public Long getAdvertId() {
            return advertId;
        }

        public void setAdvertId(Long advertId) {
            this.advertId = advertId;
        }

        public Long getPackageId() {
            return packageId;
        }

        public void setPackageId(Long packageId) {
            this.packageId = packageId;
        }


        public Long getAppId() {
            return appId;
        }

        public void setAppId(Long appId) {
            this.appId = appId;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;
            BudgetQuery that = (BudgetQuery) o;
            return Objects.equals(advertId, that.advertId) &&
                    Objects.equals(packageId, that.packageId) &&
                    Objects.equals(appId, that.appId);
        }

        @Override
        public int hashCode() {
            return Objects.hash(advertId, packageId, appId);
        }
    }

}
