package cn.com.duiba.nezha.engine.biz.remoteservice.impl.advert;

import cn.com.duiba.boot.perftest.PerfTestContext;
import cn.com.duiba.nezha.alg.alg.vo.BiddingRateDo;
import cn.com.duiba.nezha.alg.alg.vo.BiddingStatDo;
import cn.com.duiba.nezha.alg.alg.vo.NezhaStatDto;
import cn.com.duiba.nezha.engine.api.dto.*;
import cn.com.duiba.nezha.engine.api.enums.*;
import cn.com.duiba.nezha.engine.api.remoteservice.advert.RemoteAdvertRecommendService;
import cn.com.duiba.nezha.engine.api.support.RecommendEngineException;
import cn.com.duiba.nezha.engine.biz.domain.*;
import cn.com.duiba.nezha.engine.biz.domain.advert.Advert;
import cn.com.duiba.nezha.engine.biz.domain.advert.Material;
import cn.com.duiba.nezha.engine.biz.domain.advert.OrientationPackage;
import cn.com.duiba.nezha.engine.biz.domain.advert.SmoothFusePackage;
import cn.com.duiba.nezha.engine.biz.entity.nezha.advert.BizLogEntity;
import cn.com.duiba.nezha.engine.biz.enums.AdvertType;
import cn.com.duiba.nezha.engine.biz.enums.RecommendMaterialType;
import cn.com.duiba.nezha.engine.biz.log.BaseInnerLog;
import cn.com.duiba.nezha.engine.biz.service.advert.AbstractAdvertRecommendService;
import cn.com.duiba.nezha.engine.biz.service.advert.DataHandleBo;
import cn.com.duiba.nezha.engine.biz.service.advert.InteractAdvertRecommendService;
import cn.com.duiba.nezha.engine.biz.service.advert.ShowAdvertRecommendService;
import cn.com.duiba.nezha.engine.biz.service.advert.ctr.NewAdvertSupportService;
import cn.com.duiba.nezha.engine.biz.vo.advert.AdvertRecommendRequestVo;
import cn.com.duiba.nezha.engine.common.utils.AssertUtil;
import cn.com.duiba.nezha.engine.common.utils.RedisKeyUtil;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import cn.com.duiba.wolf.utils.DateUtils;
import com.dianping.cat.Cat;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.net.Inet4Address;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.TimeUnit;

import static java.util.Comparator.comparing;
import static java.util.stream.Collectors.*;


@RestController
public class RemoteAdvertRecommendServiceImpl implements RemoteAdvertRecommendService {

    private static final Logger LOGGER = LoggerFactory.getLogger(RemoteAdvertRecommendServiceImpl.class);

    @Autowired
    private ShowAdvertRecommendService showAdvertRecommendService;

    @Autowired
    private InteractAdvertRecommendService interactAdvertRecommendService;

    @Autowired
    private DataHandleBo dataHandleBo;

    @Autowired
    private NewAdvertSupportService newAdvertSupportService;

    @Resource
    private StringRedisTemplate nezhaStringRedisTemplate;

    private static String ip;

    static {
        List<String> ipList = new ArrayList<>();
        Enumeration<NetworkInterface> networkInterfaces = null;
        try {
            networkInterfaces = NetworkInterface.getNetworkInterfaces();
        } catch (SocketException e) {
            // ignore
        }
        NetworkInterface networkInterface;
        Enumeration<InetAddress> inetAddresses;
        InetAddress inetAddress;
        while (networkInterfaces.hasMoreElements()) {
            networkInterface = networkInterfaces.nextElement();
            if (networkInterface.getName().equals("eth0")) {
                inetAddresses = networkInterface.getInetAddresses();

                while (inetAddresses.hasMoreElements()) {

                    inetAddress = inetAddresses.nextElement();
                    if (inetAddress instanceof Inet4Address) { // IPV4
                        ipList.add(inetAddress.getHostAddress());
                    }
                }
            }
        }
        if (!ipList.isEmpty()) {
            ip = ipList.get(0);
        }
    }

    @Override
    public DubboResult<RcmdAdvertDto> recommend(ReqAdvertNewDto reqAdvertNewDto, String strategyId) {
        // 获取算法类型
        AdvertAlgEnum interactAdvertAlgEnum = Objects.requireNonNull(InteractAdvertAlgEnum.get(strategyId),
                "STRATEGY_ID_NOT_EXIST:" + strategyId);

        List<RcmdAdvertDto> rcmdAdvertDtos = this.doRecommend(reqAdvertNewDto, interactAdvertAlgEnum);

        if (!rcmdAdvertDtos.isEmpty()) {
            return DubboResult.successResult(rcmdAdvertDtos.get(0));
        }

        return DubboResult.successResult(null);
    }

    @Override
    public List<RcmdAdvertDto> batchRecommend(ReqAdvertNewDto reqAdvertNewDto, String strategyId) {
        AdvertAlgEnum advertAlgEnum = Optional.ofNullable(InteractAdvertAlgEnum.get(strategyId))
                .orElse(ShowAdvertAlgEnum.get(strategyId));
        Objects.requireNonNull(advertAlgEnum, "STRATEGY_ID_NOT_EXIST:" + strategyId);

        return this.doRecommend(reqAdvertNewDto, advertAlgEnum);
    }

    private List<RcmdAdvertDto> doRecommend(ReqAdvertNewDto reqAdvertNewDto, AdvertAlgEnum advertAlgEnum) {
        try {

            // 检查参数
            this.checkParam(reqAdvertNewDto);

            RecommendType recommendType;

            AdvertType advertType;

            AbstractAdvertRecommendService abstractAdvertRecommendService;
            // 展示广告
            if (!(advertAlgEnum instanceof ShowAdvertAlgEnum)) {// 互动广告
                recommendType = RecommendType.INTERACT;
                abstractAdvertRecommendService = interactAdvertRecommendService;
                advertType = AdvertType.INTERACT;
            } else {
                recommendType = RecommendType.SHOW;
                abstractAdvertRecommendService = showAdvertRecommendService;
                advertType = AdvertType.SHOW;
            }

            DBTimeProfile.enter(recommendType.getDesc() + "Recommend:" + advertAlgEnum.getType());

            // 记录Cat曲线图
            Cat.logMetricForCount(advertAlgEnum.toString());

            // 准备参数
            AdvertRecommendRequestVo advertRecommendRequestVo = new AdvertRecommendRequestVo();
            advertRecommendRequestVo.setIp(ip);

            // 设置算法策略信息
            advertRecommendRequestVo.setAdvertAlgEnum(advertAlgEnum);

            // 基础参数准备.不同的策略参数不同,可查看子类方法实现
            abstractAdvertRecommendService.prepareStrategyParameter(advertRecommendRequestVo);

            List<AdvertNewDto> advertList = reqAdvertNewDto.getAdvertList();
            Boolean invokeWeakFilter = advertRecommendRequestVo.getInvokeWeakFilter();//是否启动弱条件过滤
            RecommendMaterialType recommendMaterialType = advertRecommendRequestVo.getRecommendMaterialType();//素材推荐方式

            Map<Long, Advert> advertMap = new HashMap<>(advertList.size());
            Set<OrientationPackage> orientationPackages = new HashSet<>(advertList.size());

            // 设置请求相关信息--提前取出slotId
            AppDo appDo = AppDo.convert(reqAdvertNewDto.getAppDto());
            Long slotId = appDo.getSlotId();

            // 处理广告数据,比较复杂 提取广告数据 配置数据 素材数据 和各个维度竞价成功率 竞价次数
            this.handleData(advertList,
                    advertType,
                    invokeWeakFilter,
                    recommendMaterialType,
                    advertMap,
                    orientationPackages,
                    slotId);


            // 如果配置包列表为空,则直接返回
            if (orientationPackages.isEmpty()) {
                return new ArrayList<>();
            }


            Collection<Advert> adverts = advertMap.values();
            advertRecommendRequestVo.setAdvertMap(advertMap);

            // 设置广告相关信息
            advertRecommendRequestVo.setAdvertOrientationPackages(orientationPackages);
            this.ffmHandle(advertRecommendRequestVo);

            // 设置请求相关信息
            ConsumerDo consumerDo = ConsumerDo.convert(reqAdvertNewDto.getConsumerDto());
            RequestDo requestDo = RequestDo.convert(reqAdvertNewDto.getRequestDto());
            ActivityDo activityDo = ActivityDo.convert(reqAdvertNewDto.getAdvertActivityDto());

            Long appId = appDo.getId();

            // 处理融合数据
            dataHandleBo.handleMergeData(adverts,appId);
            // 获取调价因子
            dataHandleBo.handleAdjustPriceFactor(orientationPackages, slotId, appId);
            // 处理统计数据
            dataHandleBo.handleStatisticData(adverts, orientationPackages, appId);
            // 处理时段数据
            dataHandleBo.handleHourlyData(adverts, appId);
            // 处理标签数据
            dataHandleBo.handleTagData(adverts, appId);

            advertRecommendRequestVo.setCommonInfo(appDo, consumerDo, requestDo, activityDo);

            //设置低arpu值券阈值
            advertRecommendRequestVo.setLowArpuThresholdValue(requestDo.getLowArpuThresholdValue());


            List<OrientationPackage> orientationPackageList = abstractAdvertRecommendService.recommend(advertRecommendRequestVo);

            return this.returnHandle(orientationPackageList, advertRecommendRequestVo);
        } catch (Exception e) {
            LOGGER.error("advert recommend happened error :{},strategy:{}", e.getStackTrace(),advertAlgEnum.toString());
            return new ArrayList<>();
        } finally {
            DBTimeProfile.release();
        }
    }

    private void ffmHandle(AdvertRecommendRequestVo advertRecommendRequestVo) {
        Set<OrientationPackage> advertOrientationPackages = advertRecommendRequestVo.getAdvertOrientationPackages();
        if (advertOrientationPackages.size() <= 50) {
            return;
        }

        ModelKeyEnum ctrModelKey = advertRecommendRequestVo.getCtrModelKey();
        ModelKeyEnum cvrModelKey = advertRecommendRequestVo.getCvrModelKey();


        if (ctrModelKey != null && ctrModelKey.getModelType().equals(ModelType.FFM)) {
            advertRecommendRequestVo.setCtrModelKey(ModelKeyEnum.FTRL_FM_CTR_MODEL_v007);
        }
        if (cvrModelKey != null && cvrModelKey.getModelType().equals(ModelType.FFM)) {
            advertRecommendRequestVo.setCvrModelKey(ModelKeyEnum.FTRL_FM_BCVR_MODEL_v007);
        }

    }

    /**
     * @param advertList 配置(附带广告信息,历史遗留)
     * @param advertType 广告类型:1-互动广告,2-展示广告
     */
    public void handleData(Collection<AdvertNewDto> advertList,
                           AdvertType advertType,
                           Boolean invokeWeakFilter,
                           RecommendMaterialType recommendMaterialType,
                           Map<Long, Advert> advertMap,
                           Set<OrientationPackage> advertOrientationPackages,
                           Long slotId) {

        Map<NewAdvertSupportService.BiddingKey, Long> countMap = new HashMap<>();

        advertList.forEach((AdvertNewDto advertDto) -> {

            Long advertId = advertDto.getAdvertId();
            String newTradeTagNum = advertDto.getNewTradeTagNum();//新行业标签
            // 如果该配置为免费的.则排除
            if (advertDto.getFee() <= 0L) {
                return;
            }
            // 提取配置包基础数据
            OrientationPackage orientationPackage = OrientationPackage.convert(advertDto);

            // 如果执行弱定向过滤, 非强定向+弱定向
            if (invokeWeakFilter && orientationPackage.isWeakTarget()) {
                return;
            }

            // 提取广告基础数据
            Advert advert = Optional.ofNullable(advertMap.get(advertId)).orElse(Advert.convert(advertDto, advertType));

            // 提取素材数据
            Set<Material> materials;
            // 如果不推荐素材.则置空素材
            if (recommendMaterialType.equals(RecommendMaterialType.NONE)) {
                materials = new HashSet<>();
            } else {
                materials = Material.convert(advertDto);
            }

            // 将构建的素材数据添加到配置包中
            orientationPackage.setMaterials(materials);
            // 将构建的竞价成功率添加到配置包中 构建各个维度进价成功率 和各个维度竞价次数
            if(StringUtils.isNotEmpty(newTradeTagNum)){
                BiddingStatDo biddingStatDo = getBiddingStatDo(slotId, countMap, advertId, newTradeTagNum, orientationPackage);
                orientationPackage.setBiddingStatDo(biddingStatDo);
            }
            // 将获取到的配置包数据添加到广告里
            Set<OrientationPackage> orientationPackages = advert.getOrientationPackages();
            orientationPackages.add(orientationPackage);
            advert.setOrientationPackages(orientationPackages);

            advertOrientationPackages.add(orientationPackage);
            // 广告集合中添加广告
            advertMap.put(advertId, advert);

        });

        countMap.forEach((biddingKey,count)->newAdvertSupportService.incrBidingCount(biddingKey,count));

    }

    private BiddingStatDo getBiddingStatDo(Long slotId, Map<NewAdvertSupportService.BiddingKey, Long> countMap, Long advertId, String newTradeTagNum, OrientationPackage orientationPackage) {
        NewAdvertSupportService.BiddingKey advertKey = NewAdvertSupportService.BiddingKey.getAdvertKey(advertId, slotId);
        Long packageId = orientationPackage.getId();
        NewAdvertSupportService.BiddingKey advertAndOrientationKey = NewAdvertSupportService.BiddingKey.getAdvertAndOrientationKey(advertId, packageId, slotId);
        NewAdvertSupportService.BiddingKey slotKey = NewAdvertSupportService.BiddingKey.getSlotKey(slotId);
        NewAdvertSupportService.BiddingKey tradeAndSlotKey = NewAdvertSupportService.BiddingKey.getTradeAndSlotKey(slotId, newTradeTagNum);

        BiddingStatDo biddingStatDo = new BiddingStatDo();
        biddingStatDo.setAdvertId(advertId);
        biddingStatDo.setPlanId(packageId);
        biddingStatDo.setSlotId(slotId);


        BiddingRateDo advertBiddingRate = newAdvertSupportService.getBiddingRate(advertKey);
        BiddingRateDo advertAndOrientationBiddingRate = newAdvertSupportService.getBiddingRate(advertAndOrientationKey);
        BiddingRateDo slotBiddingRate = newAdvertSupportService.getBiddingRate(slotKey);
        BiddingRateDo tradeAndSlotBiddingRate = newAdvertSupportService.getBiddingRate(tradeAndSlotKey);

        biddingStatDo.setAdvertBiddingRateDo(advertBiddingRate);
        biddingStatDo.setAdvertAndPlanBiddingRateDo(advertAndOrientationBiddingRate);
        biddingStatDo.setTradeBiddingBiddingRateDo(tradeAndSlotBiddingRate);
        biddingStatDo.setGlobalBiddingBiddingRateDo(slotBiddingRate);


        Long advertCount = countMap.getOrDefault(advertKey, 0L);
        Long advertAndOrientationCount = countMap.getOrDefault(advertAndOrientationKey, 0L);
        Long slotCount = countMap.getOrDefault(slotKey, 0L);
        Long tradeAndSlotCount = countMap.getOrDefault(tradeAndSlotKey, 0L);
        advertCount++;
        advertAndOrientationCount++;
        slotCount++;
        tradeAndSlotCount++;
        countMap.put(advertKey, advertCount);
        countMap.put(advertAndOrientationKey, advertAndOrientationCount);
        countMap.put(slotKey, slotCount);
        countMap.put(tradeAndSlotKey, tradeAndSlotCount);
        return biddingStatDo;
    }


    private void checkParam(ReqAdvertNewDto req) {
        boolean pass = true;
        try {
            // 媒体信息
            AppDto appDto = req.getAppDto();
            // 用户信息
            ConsumerDto consumerDto = req.getConsumerDto();
            // 活动信息
            AdvertActivityDto advertActivityDto = req.getAdvertActivityDto();

            RequestDto requestDto = req.getRequestDto();
            if (AssertUtil.isAnyEmpty(appDto, consumerDto, advertActivityDto, requestDto)) {
                LOGGER.warn(" paramCheck 0 error, req = [{}], please check the req ", req);
                pass = false;
            }

            Long appId = appDto.getAppId();
            Long consumerId = consumerDto.getConsumerId();
            Long operatingActivityId = advertActivityDto.getOperatingActivityId();


            if (AssertUtil.isAnyEmpty(consumerId, appId, operatingActivityId)) {
                LOGGER.warn(" paramCheck 1 error, req = [{}], please check the req ", req);
                pass = false;
            }

            String ua = requestDto.getUa();
            String ip = requestDto.getIp();
            List<String> orderIds = requestDto.getOrderIds();

            if (AssertUtil.isAnyEmpty(ua, ip, orderIds)) {
                LOGGER.warn("paramCheck 2 error, req = [{}], please check the req ", req);
                pass = false;
            }
        } catch (Exception e) {
            LOGGER.error("paramCheck happened error :{}", e);
            pass = false;
        }

        if (!pass) {
            LOGGER.warn("req is invalid", ResultCodeEnum.PARAMS_INVALID.getDesc());
            throw new RecommendEngineException("req is invalid");
        }
    }


    @SuppressWarnings("squid:S3776")
    private List<RcmdAdvertDto> returnHandle(List<OrientationPackage> orientationPackages,
                                             AdvertRecommendRequestVo advertRecommendRequestVo) {

        Map<Long, Advert> advertMap = advertRecommendRequestVo.getAdvertMap();
        Boolean interactAdvert = advertRecommendRequestVo.getAdvertAlgEnum() instanceof InteractAdvertAlgEnum;
        RequestDo requestDo = advertRecommendRequestVo.getRequestDo();
        Double lowArpuThresholdValue = advertRecommendRequestVo.getLowArpuThresholdValue();
        List<String> orderIds = requestDo.getOrderIds();
        Long needCount = requestDo.getNeedCount();
        Long startCount = requestDo.getStartCount();
        if (interactAdvert) {
            needCount += 9;
        }

        // 根据该流量是否放弃分组
        Map<Boolean, List<OrientationPackage>> orientationPackageMap = orientationPackages.stream().collect(partitioningBy(OrientationPackage::getGiveUp));

        // 获取被放弃的广告
        List<OrientationPackage> giveUpOrientationPackageList = orientationPackageMap.getOrDefault(true, new ArrayList<>());

        // 获取没有放弃的广告
        List<OrientationPackage> onTargetOrientationPackageList = orientationPackageMap.getOrDefault(false, new ArrayList<>());
        if (onTargetOrientationPackageList.isEmpty()) {
            return new ArrayList<>();
        }

        // 根据rankScore分组.选取分数最大中随机的一个配置进行投放
        List<OrientationPackage> selectOrientationPackages = onTargetOrientationPackageList.stream()
                // 根据rankScore分组
                .collect(groupingBy(OrientationPackage::getRankScore,
                        // 随机选取其中一个广告
                        collectingAndThen(toList(), this::getRandomOne)))
                .entrySet()
                .stream()
                // 根据rankScore从打到小排序
                .sorted(comparing(Map.Entry<Double, OrientationPackage>::getKey).reversed())
                // 获取所需要的广告个数
                .limit(needCount)
                .map(Map.Entry::getValue)
                .collect(toList());

        // 获取排名第一广告的 rankScore
        Double topOneRankScore = selectOrientationPackages.get(0).getRankScore();

        // 获取放弃的流量中,rankScore大于命中广告 topOne 的第一条广告
        Optional<OrientationPackage> giveUpTopOneOrientationPackage = giveUpOrientationPackageList.stream()
                .filter(advertResortVo -> advertResortVo.getFlowTag().equals(1L))
                .filter(advertResortVo -> advertResortVo.getRankScore() >= topOneRankScore)
                .max(comparing(OrientationPackage::getRankScore));

        // 初始化需要打印日志的广告列表(未放弃的广告的topOne & 放弃广告中排名大于命中广告的 topOne)
        List<OrientationPackage> needLogPackageList = new ArrayList<>(selectOrientationPackages.size() + 1);

        SmoothFusePackage smoothFusePackage = null;

        if (giveUpTopOneOrientationPackage.isPresent()) {
            OrientationPackage giveOrientationPackage = giveUpTopOneOrientationPackage.get();
            needLogPackageList.add(giveOrientationPackage);
            if (giveOrientationPackage.getSmoothFuse()) {

                smoothFusePackage = new SmoothFusePackage();
                Long finalFee = giveOrientationPackage.getFinalFee();
                Double ctr = giveOrientationPackage.getCtr();
                double arpu = BigDecimal.valueOf(finalFee * ctr).setScale(2, RoundingMode.HALF_UP).doubleValue();
                smoothFusePackage.setAdvertId(giveOrientationPackage.getAdvertId());
                smoothFusePackage.setArpu(arpu);
            }
        }

        giveUpTopOneOrientationPackage.ifPresent(giveOrientationPackage -> {


        });

        Map<FeatureIndex, Map<String, String>> featureMap = advertRecommendRequestVo.getFeatureMap();

        List<RcmdAdvertDto> adverts = new ArrayList<>();


        int advertSize = selectOrientationPackages.size();
        if (interactAdvert && advertSize >= 2) {
            advertSize = 1;
        }

        for (int i = 0; i < advertSize; i++) {
            OrientationPackage orientationPackage = selectOrientationPackages.get(i);
            RcmdAdvertDto rcmdAdvertDto = new RcmdAdvertDto();
            Long advertId = orientationPackage.getAdvertId();
            Long packageId = orientationPackage.getId();
            rcmdAdvertDto.setOriginalAdvertId(advertId);
            Advert advert = advertMap.get(advertId);
            Set<Long> backupAdvertIds = advert.getBackupAdvertIds();

            // 如果备用广告列表不为空
            if (!backupAdvertIds.isEmpty()) {
                rcmdAdvertDto.setNeedReplace(true);// 本次广告需要替换

                // 获取该广告的备用广告(只传一个过来)
                Long backupAdvertId = new ArrayList<>(backupAdvertIds).get(0);

                // 从所有命中广告获取到该广告,替换
                orientationPackage = orientationPackages.stream()
                        .filter(orientationPackage1 -> orientationPackage1.getAdvertId().equals(backupAdvertId))
                        .findAny()
                        .orElse(orientationPackage);

            }
            Integer chargeType = orientationPackage.getChargeType();
            // 设置订单id
            String orderId = orderIds.get(i);
            orientationPackage.setOrderId(orderId);
            needLogPackageList.add(orientationPackage);
            Long materialId = orientationPackage.getMaterials().stream().findAny().map(Material::getId).orElse(null);
            FeatureIndex featureIndex = new FeatureIndex(advertId, packageId, materialId);
            rcmdAdvertDto.setOrderId(orderId);
            rcmdAdvertDto.setAdvertId(advertId);
            rcmdAdvertDto.setPackageId(packageId);
            rcmdAdvertDto.setMaterialId(materialId);
            rcmdAdvertDto.setCtr(orientationPackage.getCtr());
            rcmdAdvertDto.setStatCtr(orientationPackage.getStatCtr());
            rcmdAdvertDto.setPreCtr(orientationPackage.getPreCtr());
            rcmdAdvertDto.setCvr(orientationPackage.getCvr());
            rcmdAdvertDto.setStatCvr(orientationPackage.getStatCvr());
            rcmdAdvertDto.setPreCvr(orientationPackage.getPreCvr());
            rcmdAdvertDto.setFee(orientationPackage.getFinalFee());
            rcmdAdvertDto.setTag(orientationPackage.getFlowTag());
            rcmdAdvertDto.setRecommendApps(advertRecommendRequestVo.getRecommendApps());
            rcmdAdvertDto.setFusingOrientationPackages(new HashSet<>(advertRecommendRequestVo.getFusingOrientationPackages()));
            rcmdAdvertDto.setSupportSuccess(orientationPackage.getSupportSuccess());
            rcmdAdvertDto.setSupportWeight(orientationPackage.getSupportWeight());
            rcmdAdvertDto.setAdjustPriceFactor(orientationPackage.getAdjustPriceFactor());
            if (orientationPackage.getSmoothResultDo() != null && orientationPackage.getSmoothResultDo().getSmoothFactor() != null) {
                rcmdAdvertDto.setSmoothFactor(orientationPackage.getSmoothResultDo().getSmoothFactor());
            }

            Integer finalLowArpu;
            if(advert.getCanReplaceLowArpu().equals(1)){
                finalLowArpu = isFanlLowArpu(orientationPackage, lowArpuThresholdValue);
                if(finalLowArpu.equals(1)){
                    rcmdAdvertDto.setFinalLowArpu(finalLowArpu);
                    adverts.add(rcmdAdvertDto);
                    continue;
                }
            }else{
                finalLowArpu = 0;
            }
            rcmdAdvertDto.setFinalLowArpu(finalLowArpu);
            adverts.add(rcmdAdvertDto);
            this.incrSupportCount(orientationPackage);
            // 如果是预推荐或者压测模式.则不打印日志
            if (PerfTestContext.isCurrentInPerfTestMode()) {
                continue;
            }

            long finalStartCount = startCount + i;
            Optional.ofNullable(featureMap.get(featureIndex)).ifPresent(feature -> {
                feature.put("time", new SimpleDateFormat("yyyyMMddHHmmss").format(new Date()));
                feature.put("orderId", orderId);
                feature.put("advertId", String.valueOf(advertId));
                feature.put("f8807", Long.toString(finalStartCount));
                feature.put("chargeType", Long.toString(chargeType));
                BaseInnerLog.log(feature);
            });
        }

        // 将竞价成功的放入竞价成功率缓存中
        if(selectOrientationPackages.size() > 0 ){
            OrientationPackage orientationPackage = selectOrientationPackages.get(0);
            newAdvertSupportService.insertOutGoing(orientationPackage,advertRecommendRequestVo.getAppDo().getSlotId());
        }

        if (interactAdvert&& selectOrientationPackages.size() >= 2) {
            List<RcmdAdvertDto> otherAdverts = selectOrientationPackages.stream().skip(1).limit(9).map(orientationPackage1 -> {
                RcmdAdvertDto otherAdvert = new RcmdAdvertDto();
                otherAdvert.setAdvertId(orientationPackage1.getAdvertId());
                otherAdvert.setPackageId(orientationPackage1.getId());
                otherAdvert.setCtr(orientationPackage1.getCtr());
                otherAdvert.setCvr(orientationPackage1.getCvr());
                otherAdvert.setFee(orientationPackage1.getFinalFee());
                return otherAdvert;
            }).collect(toList());

            adverts.addAll(otherAdverts);
        }

        Map<Long, NezhaStatDto> nezhaStatDtoMap = advertRecommendRequestVo.getNezhaStatDtoMap();
        Map<Long, Double> ctrReconstructionFactorMap = advertRecommendRequestVo.getCtrReconstructionFactorMap();
        Map<Long, Double> cvrReconstructionFactorMap = advertRecommendRequestVo.getCvrReconstructionFactorMap();
        Map<Long, Double> ctrCorrectionFactorMap = advertRecommendRequestVo.getCtrCorrectionFactorMap();
        Map<Long, Double> cvrCorrectionFactorMap = advertRecommendRequestVo.getCvrCorrectionFactorMap();
        Map<Long, AdvertStatFeatureDo> advertStatFeatureMap = advertRecommendRequestVo.getAdvertStatFeatureMap();
        // 需要打印的广告列表中筛选出大于第一名的广告进行打印
        needLogPackageList.forEach(orientationPackage -> {
            BizLogEntity bizLogEntity = new BizLogEntity();
            bizLogEntity.setIp(advertRecommendRequestVo.getIp());
            bizLogEntity.setPredictCost(advertRecommendRequestVo.getPredictCost());
            bizLogEntity.setReleaseTarget(orientationPackage.getReleaseTarget());

            Long advertId = orientationPackage.getAdvertId();
            Advert advert = advertMap.get(advertId);
            if(advert.getCanReplaceLowArpu() == 1){
                Integer finalLowArpu = isFanlLowArpu(orientationPackage, lowArpuThresholdValue);
                if(finalLowArpu .equals(1) ){
                   return;
                }
            }
            Long packageId = orientationPackage.getId();
            Long materialId = orientationPackage.getMaterials().stream().findAny().map(Material::getId).orElse(null);
            Double ctr = orientationPackage.getCtr();
            Long finalFee = orientationPackage.getFinalFee();

            // 请求信息
            bizLogEntity.setTime(new SimpleDateFormat("yyyyMMddHHmmss").format(new Date()));
            String orderId = Optional.ofNullable(orientationPackage.getOrderId()).orElse(requestDo.getOrderId());
            bizLogEntity.setOrderId(orderId);
            bizLogEntity.setAlgType(advertRecommendRequestVo.getAdvertAlgEnum().getType());

            // 广告基础信息
            bizLogEntity.setAdvertId(advertId);
            bizLogEntity.setPackageId(packageId);
            bizLogEntity.setPackageType(orientationPackage.getPackageType());
            bizLogEntity.setTargetAppLimit(orientationPackage.getTargetAppLimit());
            bizLogEntity.setMaterialId(materialId);
            bizLogEntity.setFee(finalFee);
            bizLogEntity.setOriginalFee(orientationPackage.getOriginalFee());
            bizLogEntity.setNew(advert.isNew());
            bizLogEntity.setCount(advert.getCurrentCount());
            bizLogEntity.setChargeType(orientationPackage.getChargeType());
            bizLogEntity.setDiscountRate(advert.getDiscountRate());
            bizLogEntity.setAdvertWeight(advert.getWeight());

            // 媒体信息
            AppDo appDo = advertRecommendRequestVo.getAppDo();
            bizLogEntity.setAppId(appDo.getId());
            bizLogEntity.setSlotIndustryTagId(appDo.getSlotIndustryTagId());
            bizLogEntity.setSlotIndustryTagPid(appDo.getSlotIndustryTagPid());
            bizLogEntity.setAppIndustryTagId(appDo.getIndustryTagId());
            bizLogEntity.setAppIndustryTagPid(appDo.getIndustryTagPid());
            bizLogEntity.setTrafficTagId(appDo.getTrafficTagId());
            bizLogEntity.setTrafficTagPid(appDo.getTrafficTagPid());

            // 统计信息
            bizLogEntity.setFactor(orientationPackage.getAdjustPriceFactor());
            bizLogEntity.setStatCtr(orientationPackage.getStatCtr());
            bizLogEntity.setStatCvr(orientationPackage.getStatCvr());
            bizLogEntity.setPreCtr(orientationPackage.getPreCtr());
            bizLogEntity.setPreCvr(orientationPackage.getPreCvr());
            bizLogEntity.setBackendType(orientationPackage.getCvrType().toString());

            bizLogEntity.setCtr(ctr);
            bizLogEntity.setCvr(orientationPackage.getCvr());

            bizLogEntity.setArpu(BigDecimal.valueOf(finalFee * ctr).setScale(2, RoundingMode.HALF_UP).doubleValue());
            bizLogEntity.setNotFreeAdvertNum(advertMap.size());
            bizLogEntity.setBiddingAdvertNum(advertMap.size());
            bizLogEntity.setTagWeight(advert.getWeight());

            // 流量信息
            bizLogEntity.setTag(orientationPackage.getFlowTag());
            bizLogEntity.setQualityLevel(orientationPackage.getQualityLevel());
            bizLogEntity.setSimpleSupportType(orientationPackage.getSimpleSupportType());

            // 排序信息
            bizLogEntity.setRankScore(orientationPackage.getRankScore());
            bizLogEntity.setqScore(orientationPackage.getqScore());
            bizLogEntity.setRank(orientationPackage.getRank());

            // 平滑因子
            if (orientationPackage.getSmoothResultDo() != null && orientationPackage.getSmoothResultDo().getSmoothFactor() != null) {
                bizLogEntity.setSmoothFactor(orientationPackage.getSmoothResultDo().getSmoothFactor());
            }

            // 竞价成功率对象
            bizLogEntity.setBiddingStatDo(orientationPackage.getBiddingStatDo());

            // 活动信息
            bizLogEntity.setActivityId(advertRecommendRequestVo.getActivityDo().getOperatingId());
            Integer activityUseType = advertRecommendRequestVo.getActivityDo().getUseType();
            Long slotId = appDo.getSlotId();
            if (activityUseType.equals(0) || activityUseType.equals(1)) {
                bizLogEntity.setDuibaSlotId(slotId);
            } else {
                bizLogEntity.setSlotId(slotId);
            }
            bizLogEntity.setCtrCorrectionFactor(ctrCorrectionFactorMap.get(advertId));
            bizLogEntity.setCvrCorrectionFactor(cvrCorrectionFactorMap.get(advertId));
            bizLogEntity.setCtrReconstructionFactor(ctrReconstructionFactorMap.get(advertId));
            bizLogEntity.setCvrReconstructionFactor(cvrReconstructionFactorMap.get(advertId));

            Optional.ofNullable(nezhaStatDtoMap.get(advertId)).ifPresent(nezhaStatDto -> {
                bizLogEntity.setPreCtrAvg(nezhaStatDto.getPreCtrAvg());
                bizLogEntity.setPreCvrAvg(nezhaStatDto.getPreCvrAvg());
                bizLogEntity.setStatCtrAvg(nezhaStatDto.getStatCtrAvg());
                bizLogEntity.setStatCvrAvg(nezhaStatDto.getStatCvrAvg());
            });

            Optional.ofNullable(advertStatFeatureMap.get(advertId)).ifPresent(statFeatureDo -> {
                bizLogEntity.setAdvertCtr(statFeatureDo.getAdvertCtr());
                bizLogEntity.setAdvertCvr(statFeatureDo.getAdvertCvr());
                bizLogEntity.setAdvertAppCtr(statFeatureDo.getAdvertAppCtr());
                bizLogEntity.setAdvertAppCvr(statFeatureDo.getAdvertAppCvr());
                bizLogEntity.setAdvertSlotCtr(statFeatureDo.getAdvertSlotCtr());
                bizLogEntity.setAdvertSlotCvr(statFeatureDo.getAdvertSlotCvr());
                bizLogEntity.setAdvertActivityCtr(statFeatureDo.getAdvertActivityCtr());
                bizLogEntity.setAdvertActivityCvr(statFeatureDo.getAdvertActivityCvr());
            });
            if (PerfTestContext.isCurrentInPerfTestMode()) {
                return;
            }
            BaseInnerLog.log(bizLogEntity);
        });
        return adverts;

    }

    /**
     * 判断是否是低arpu值得券
     *
     * @return 1-是 0-不是
     */
    public Integer isFanlLowArpu(OrientationPackage orientationPackage, Double lowArpuThresholdValue){
        Integer finalLowArpu;
        BigDecimal arpu = BigDecimal.valueOf(orientationPackage.getFinalFee() * orientationPackage.getCtr()).setScale(2, RoundingMode.HALF_UP);
        if (arpu.compareTo(BigDecimal.valueOf(lowArpuThresholdValue)) < 0) {
            finalLowArpu = 1;
        }else{
            finalLowArpu = 0;
        }
        return finalLowArpu;
    }

    private <T> T getRandomOne(List<T> list) {
        if (list.size() == 1) {
            return list.get(0);
        }
        return list.get(new Random().nextInt(list.size()));
    }

    private void incrSupportCount(OrientationPackage orientationPackage) {
        if (orientationPackage.getRank().equals(orientationPackage.getReRank())) {
            return;
        }
        Long advertId = orientationPackage.getAdvertId();
        String supportCountKey = RedisKeyUtil.getSupportCount(advertId);
        Long increment = nezhaStringRedisTemplate.opsForValue().increment(supportCountKey, 1);
        LOGGER.info("advert :{}, support count :{}", advertId, increment);
        if (increment.equals(1L)) {
            nezhaStringRedisTemplate.expire(supportCountKey, (long) DateUtils.getToTomorrowSeconds() + new Random().nextInt(100),
                    TimeUnit.SECONDS);
        }
    }

}
