package cn.com.duiba.nezha.engine.biz.service.advert;

import cn.com.duiba.nezha.engine.biz.domain.advert.OrientationPackage;
import cn.com.duiba.nezha.engine.biz.service.CacheService;
import cn.com.duiba.nezha.engine.common.utils.RedisKeyUtil;
import cn.com.duiba.nezha.engine.common.utils.RoiHashKeyUtil;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.concurrent.TimeUnit;

@Service
public class OrientationPackageAdjustPriceFactorService extends CacheService {

    // 调价因子的缓存
    private LoadingCache<String, Double> factorCache = CacheBuilder.newBuilder()
            .expireAfterWrite(10, TimeUnit.SECONDS)
            .build(new CacheLoader<String, Double>() {
                @Override
                public Double load(String key) {
                    throw new UnsupportedOperationException("not support single query");
                }

                @Override
                public Map<String, Double> loadAll(Iterable<? extends String> keys) {
                    List<String> keyList = Lists.newArrayList(keys);
                    Map<String, Double> factorMap = new HashMap<>(keyList.size());
                    List<String> values = nezhaStringRedisTemplate.opsForValue().multiGet(keyList);
                    for (int index = 0; index < keyList.size(); index++) {
                        String key = keyList.get(index);
                        String value = values.get(index);
                        factorMap.put(key, Optional.ofNullable(value).map(Double::parseDouble).orElse(1D));
                    }
                    return factorMap;

                }
            });

    public void handle(Set<OrientationPackage> orientationPackages, Long appId, Long slotId) {
        try {
            DBTimeProfile.enter("loadAdjustPriceFactors");

            //加载所有需要的调价因子
            List<String> keys = new ArrayList<>(orientationPackages.size() * 3);

            Map<OrientationPackage, List<String>> advert2dimKeysMap = new HashMap<>(orientationPackages.size());

            orientationPackages.forEach(orientationPackage -> {
                Long advertId = orientationPackage.getAdvertId();
                Long packageId = orientationPackage.getId();
                Integer cvrType = orientationPackage.getCvrType();
                String slotKey = RedisKeyUtil.factorKey(advertId, packageId, RoiHashKeyUtil.getSlotKey(slotId,cvrType));
                String appKey = RedisKeyUtil.factorKey(advertId, packageId, RoiHashKeyUtil.getAppKey(appId,cvrType));
                String defaultKey = RedisKeyUtil.factorKey(advertId, packageId, RoiHashKeyUtil.getDefault(cvrType));
                List<String> dimKeys = Arrays.asList(slotKey, appKey, defaultKey);
                keys.addAll(dimKeys);
                advert2dimKeysMap.put(orientationPackage, dimKeys);
            });

            Map<String, Double> factorMap = new HashMap<>(0);
            try {
                factorMap = factorCache.getAll(keys);
            } catch (Exception e) {
                logger.error("load factor error :{}", e);
            }

            //根据广告提取存在且最细粒度的调价因子
            for (OrientationPackage orientationPackage : orientationPackages) {

                List<String> dimKeys = advert2dimKeysMap.get(orientationPackage);
                for (String dimKey : dimKeys) {
                    //dimkey 粒度依次增大，故排在前面的维度优先使用
                    Double factor = factorMap.get(dimKey);
                    if (checkFactor(factor)) {
                        //找到合适的调价因子后，直接结束查找
                        orientationPackage.setAdjustPriceFactor(factor);
                        break;
                    }
                }

            }
        } finally {
            DBTimeProfile.release();
        }

    }

    private boolean checkFactor(Double value) {
        return value != null && !value.isNaN() && !value.equals(1D);
    }
}
