package cn.com.duiba.nezha.engine.biz.service.advert.ctr.impl;

import cn.com.duiba.nezha.engine.biz.domain.advert.Advert;
import cn.com.duiba.nezha.engine.biz.entity.nezha.advert.AdvertStatisticMergeEntity;
import cn.com.duiba.nezha.engine.biz.entity.nezha.advert.TagCvrData;
import cn.com.duiba.nezha.engine.biz.service.CacheService;
import cn.com.duiba.nezha.engine.biz.service.advert.ctr.TagStatAssociationService;
import cn.com.duiba.nezha.engine.common.utils.RedisKeyUtil;
import cn.com.duiba.nezha.engine.common.utils.StringRedisHelper;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Stream;

import static java.util.function.Function.identity;
import static java.util.stream.Collectors.*;

/**
 * @author ZhouFeng zhoufeng@duiba.com.cn
 * @version $Id: TagStatAssociationServiceImpl.java , v 0.1 2018/2/5 下午8:34 ZhouFeng Exp $
 */
@Service
public class TagStatAssociationServiceImpl extends CacheService implements TagStatAssociationService {

    private static final Logger LOGGER = LoggerFactory.getLogger(TagStatAssociationServiceImpl.class);

    /**
     * 每天统计缓存时间
     */
    private static final CacheDuration DAILY_STAT_CACHE_DURATION = new CacheDuration(7L, TimeUnit.DAYS);
    /**
     * 当天统计缓存时间
     */
    private static final CacheDuration CURRENT_DAY_STAT_CACHE_DURATION = new CacheDuration(5L, TimeUnit.MINUTES);
    /**
     * 当前小时
     */
    private static final CacheDuration CURRENT_HOUR_STAT_CACHE_DURATION = new CacheDuration(2L, TimeUnit.MINUTES);


    /**
     * 每天统计缓存
     * key:各维度ke+yyyyyMMdd
     */
    private LoadingCache<String, TagCvrData> dailyStatCache = CacheBuilder.newBuilder()
            .expireAfterWrite(DAILY_STAT_CACHE_DURATION.getDuration(), DAILY_STAT_CACHE_DURATION.getTimeUnit())
            .recordStats()
            .build(new CacheLoader<String, TagCvrData>() {
                @Override
                public TagCvrData load(String key) throws Exception {
                    throw new IllegalAccessException("not suppose single query");
                }

                @Override
                public Map<String, TagCvrData> loadAll(Iterable<? extends String> keys) throws Exception {

                    return StringRedisHelper.of(nezhaStringRedisTemplate).valueMultiGet
                            (keys, TagCvrData.class, TagCvrData::new);
                }
            });


    /**
     * 当天统计缓存
     * key:各维度ke+yyyyyMMdd
     */
    private LoadingCache<String, TagCvrData> currentDayStatCache = CacheBuilder.newBuilder()
            .expireAfterWrite(CURRENT_DAY_STAT_CACHE_DURATION.getDuration(), CURRENT_DAY_STAT_CACHE_DURATION.getTimeUnit())
            .recordStats()
            .build(new CacheLoader<String, TagCvrData>() {
                @Override
                public TagCvrData load(String key) throws Exception {
                    throw new IllegalAccessException("not suppose single query");
                }

                @Override
                public Map<String, TagCvrData> loadAll(Iterable<? extends String> keys) throws Exception {

                    return StringRedisHelper.of(nezhaStringRedisTemplate).valueMultiGet
                            (keys, TagCvrData.class, TagCvrData::new);
                }
            });


    /**
     * 当前小时统计缓存
     * key:各维度ke+yyyyyMMddHH
     */
    private LoadingCache<String, TagCvrData> currentHourStatCache = CacheBuilder.newBuilder()
            .expireAfterWrite(CURRENT_HOUR_STAT_CACHE_DURATION.getDuration(), CURRENT_HOUR_STAT_CACHE_DURATION.getTimeUnit())
            .recordStats()
            .build(new CacheLoader<String, TagCvrData>() {
                @Override
                public TagCvrData load(String key) throws Exception {
                    throw new IllegalAccessException("not suppose single query");
                }

                @Override
                public Map<String, TagCvrData> loadAll(Iterable<? extends String> keys) throws Exception {

                    return StringRedisHelper.of(nezhaStringRedisTemplate).valueMultiGet
                            (keys, TagCvrData.class, TagCvrData::new);
                }
            });

    @Override
    public void handleTagStat(Collection<Advert> adverts, Long appId) {
        try {
            DBTimeProfile.enter("tagStatAssociationService.handleTagStat");

            Map<Advert, Set<String>> advertMatchTags = adverts.stream()
                    .filter(advert -> StringUtils.isNotBlank(advert.getMatchTags()))
                    .collect(toMap(identity(), advert -> Stream.of(advert.getMatchTags().split(",")).collect(toSet())));
            Set<String> tagIds = advertMatchTags.values().stream().flatMap(Set::stream).collect(toSet());


            Map<String, String> appTag2ChKeyMap = new HashMap<>(tagIds.size());// 媒体当前小时
            Map<String, String> globalTag2ChKeyMap = new HashMap<>(tagIds.size());//全局当前小时
            Map<String, String> appTag2CdKeyMap = new HashMap<>(tagIds.size());// 媒体当天
            Map<String, String> globalTag2CdKeyMap = new HashMap<>(tagIds.size());// 全局当天
            Map<String, List<String>> appTag2Last6DayKeyMap = new HashMap<>(tagIds.size());//媒体最近6天
            Map<String, List<String>> globalTag2Last6DayKeyMap = new HashMap<>(tagIds.size());//全局最近6天
            String nowHour = LocalDateTime.now().format(HOUR_FORMATTER);
            String nowDay = LocalDate.now().format(DAY_FORMATTER);
            for (String tagId : tagIds) {
                appTag2ChKeyMap.put(tagId, RedisKeyUtil.tagHourlyStatKey(appId, tagId, nowHour));
                globalTag2ChKeyMap.put(tagId, RedisKeyUtil.tagHourlyStatKey(null, tagId, nowHour));

                appTag2CdKeyMap.put(tagId, RedisKeyUtil.tagDailyStatKey(appId, tagId, nowDay));
                globalTag2CdKeyMap.put(tagId, RedisKeyUtil.tagDailyStatKey(null, tagId, nowDay));

                List<String> last6DayTimestamps = last6DayTimestampCache.get(nowDay);
                appTag2Last6DayKeyMap.put(tagId, last6DayTimestamps.stream().map(time -> RedisKeyUtil.tagDailyStatKey(appId, tagId, time)).collect(toList()));
                globalTag2Last6DayKeyMap.put(tagId, last6DayTimestamps.stream().map(time -> RedisKeyUtil.tagDailyStatKey(null, tagId, time)).collect(toList()));
            }

            Set<String> currentHourlyKeys = new HashSet<>(tagIds.size() * 2);
            currentHourlyKeys.addAll(appTag2ChKeyMap.values());
            currentHourlyKeys.addAll(globalTag2ChKeyMap.values());
            ImmutableMap<String, TagCvrData> currentHourStat = currentHourStatCache.getAll(currentHourlyKeys);


            Set<String> currentDayKeys = new HashSet<>(tagIds.size() * 2);
            currentDayKeys.addAll(appTag2CdKeyMap.values());
            currentDayKeys.addAll(globalTag2CdKeyMap.values());
            ImmutableMap<String, TagCvrData> currentDayStat = currentDayStatCache.getAll(currentDayKeys);

            Set<String> dailyKeys = new HashSet<>(tagIds.size() * 12);
            dailyKeys.addAll(appTag2Last6DayKeyMap.values().stream().flatMap(Collection::stream).collect(toList()));
            dailyKeys.addAll(globalTag2Last6DayKeyMap.values().stream().flatMap(Collection::stream).collect(toList()));
            ImmutableMap<String, TagCvrData> dailyStat = dailyStatCache.getAll(dailyKeys);


            adverts.forEach(advert -> {
                        Set<String> tags = advertMatchTags.getOrDefault(advert, new HashSet<>());
                        advert.getOrientationPackages().forEach(orientationPackage -> {
                            Integer cvrType = orientationPackage.getCvrType();
                            List<AdvertStatisticMergeEntity> entities = tags.stream().map(tagId -> {

                                Double appCh = currentHourStat.get(appTag2ChKeyMap.get(tagId)).getCvr(cvrType);
                                Double appCd = currentDayStat.get(appTag2CdKeyMap.get(tagId)).getCvr(cvrType);
                                Double app7d = Stream.concat(
                                        appTag2Last6DayKeyMap.get(tagId).stream().map(dailyStat::get).map(x -> x.getCvr(cvrType))
                                        , Lists.newArrayList(appCd).stream())

                                        .filter(Objects::nonNull)
                                        .mapToDouble(x -> x).filter(value -> value > 0).average().orElse(0D);

                                Double globalCh = currentHourStat.get(globalTag2ChKeyMap.get(tagId)).getCvr(cvrType);
                                Double globalCd = currentDayStat.get(globalTag2CdKeyMap.get(tagId)).getCvr(cvrType);
                                Double global7D = Stream.concat(
                                        globalTag2Last6DayKeyMap.get(tagId).stream().map(dailyStat::get)
                                            .map(x -> x.getCvr(cvrType))
                                        , Lists.newArrayList(globalCd).stream()).filter(value -> value > 0)
                                        .mapToDouble(x -> x).average().orElse(0D);
                                return new AdvertStatisticMergeEntity.Builder().appCurrentlyHour(appCh).appCurrentlyDay
                                        (appCd).appRecently7Day(app7d).globalCurrentlyHour(globalCh).globalCurrentlyDay(globalCd)
                                        .globalRecently7Day(global7D).build();

                            }).collect(toList());
                            double appCh = entities.stream().map(AdvertStatisticMergeEntity::getAppCurrentlyHour)
                                    .filter(Objects::nonNull)
                                    .mapToDouble(Double::doubleValue)
                                    .average()
                                    .orElse(0D);

                            double appCd = entities.stream().map(AdvertStatisticMergeEntity::getAppCurrentlyDay)
                                    .filter(Objects::nonNull)
                                    .mapToDouble(Double::doubleValue)
                                    .average()
                                    .orElse(0D);

                            double app7d = entities.stream().map(AdvertStatisticMergeEntity::getAppRecently7Day)
                                    .filter(Objects::nonNull)
                                    .mapToDouble(Double::doubleValue)
                                    .average()
                                    .orElse(0D);

                            double globalCh = entities.stream().map(AdvertStatisticMergeEntity::getGlobalCurrentlyHour)
                                    .filter(Objects::nonNull)
                                    .mapToDouble(Double::doubleValue)
                                    .average()
                                    .orElse(0D);

                            double globalCd = entities.stream()
                                    .map(AdvertStatisticMergeEntity::getGlobalCurrentlyDay)
                                    .filter(Objects::nonNull)
                                    .mapToDouble(Double::doubleValue)
                                    .average()
                                    .orElse(0D);

                            double global7D = entities.stream().map(AdvertStatisticMergeEntity::getGlobalRecently7Day)
                                    .filter(Objects::nonNull)
                                    .mapToDouble(Double::doubleValue)
                                    .average()
                                    .orElse(0D);

                            orientationPackage.setTagStatisticData(new AdvertStatisticMergeEntity.Builder()
                                    .appCurrentlyHour(appCh)
                                    .appCurrentlyDay(appCd)
                                    .appRecently7Day(app7d)
                                    .globalCurrentlyHour(globalCh)
                                    .globalCurrentlyDay(globalCd)
                                    .globalRecently7Day(global7D)
                                    .build());
                        });
                    }
            );

        } catch (Exception e) {
            LOGGER.error("get tag stat error:{}", e);
        } finally {
            DBTimeProfile.release();
        }
    }

    @Override
    public Map<String, CacheInfo> getCacheInfo() {
        Map<String, CacheInfo> map = new HashMap<>();
        map.put("dailyStatCache", CacheInfo.generate(dailyStatCache));
        map.put("currentHourStatCache", CacheInfo.generate(currentHourStatCache));
        map.put("currentDayStatCache", CacheInfo.generate(currentDayStatCache));
        return map;
    }
}
