package cn.com.duiba.nezha.engine.biz.service.advert.floorprice;

import cn.com.duiba.nezha.alg.alg.basepricecontrol.BasePriceControl;
import cn.com.duiba.nezha.alg.alg.basepricecontrol.BasePriceInfo;
import cn.com.duiba.nezha.alg.alg.basepricecontrol.BasePriceParams;
import cn.com.duiba.nezha.alg.alg.basepricecontrol.BasePriceResult;
import cn.com.duiba.nezha.engine.api.support.RecommendEngineException;
import cn.com.duiba.nezha.engine.biz.domain.AppDo;
import cn.com.duiba.nezha.engine.biz.domain.FloorPriceFeatureIndex;
import cn.com.duiba.nezha.engine.biz.domain.FloorPriceStatisticData;
import cn.com.duiba.nezha.engine.biz.domain.advert.OrientationPackage;
import cn.com.duiba.nezha.engine.common.utils.MapUtils;
import cn.com.duiba.nezha.engine.common.utils.RedisKeyUtil;
import cn.com.duiba.nezha.engine.common.utils.StringRedisHelper;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

import static java.util.stream.Collectors.toMap;

/**
 * @author michealWang
 */
@Service
public class FloorPriceService {

    private static final Logger logger = LoggerFactory.getLogger(FloorPriceService.class);
    @Autowired
    private StringRedisTemplate nezhaStringRedisTemplate;

    @Resource
    private ExecutorService executorService ;
    /**
     * 广告id+配置id+应用id+cvrtype+depthCvrType 的深度调价因子
     */
    private LoadingCache<BasePriceInfo, BasePriceResult> floorPriceFactorCache = CacheBuilder.newBuilder()
            .refreshAfterWrite(1L, TimeUnit.MINUTES)
            .build(new CacheLoader<BasePriceInfo, BasePriceResult>() {
                @Override
                public BasePriceResult load(BasePriceInfo key) throws Exception {
                    return loadCache(key, null);
                }

                @Override
                public Map<BasePriceInfo, BasePriceResult> loadAll(Iterable<? extends BasePriceInfo> keys) {
                    List basePriceInfos = Lists.newArrayList(keys);
                    return loadAllCache(basePriceInfos);
                }

                @Override
                public ListenableFuture<BasePriceResult> reload(BasePriceInfo key, BasePriceResult oldValue) throws Exception {
                    ListenableFutureTask<BasePriceResult> task = ListenableFutureTask.create(() -> loadCache(key, oldValue));
                    executorService.submit(task);
                    return task;
                }
            });

    private LoadingCache<String, BasePriceParams> basePriceParamsCache = CacheBuilder.newBuilder()
            .expireAfterWrite(2L, TimeUnit.HOURS)
            .build(new CacheLoader<String, BasePriceParams>() {
                @Override
                public BasePriceParams load(String key) throws Exception {
                    return loadBasePriceParamsCache(key).orElse(null);
                }

                // 如果加载不到新值 返回 老值
                @Override
                public ListenableFuture<BasePriceParams> reload(String key, BasePriceParams oldValue) throws Exception {
                    ListenableFutureTask<BasePriceParams> task = ListenableFutureTask.create(() -> loadBasePriceParamsCache(key).orElse(oldValue));
                    executorService.submit(task);
                    return task;
                }
            });

    /**
     * 加载 算法 底价放弃 调整参数
     * @param key
     * @return
     */
    private Optional<BasePriceParams> loadBasePriceParamsCache(String key) {

        try {
            DBTimeProfile.enter("loadBasePriceParamsCache");
            return  Optional.ofNullable(key)
                    .map(modelKey -> nezhaStringRedisTemplate.opsForValue().get(key))
                    .map(json -> JSON.parseObject(json, BasePriceParams.class));
        } catch (Exception e) {
            logger.error("load Base PriceParams,PriceParamsKey:{}", key);
            throw new RecommendEngineException("load Base Price Params exception", e);
        } finally {
            DBTimeProfile.release();
        }

    }

    /**
     * 一分钟计算一次
     * 计算 底价放弃比例对象 并更新缓存
     * 读取 redis算法调整对象
     *
     * @param appDo
     * @param floorPricePackages
     * @return
     */
    public Map<FloorPriceFeatureIndex, BasePriceResult> getFloorPriceFactor(AppDo appDo, List<OrientationPackage> floorPricePackages) {

        try {
            Map<FloorPriceFeatureIndex, BasePriceInfo> floorPriceFeatureIndexBasePriceInfoMap = buildBasePriceInfo(appDo, floorPricePackages);

            Map<BasePriceInfo, BasePriceResult> priceFactorCacheAll = floorPriceFactorCache.getAll(floorPriceFeatureIndexBasePriceInfoMap.values());
            Map<FloorPriceFeatureIndex, BasePriceResult> resultMap = MapUtils.translate(floorPriceFeatureIndexBasePriceInfoMap, priceFactorCacheAll);
            return resultMap;
        } catch (Exception e) {
            logger.warn("FloorPriceService#getFloorPriceFactor happend error!", e);
            return new HashMap<>();
        }
    }

    private Map<FloorPriceFeatureIndex, BasePriceInfo> buildBasePriceInfo(AppDo appDo, List<OrientationPackage> floorPricePackages) {
        return floorPricePackages.stream().collect(toMap(
                orientationPackage -> FloorPriceFeatureIndex.conver(orientationPackage, appDo),
                orientationPackage -> getBasePriceInfo(orientationPackage, appDo)
        ));
    }

    /**
     * 构建缓存BasePriceInfo
     */
    private BasePriceInfo getBasePriceInfo(OrientationPackage orientationPackage, AppDo appDo) {
        BasePriceInfo basePriceInfo = new BasePriceInfo();
        Long advertId = orientationPackage.getAdvertId();
        Long packageId = orientationPackage.getId();
        Long slotId = appDo.getSlotId();

        basePriceInfo.setAdvertId(advertId);
        basePriceInfo.setOrientId(packageId);

        basePriceInfo.setAppId(appDo.getId());
        basePriceInfo.setSlotId(slotId);
        basePriceInfo.setAfee(orientationPackage.getConvertCost());
        basePriceInfo.setManageType(orientationPackage.getTargetAppLimit());
        return basePriceInfo;
    }

    /**
     * 批量加载缓存
     *
     * @param basePriceInfos
     * @return
     */
    private Map<BasePriceInfo, BasePriceResult> loadAllCache(List<BasePriceInfo> basePriceInfos) {
        int size = basePriceInfos.size();
        List<String> allKeys = Lists.newArrayListWithCapacity(size * 3);
        Map<BasePriceInfo, String> adPkSlKeys = Maps.newHashMapWithExpectedSize(size);
        Map<BasePriceInfo, String> adPkKeys = Maps.newHashMapWithExpectedSize(size);
        Map<BasePriceInfo, String> slKeys = Maps.newHashMapWithExpectedSize(size);

        basePriceInfos.forEach(basePriceInfo -> {
            Long advertId = basePriceInfo.getAdvertId();
            Long packageId = basePriceInfo.getOrientId();
            Long slotId = basePriceInfo.getSlotId();
            String nowDate = LocalDate.now().format(DateTimeFormatter.ofPattern("yyyyMMdd"));
            // redis 读取 数据
            // 构建keys
            // 配置+ 广告位
            String adPkSlKey = RedisKeyUtil.getNezhaFloorPriceKey(advertId, packageId, slotId,nowDate);

            // 配置维度
            String adPkKey = RedisKeyUtil.getNezhaFloorPriceKey(advertId, packageId,nowDate);

            // 广告位维度
            String slKey = RedisKeyUtil.getNezhaFloorPriceKey(slotId,nowDate);

            allKeys.add(adPkSlKey);
            adPkSlKeys.put(basePriceInfo, adPkSlKey);

            allKeys.add(adPkKey);
            adPkKeys.put(basePriceInfo, adPkKey);

            allKeys.add(slKey);
            slKeys.put(basePriceInfo, slKey);

        });
        Map<String, FloorPriceStatisticData> statisticDataMap = StringRedisHelper.of(nezhaStringRedisTemplate).valueMultiGet(allKeys, FloorPriceStatisticData.class, FloorPriceStatisticData::new);

        basePriceInfos.forEach(basePriceInfo -> {
            // 设置配置维度数据
            String adPkKey = adPkKeys.get(basePriceInfo);
            setOrientSlotFloorPriceStatisticData(basePriceInfo, adPkKey, statisticDataMap);

            // 设置配置+广告位维度数据
            String adPkSlKey = adPkSlKeys.get(basePriceInfo);
            setOrientFloorPriceStatisticData(basePriceInfo, adPkSlKey, statisticDataMap);

            // 设置广告位维度数据
            String slKey = slKeys.get(basePriceInfo);
            setSlotFloorPriceStatisticData(basePriceInfo, slKey, statisticDataMap);
        });

        // redis 读取 算法参数
        BasePriceParams basePriceParams = null;
        try {
            basePriceParams = basePriceParamsCache.get(RedisKeyUtil.getBasePriceParamsKey());
        } catch (ExecutionException e) {
          logger.error("basePriceParams load error !", e);
        }


        // 调用 算法 接口 第一次缓存接口的那些key 会进来
        Map<BasePriceInfo, BasePriceResult> resultMap = BasePriceControl.basePriceControl(basePriceInfos,null, basePriceParams);

        logger.info("调用算法获取 放弃比例 第一次加载缓存 开始");
        //todo 测试日志 记得删除
        logger.info("all basePriceInfos= {}", JSONObject.toJSONString(basePriceInfos));
        basePriceInfos.forEach(basePriceInfo -> {
            BasePriceResult basePriceResult = resultMap.get(basePriceInfo);
            logger.info("返回结果：advertId={} ,pkgId={}, sloatId={} ,GiveUpProb5={},GiveUpProb10={} ",
                    basePriceInfo.getAdvertId(),basePriceInfo.getOrientId(),basePriceInfo.getSlotId(),basePriceResult.getGiveUpProb5(), basePriceResult.getGiveUpProb10());
        });
        logger.info("调用算法获取 放弃比例 第一次加载缓存 结束");
        return  basePriceInfos.stream().collect(toMap(Function.identity(), basePriceInfo -> resultMap.getOrDefault(basePriceInfo, new BasePriceResult())));
    }




    /**
     * 单个加载缓存
     *
     * @param basePriceInfo
     * @param oldBasePriceResult
     * @return
     */
    private BasePriceResult loadCache(BasePriceInfo basePriceInfo, BasePriceResult oldBasePriceResult) {
        Long advertId = basePriceInfo.getAdvertId();
        Long packageId = basePriceInfo.getOrientId();
        Long slotId = basePriceInfo.getSlotId();
        String nowDate = LocalDate.now().format(DateTimeFormatter.ofPattern("yyyyMMdd"));

        // redis 读取 数据
        // 构建keys
        // 配置+ 广告位
        String adPkSlKey = RedisKeyUtil.getNezhaFloorPriceKey(advertId, packageId, slotId,nowDate);
        // 配置维度
        String adPkKey = RedisKeyUtil.getNezhaFloorPriceKey(advertId, packageId,nowDate);
        // 广告位维度
        String slKey = RedisKeyUtil.getNezhaFloorPriceKey(slotId,nowDate);

        List<String> keys = Lists.newArrayList(adPkKey, adPkSlKey, slKey);

        Map<String, FloorPriceStatisticData> statisticDataMap = StringRedisHelper.of(nezhaStringRedisTemplate).valueMultiGet(keys, FloorPriceStatisticData.class, FloorPriceStatisticData::new);

        // 设置配置维度数据
        setOrientFloorPriceStatisticData(basePriceInfo, adPkKey, statisticDataMap);
        // 设置广告位维度数据
        setSlotFloorPriceStatisticData(basePriceInfo, slKey, statisticDataMap);
        // 设置配置+广告位维度数据
        setOrientSlotFloorPriceStatisticData(basePriceInfo, adPkSlKey, statisticDataMap);
        // redis 读取 算法参数
        BasePriceParams basePriceParams = null;
        try {
            basePriceParams = basePriceParamsCache.get(RedisKeyUtil.getBasePriceParamsKey());
        } catch (ExecutionException e) {
            logger.error("basePriceParams load error !", e);
        }
        logger.info("每隔一分钟重载一次 放弃比例 开始");
        //todo 测试删除
        logger.info("all basePriceInfos= {}",JSONObject.toJSONString(basePriceInfo));
        // 调用 算法 推荐
        BasePriceResult basePriceResult = BasePriceControl.basePriceControl(basePriceInfo,oldBasePriceResult,basePriceParams);
        //todo 测试删除
        logger.info("上次对象返回结果：{}", JSONObject.toJSONString(oldBasePriceResult));
        logger.info("本次次对象返回结果：{}", JSONObject.toJSONString(basePriceResult));
        logger.info("每隔一分钟重载一次 放弃比例 结束");
        return basePriceResult;
    }

    /**
     * 设置配置+广告位维度
     *
     * @param basePriceInfo
     * @param adPkKey
     * @param statisticDataMap
     */
    private void setOrientSlotFloorPriceStatisticData(BasePriceInfo basePriceInfo, String adPkKey, Map<String, FloorPriceStatisticData> statisticDataMap) {
        FloorPriceStatisticData adPkSlData = statisticDataMap.get(adPkKey);
        if (adPkSlData != null) {
            basePriceInfo.setOrientSlotCostToday(adPkSlData.getCostToday());
            basePriceInfo.setOrientSlotLaunchToday(adPkSlData.getLaunchToday());
            basePriceInfo.setOrientSlotLevel5PriceLaunchToday(adPkSlData.getLevel5PriceLaunchToday());
            basePriceInfo.setOrientSlotLevel10PriceLaunchToday(adPkSlData.getLevel10PriceLaunchToday());
            basePriceInfo.setOrientSlotLevel15PriceLaunchToday(adPkSlData.getLevel15PriceLaunchToday());
            basePriceInfo.setOrientSlotConvertToday(adPkSlData.getConvertToday());
            basePriceInfo.setOrientSlotBasePriceCostToday(adPkSlData.getPriceCostToday());
            basePriceInfo.setOrientSlotLaunchG3day(adPkSlData.getLaunchG3day());
        }
    }

    /**
     * 设置广告位维度
     *
     * @param basePriceInfo
     * @param slKey
     * @param statisticDataMap
     */
    private void setSlotFloorPriceStatisticData(BasePriceInfo basePriceInfo, String slKey, Map<String, FloorPriceStatisticData> statisticDataMap) {
        FloorPriceStatisticData slData = statisticDataMap.get(slKey);
        if (slData != null) {
            basePriceInfo.setSlotCostToday(slData.getCostToday());
            basePriceInfo.setSlotLaunchToday(slData.getLaunchToday());
            basePriceInfo.setSlotLevel5PriceLaunchToday(slData.getLevel5PriceLaunchToday());
            basePriceInfo.setSlotLevel10PriceLaunchToday(slData.getLevel10PriceLaunchToday());
            basePriceInfo.setSlotLevel15PriceLaunchToday(slData.getLevel15PriceLaunchToday());
            basePriceInfo.setSlotBasePriceCostToday(slData.getPriceCostToday());
        }
    }

    /**
     * 设置配置维度
     *
     * @param basePriceInfo
     * @param adPkKey
     * @param statisticDataMap
     */
    private void setOrientFloorPriceStatisticData(BasePriceInfo basePriceInfo, String adPkKey, Map<String, FloorPriceStatisticData> statisticDataMap) {
        FloorPriceStatisticData adPkData = statisticDataMap.get(adPkKey);
        if (adPkData != null) {
            basePriceInfo.setOrientCostToday(adPkData.getCostToday());
            basePriceInfo.setOrientLaunchToday(adPkData.getLaunchToday());
            basePriceInfo.setOrientLevel5PriceLaunchToday(adPkData.getLevel5PriceLaunchToday());
            basePriceInfo.setOrientLevel10PriceLaunchToday(adPkData.getLevel10PriceLaunchToday());
            basePriceInfo.setOrientLevel15PriceLaunchToday(adPkData.getLevel15PriceLaunchToday());
            basePriceInfo.setOrientConvertToday(adPkData.getConvertToday());
            basePriceInfo.setOrientBasePriceCostToday(adPkData.getPriceCostToday());
        }
    }

}
