package cn.com.duiba.nezha.engine.biz.service.advert.rerank;

import cn.com.duiba.nezha.engine.biz.domain.advert.OrientationPackage;
import cn.com.duiba.nezha.engine.biz.service.CacheService;
import cn.com.duiba.nezha.engine.common.utils.MapUtils;
import cn.com.duiba.nezha.engine.common.utils.RedisKeyUtil;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import com.google.common.collect.Lists;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.RandomUtils;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static java.util.Comparator.comparing;
import static java.util.stream.Collectors.*;


/**
 * Created by pc on 2016/10/18.
 */
@Service
@RefreshScope
public class AdvertReRankService extends CacheService {

    @Value("${supportInfo}")
    private String supportInfo;


    @SuppressWarnings("squid:S3776")
    public List<OrientationPackage> reRank(List<OrientationPackage> orientationPackages) {

        try {
            DBTimeProfile.enter("reRank");

            // 如果配置列表为空,则没有排序的必要
            if (orientationPackages.isEmpty()) {
                return orientationPackages;
            }
            // 如果配置列表只有一个
            if (orientationPackages.size() == 1) {
                orientationPackages.get(0).setRank(0);
                orientationPackages.get(0).setReRank(0);
            }

            // 老的排序列表
            List<OrientationPackage> oldList = this.sortAndRank(orientationPackages);

            // 获取扶持的次数和扶持信息
            Map<Long, Long> advertSupportCount = new HashMap<>();
            Map<Integer, Map<Long, SupportInfo>> supportInfoMap = new HashMap<>();
            this.getSupportInfoAndCount(advertSupportCount, supportInfoMap);

            // 新的扶持列表
            List<OrientationPackage> newList = this.calculateNewRankScoreAndSetRank(orientationPackages, advertSupportCount, supportInfoMap);


            OrientationPackage oldFirst = oldList.get(0);
            OrientationPackage newFirst = newList.get(0);
            if (oldFirst.equals(newFirst)) { // 如果新老排名的第一名没有发生变化.扶持不生效
                return oldList;
            }
            // 程序走到这里则说明第一名被改变了.
            Map<Long, SupportInfo> accountSupportMap = supportInfoMap.getOrDefault(2, new HashMap<>());

            // 如果老的广告是在账号扶持中.
            if (accountSupportMap.keySet().contains(oldFirst.getAccountId())) {
                SupportInfo supportInfo = accountSupportMap.get(oldFirst.getAccountId());


                int i = RandomUtils.nextInt(1, 101);
                if (i > supportInfo.getRate()) {// 看第一名是否命中概率
                    return oldList;
                }

                // 程序走到这里说明命中了

                double oldArpu = oldFirst.getFinalFee() * oldFirst.getCtr();
                double newArpu = newFirst.getFinalFee() * newFirst.getCtr();
                if ((oldArpu - newArpu) / oldArpu > supportInfo.getTolerance()) {
                    return oldList;
                }

                return newList;


            } else {// 如果不在账号扶持中.直接返回新
                return newList;
            }

        } finally {
            DBTimeProfile.release();
        }
    }

    private List<OrientationPackage> sortAndRank(List<OrientationPackage> orientationPackages) {
        List<OrientationPackage> collect = orientationPackages.stream().sorted(comparing(OrientationPackage::getRankScore).reversed()).collect(Collectors.toList());
        int size = collect.size();
        for (int r = 0; r < size; r++) {
            OrientationPackage orientationPackage = collect.get(r);
            orientationPackage.setRank(r);
            orientationPackage.setSimpleSupportType("0");

        }
        return collect;
    }

    // 用来验证 supportInfo 是否合法，勿删
    public static void main(String[] args) {
        String supportInfo = "47380:1:100:1:1.3:150000:2019-05-25;47291:1:100:1:1.3:150000:2019-05-25;45756:1:100:1:1.3:150000:2019-05-25;47412:1:100:1:1.5:150000:2019-05-25;46310:1:100:1:1.3:150000:2019-05-25;46301:1:100:1:1.3:150000:2019-05-25;46039:1:100:1:1.3:150000:2019-05-25;46038:1:100:1:1.3:150000:2019-05-25;47668:1:100:1:1.3:150000:2019-05-25;47585:1:100:1:1.3:150000:2019-05-25;48251:1:100:1:1.3:150000:2019-05-25;48252:1:100:1:1.3:150000:2019-05-25;48253:1:100:1:1.3:150000:2019-05-25;48254:1:100:1:1.3:150000:2019-05-25;46131:1:100:1:1.3:150000:2019-05-25;46599:1:100:1:1.3:150000:2019-05-25;47931:1:100:1:1.3:150000:2019-05-25;47968:1:100:1:0.8:150000:2019-05-25;46996:1:100:1:1.3:150000:2019-05-25;48563:1:100:1:1.3:150000:2019-05-25;47974:1:100:1:0.9:150000:2019-05-30;47237:1:100:1:0.9:150000:2019-05-30;44916:1:100:1:0.9:150000:2019-05-30;49147:1:100:1:1.3:100000:2019-05-30;15809:2:5:0.05:0.938:-1:2019-10-10;49344:1:100:1:1.5:100000:2019-05-30;47403:1:100:1:1.5:100000:2019-05-30;47361:1:100:1:1.5:100000:2019-05-30";
        Stream.of(supportInfo.split(";"))
                .map(SupportInfo::parse)
                .filter(si -> DateTime.now().withTimeAtStartOfDay().toDate().getTime() <= si.getEndDate().getTime())
                .collect(groupingBy(SupportInfo::getType,
                        collectingAndThen(toList(),
                                list -> list.stream().collect(toMap(SupportInfo::getId, Function.identity())))));
    }

    private void getSupportInfoAndCount(Map<Long, Long> advertCount, Map<Integer, Map<Long, SupportInfo>> supportInfoMap) {
        if (StringUtils.isBlank(supportInfo)) {
            return;
        }

        supportInfoMap.putAll(Stream.of(supportInfo.split(";"))
                .map(SupportInfo::parse)
                .filter(si -> this.nowIsBefore(si.getEndDate()))
                .collect(groupingBy(SupportInfo::getType,
                        collectingAndThen(toList(),
                                list -> list.stream().collect(toMap(SupportInfo::getId, Function.identity()))))));

        if (supportInfoMap.isEmpty()) {
            return;
        }

        Map<Long, SupportInfo> advertSupportInfoMap = supportInfoMap.getOrDefault(1, new HashMap<>());
        if (advertSupportInfoMap.isEmpty()) {
            return;
        }


        Map<Long, String> advertKeyMap = advertSupportInfoMap.keySet().stream().collect(toMap(Function.identity(), RedisKeyUtil::getSupportCount));
        List<String> keys = Lists.newArrayList(advertKeyMap.values());
        List<String> record = nezhaStringRedisTemplate.opsForValue().multiGet(keys);


        Map<String, Long> map = new HashMap<>();
        for (int index = 0; index < record.size(); index++) {

            String key = keys.get(index);
            String value = record.get(index);

            if (StringUtils.isNotBlank(value)) {
                map.put(key, Long.parseLong(value));
            } else {
                map.put(key, 0L);
            }

        }
        advertCount.putAll(MapUtils.translate(advertKeyMap, map));

    }

    private Boolean nowIsBefore(Date date) {
        return DateTime.now().withTimeAtStartOfDay().toDate().getTime() <= date.getTime();
    }

    private List<OrientationPackage> calculateNewRankScoreAndSetRank(List<OrientationPackage> orientationPackages, Map<Long, Long> advertSupportCount, Map<Integer, Map<Long, SupportInfo>> supportInfoMap) {

        Map<Long, SupportInfo> advertSupportInfoMap = supportInfoMap.getOrDefault(1, new HashMap<>());
        Map<Long, SupportInfo> accountSupportInfoMap = supportInfoMap.getOrDefault(2, new HashMap<>());

        // 计算新的rankScore
        orientationPackages.forEach(orientationPackage -> {
            Long advertId = orientationPackage.getAdvertId();
            Long accountId = orientationPackage.getAccountId();

            // 广告纬度扶持
            Optional.ofNullable(advertSupportInfoMap.get(advertId)).ifPresent(advertSupportInfo -> {
                Long count = advertSupportCount.getOrDefault(advertId, 0L);
                if (count == -1 || count < advertSupportInfo.getCount()) {
                    orientationPackage.setRankScore(orientationPackage.getRankScore() * advertSupportInfo.getWeight());
                    orientationPackage.setSimpleSupportType(orientationPackage.getSimpleSupportType() + "1");
                    orientationPackage.setSupportInfoSupportWeight(advertSupportInfo.getWeight());
                }
            });

            // 账号纬度扶持
            Optional.ofNullable(accountSupportInfoMap.get(accountId)).ifPresent(accountSupportInfo -> {
                orientationPackage.setRankScore(orientationPackage.getRankScore() * accountSupportInfo.getWeight());
                orientationPackage.setSimpleSupportType(orientationPackage.getSimpleSupportType() + "2");
                orientationPackage.setSupportInfoSupportWeight(accountSupportInfo.getWeight());
            });

        });

        // 用新的rankScore进行排序
        List<OrientationPackage> newList = orientationPackages.stream().sorted(comparing(OrientationPackage::getRankScore).reversed()).collect(toList());
        int size = newList.size();

        // 计算新的reRank
        for (int r = 0; r < size; r++) {
            OrientationPackage orientationPackage = newList.get(r);
            orientationPackage.setReRank(r);
        }

        return newList;

    }

    public static class SupportInfo {
        private Long id;
        private Integer type;
        private Integer rate;
        private Double tolerance;
        private Double weight;
        private Long count;
        private Date endDate;

        private SupportInfo(Builder builder) {
            id = builder.id;
            type = builder.type;
            rate = builder.rate;
            tolerance = builder.tolerance;
            weight = builder.weight;
            count = builder.count;
            endDate = builder.endDate;
        }

        public static Builder newBuilder() {
            return new Builder();
        }

        public static SupportInfo parse(String supportInfo) {
            String[] split = supportInfo.split(":");
            long id = Long.parseLong(split[0]);
            int type = Integer.parseInt(split[1]);
            int rate = Integer.parseInt(split[2]);
            double tolerance = Double.parseDouble(split[3]);
            double weight = Double.parseDouble(split[4]);
            long count = Long.parseLong(split[5]);
            Date date = DateTime.parse(split[6]).toDate();


            return SupportInfo.newBuilder().id(id).type(type).rate(rate).tolerance(tolerance).weight(weight).count(count).endDate(date).build();
        }

        public Long getId() {
            return id;
        }

        public Integer getType() {
            return type;
        }

        public Integer getRate() {
            return rate;
        }

        public Double getWeight() {
            return weight;
        }

        public Long getCount() {
            return count;
        }

        public Date getEndDate() {
            return endDate;
        }

        public Double getTolerance() {
            return tolerance;
        }

        public static final class Builder {
            private Long id;
            private Integer type;
            private Integer rate;
            private Double tolerance;
            private Double weight;
            private Long count;
            private Date endDate;

            private Builder() {
            }

            public Builder id(Long val) {
                id = val;
                return this;
            }

            public Builder type(Integer val) {
                type = val;
                return this;
            }

            public Builder rate(Integer val) {
                rate = val;
                return this;
            }

            public Builder tolerance(Double val) {
                tolerance = val;
                return this;
            }

            public Builder weight(Double val) {
                weight = val;
                return this;
            }

            public Builder count(Long val) {
                count = val;
                return this;
            }

            public Builder endDate(Date val) {
                endDate = val;
                return this;
            }

            public SupportInfo build() {
                return new SupportInfo(this);
            }
        }
    }

}
