package cn.com.duiba.nezha.engine.biz.bo.advert.impl;

import cn.com.duiba.nezha.alg.alg.cpcautobidding.CpcAutoBidding;
import cn.com.duiba.nezha.alg.alg.cpcautobidding.CpcAutoBiddingInfo;
import cn.com.duiba.nezha.alg.alg.cpcautobidding.CpcAutoBiddingParams;
import cn.com.duiba.nezha.alg.alg.cpcautobidding.CpcAutoBiddingResult;
import cn.com.duiba.nezha.engine.biz.bo.advert.AdvertCPCAutoService;
import cn.com.duiba.nezha.engine.biz.domain.advert.OrientationPackage;
import cn.com.duiba.nezha.engine.biz.service.CacheService;
import cn.com.duiba.nezha.engine.biz.vo.advert.CpcAutoBiddingDataVo;
import cn.com.duiba.nezha.engine.common.utils.MapUtils;
import cn.com.duiba.nezha.engine.common.utils.RedisKeyUtil;
import cn.com.duiba.nezha.engine.common.utils.StringRedisHelper;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import com.alibaba.fastjson.JSONObject;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.TimeUnit;

@Service
public class AdvertCPCAutoServiceImpl  extends CacheService implements AdvertCPCAutoService {

    // 优化目标-落地页转化
    private static final String LAND_TYPE = "0";


    // 获取实时数据
    private LoadingCache<String, CpcAutoBiddingDataVo> CpcAutoBiddingCache = CacheBuilder.newBuilder()
            .expireAfterWrite(2, TimeUnit.MINUTES)
            .build(new CacheLoader<String, CpcAutoBiddingDataVo>() {
                @Override
                public CpcAutoBiddingDataVo load(String key) throws Exception {
                    throw new UnsupportedOperationException("not support single query");
                }

                @Override
                public Map<String, CpcAutoBiddingDataVo> loadAll(Iterable<? extends String> keys){
                    return StringRedisHelper.of(nezhaStringRedisTemplate).valueMultiGet
                            (keys, CpcAutoBiddingDataVo.class, CpcAutoBiddingDataVo::new);
                }
            });

    // 获取上次的配置+媒体维度的调价因子,60分钟后失效
    private LoadingCache<String, Optional<Double>> LastCpcAutoBiddingFactorCache = CacheBuilder.newBuilder()
            .expireAfterWrite(60, TimeUnit.MINUTES)
            .build(new CacheLoader<String, Optional<Double>>() {
                @Override
                public Optional<Double> load(String key) throws Exception {
                    return Optional.empty();
                }
            });


    // 获取算法自己存放的缓存
    private LoadingCache<String, CpcAutoBiddingParams> CpcAutoParamsCache = CacheBuilder.newBuilder()
            .expireAfterWrite(2, TimeUnit.HOURS)
            .build(new CacheLoader<String, CpcAutoBiddingParams>() {
                @Override
                public CpcAutoBiddingParams load(String key) throws Exception {

                    String params = nezhaStringRedisTemplate.opsForValue().get(key);

                    if(StringUtils.isBlank(params)){
                        return new CpcAutoBiddingParams();
                    }

                    JSONObject jsonObject = JSONObject.parseObject(params);

                    return JSONObject.toJavaObject(jsonObject, CpcAutoBiddingParams.class);
                }
            });



    @Override
    public List<OrientationPackage> getAutoBidding(List<OrientationPackage> newCPCList, Long appId) {

        try {
            DBTimeProfile.enter("getAutoBidding");

            // 构建查询key
            List<AutoQuery> allKeys = Lists.newArrayList();

            newCPCList.forEach(data -> {
                AutoQuery advertQuery = this.createAdvertQuery(data);

                AutoQuery advertAppQuery = this.createAdvertAppQuery(data, appId);

                allKeys.add(advertQuery);
                allKeys.add(advertAppQuery);

            });

            // 获取实时数据
            Map<String, CpcAutoBiddingInfo> dataFromCacheMap = this.getDataFromCache(allKeys, newCPCList, appId);

            if(dataFromCacheMap.isEmpty()){
                return newCPCList;
            }

            // 获取算法放入缓存的数据
            CpcAutoBiddingParams cpcAutoBiddingParams = CpcAutoParamsCache.get(RedisKeyUtil.getCpcAutoBiddingParamsKey());


            for (OrientationPackage data : newCPCList) {

                String key = data.getAdvertId() + "_" + data.getId() + "_" + appId;

                CpcAutoBiddingInfo cbInfo = dataFromCacheMap.get(key);

                Double lastCpcAutoBiddingFactor = LastCpcAutoBiddingFactorCache.get(key).orElse(null);

                CpcAutoBiddingResult autoBiddingResult = CpcAutoBidding.cpcAutoBiddingFactor(cbInfo, lastCpcAutoBiddingFactor, cpcAutoBiddingParams);

                // 调价因子
                Double cpcAutoBiddingFactor = Optional.ofNullable(autoBiddingResult.getCpcAutoBiddingFactor()).orElse(1d) ;

                //计算调价后的rankScore
                Double finalRankScore = data.getRankScore() * cpcAutoBiddingFactor;

                Double finalFee = data.getFinalFee() * cpcAutoBiddingFactor;

                data.setRankScore(finalRankScore);
                data.setCpcAutoBiddingFactor(cpcAutoBiddingFactor);
                data.setFinalFee(finalFee.longValue());

                //将这次的调价因子放到缓存中
                LastCpcAutoBiddingFactorCache.put(key, Optional.of(cpcAutoBiddingFactor));

            }

            return newCPCList;
        } catch (Exception e) {
            logger.error("getAutoBidding happened error:{}", e);
            return newCPCList;
        } finally {
            DBTimeProfile.release();
        }

    }


    /**
     * 获取实时数据
     * @param allKeys
     * @param newCPCList
     * @param appId2
     * @return
     */
    private Map<String, CpcAutoBiddingInfo> getDataFromCache(List<AutoQuery> allKeys, List<OrientationPackage> newCPCList, Long appId2){

        try{

        // 获取当天时间
        String dayFormat = LocalDate.now().format(DAY_FORMATTER);

        Map<AutoQuery, String> dayDataKeyMap = new HashMap<>();

        // 获取当前时刻
        String timeFormat = LocalDateTime.now().format(HOUR_FORMATTER);

        Map<AutoQuery, String> timeDataKeyMap = new HashMap<>();

        // 组装key
        allKeys.forEach(data -> {
            Long advertId = data.getAdvertId();
            Long pckgId = data.getPackageId();
            Long appId = data.getAppId();

            String key1 = RedisKeyUtil.getCPCAutoKey(advertId, pckgId, appId, dayFormat);

            String key2 = RedisKeyUtil.getCPCAutoKey(advertId, pckgId, appId, timeFormat);

            dayDataKeyMap.put(data, key1);

            timeDataKeyMap.put(data, key2);
        });

        // 获取当天数据
        Map<String, CpcAutoBiddingDataVo> cpcAutoDayDataMap = CpcAutoBiddingCache.getAll(dayDataKeyMap.values());

        // 获取当前时刻数据
        Map<String, CpcAutoBiddingDataVo> cpcAutoTimeDataMap = CpcAutoBiddingCache.getAll(timeDataKeyMap.values());

        Map<AutoQuery, CpcAutoBiddingDataVo> dayDataMap = MapUtils.translate(dayDataKeyMap, cpcAutoDayDataMap);

        Map<AutoQuery, CpcAutoBiddingDataVo> timeDataMap = MapUtils.translate(timeDataKeyMap, cpcAutoTimeDataMap);

        //组装数据
        return buildData(newCPCList, appId2, dayDataMap, timeDataMap);

        }catch (Exception e){
            logger.error("getAutoBidding happened error:{}", e);

            return Maps.newHashMap();
        }

    }


    /**
     * 构建数据,
     * @param newCPCList
     * @param appId
     * @param dayDataMap 今日数据
     * @param timeDataMap 时刻数据
     * @return
     */
    private Map<String, CpcAutoBiddingInfo> buildData(List<OrientationPackage> newCPCList, Long appId, Map<AutoQuery, CpcAutoBiddingDataVo> dayDataMap, Map<AutoQuery, CpcAutoBiddingDataVo> timeDataMap){

        Map<String, CpcAutoBiddingInfo> resultMap = new HashMap<>();

        newCPCList.forEach(data ->{

            Long advertId = data.getAdvertId();

            Long orientId = data.getId();

            String key = data.getAdvertId() + "_" + data.getId() + "_" + appId;

            CpcAutoBiddingInfo cbInfo = new CpcAutoBiddingInfo();

            cbInfo.setAdvertId(advertId);
            cbInfo.setOrientId(orientId);
            cbInfo.setAppId(appId);
            cbInfo.setConvertType(data.getCvrType());
            cbInfo.setAutoBiddingType(data.getAutoBiddingType());
            cbInfo.setTarget(data.getConvertCost());

            AutoQuery advertQuery = this.createAdvertQuery(data);

            AutoQuery advertAppQuery = this.createAdvertAppQuery(data, appId);


            // 优化目标
            String cvrType = data.getCvrType().toString();

            // 媒体
            // 每日维度数据
            CpcAutoBiddingDataVo dayAdvertAppVo = dayDataMap.getOrDefault(advertAppQuery, new CpcAutoBiddingDataVo());

            Map<String, Long> dayBackendCntAdvertAppMap = dayAdvertAppVo.getBackendCntMap();

            // 小时维度数据
            CpcAutoBiddingDataVo timeAdvertAppVo = timeDataMap.getOrDefault(advertAppQuery, new CpcAutoBiddingDataVo());

            Map<String, Long> timeBackendCntAdvertAppMap = timeAdvertAppVo.getBackendCntMap();

            // 每日
            cbInfo.setOrientAppCostToday(dayAdvertAppVo.getChargeFees());
            cbInfo.setOrientAppConvertToday(LAND_TYPE.equals(cvrType) ? dayAdvertAppVo.getActClickCnt() : dayBackendCntAdvertAppMap.getOrDefault(cvrType, 0L));
            cbInfo.setOrientAppBiddingCntToday(dayAdvertAppVo.getBidCnt());
            cbInfo.setOrientAppRankScore10(dayAdvertAppVo.getScore10());
            cbInfo.setOrientAppRankScore9(dayAdvertAppVo.getScore9());
            cbInfo.setOrientAppRankScore8(dayAdvertAppVo.getScore8());

            // -小时
            cbInfo.setOrientAppCostHour(timeAdvertAppVo.getChargeFees());
            cbInfo.setOrientAppConvertHour(LAND_TYPE.equals(cvrType) ? timeAdvertAppVo.getActClickCnt() : timeBackendCntAdvertAppMap.getOrDefault(cvrType, 0L));
            cbInfo.setOrientAppBiddingCntHour(timeAdvertAppVo.getBidCnt());
            cbInfo.setOrientAppHourRankScore10(timeAdvertAppVo.getScore10());
            cbInfo.setOrientAppHourRankScore9(timeAdvertAppVo.getScore9());
            cbInfo.setOrientAppHourRankScore8(timeAdvertAppVo.getScore8());


            // 配置
            // 每日数据维度
            CpcAutoBiddingDataVo dayAdvertVo = dayDataMap.getOrDefault(advertQuery, new CpcAutoBiddingDataVo());

            Map<String, Long> dayBackendCntAdvertMap = dayAdvertVo.getBackendCntMap();

            // 小时维度数据
            CpcAutoBiddingDataVo timeAdvertVo = timeDataMap.getOrDefault(advertQuery, new CpcAutoBiddingDataVo());

            Map<String, Long> timeBackendCntAdvertMap = timeAdvertVo.getBackendCntMap();


            // -每日
            cbInfo.setOrientCostToday(dayAdvertVo.getChargeFees());
            cbInfo.setOrientConvertToday(LAND_TYPE.equals(cvrType) ? dayAdvertVo.getActClickCnt() : dayBackendCntAdvertMap.getOrDefault(cvrType, 0L));
            cbInfo.setOrientBiddingCntToday(dayAdvertVo.getBidCnt());
            cbInfo.setOrientRankScore10(dayAdvertVo.getScore10());
            cbInfo.setOrientRankScore9(dayAdvertVo.getScore9());
            cbInfo.setOrientRankScore8(dayAdvertVo.getScore8());

            // -小时
            cbInfo.setOrientCostHour(timeAdvertVo.getChargeFees());
            cbInfo.setOrientConvertHour(LAND_TYPE.equals(cvrType) ? timeAdvertVo.getActClickCnt() : timeBackendCntAdvertMap.getOrDefault(cvrType, 0L));
            cbInfo.setOrientBiddingCntHour(timeAdvertVo.getBidCnt());
            cbInfo.setOrientHourRankScore10(timeAdvertVo.getScore10());
            cbInfo.setOrientHourRankScore9(timeAdvertVo.getScore9());
            cbInfo.setOrientHourRankScore8(timeAdvertVo.getScore8());

            resultMap.put(key, cbInfo);

        });

        return resultMap;

    }

    /**
     * 构建key,广告+配置
     * @param data
     * @return
     */
    private AutoQuery createAdvertQuery(OrientationPackage data){

        AutoQuery advertQuery = new AutoQuery();

        advertQuery.setAdvertId(data.getAdvertId());
        advertQuery.setPackageId(data.getId());

        return advertQuery;

    }

    /**
     * 构建key,广告+配置+媒体
     * @param data
     * @param appId
     * @return
     */
    private AutoQuery createAdvertAppQuery(OrientationPackage data, Long appId){

        AutoQuery advertAppQuery = new AutoQuery();

        advertAppQuery.setAdvertId(data.getAdvertId());
        advertAppQuery.setPackageId(data.getId());
        advertAppQuery.setAppId(appId);

        return advertAppQuery;

    }

    class AutoQuery {
        private Long advertId;
        private Long packageId;
        private Long appId;

        public Long getAdvertId() {
            return advertId;
        }

        public void setAdvertId(Long advertId) {
            this.advertId = advertId;
        }

        public Long getPackageId() {
            return packageId;
        }

        public void setPackageId(Long packageId) {
            this.packageId = packageId;
        }


        public Long getAppId() {
            return appId;
        }

        public void setAppId(Long appId) {
            this.appId = appId;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof AdvertCPCAutoServiceImpl.AutoQuery)) {
                return false;
            }
            AdvertCPCAutoServiceImpl.AutoQuery that = (AdvertCPCAutoServiceImpl.AutoQuery) o;
            return Objects.equals(advertId, that.advertId) &&
                    Objects.equals(packageId, that.packageId) &&
                    Objects.equals(appId, that.appId);
        }

        @Override
        public int hashCode() {
            return Objects.hash(advertId, packageId, appId);
        }
    }
}
