package cn.com.duiba.nezha.engine.biz.service.advert.ctr.impl;

import cn.com.duiba.nezha.engine.api.support.RecommendEngineException;
import cn.com.duiba.nezha.engine.biz.domain.advert.Advert;
import cn.com.duiba.nezha.engine.biz.domain.mergeData.MergeData;
import cn.com.duiba.nezha.engine.biz.enums.MergeDataType;
import cn.com.duiba.nezha.engine.biz.service.CacheService;
import cn.com.duiba.nezha.engine.biz.service.advert.ctr.AdvertMergeStatService;
import cn.com.duiba.nezha.engine.common.utils.AssertUtil;
import cn.com.duiba.nezha.engine.common.utils.MapUtils;
import cn.com.duiba.nezha.engine.common.utils.RedisKeyUtil;
import cn.com.duiba.nezha.engine.common.utils.StringRedisHelper;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.StreamSupport;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;

@Service
public class AdvertMergeStatServiceImpl extends CacheService implements AdvertMergeStatService {

    private static final Logger LOGGER = LoggerFactory.getLogger(AdvertMergeStatService.class);


    //发券次数无关统计数据查询时发券次数
    private static final Long TIMES_GLOBAL = 1L;

    private LoadingCache<AdvertMergeDataQuery, MergeData> advertMergeStatCache = CacheBuilder.newBuilder()
            .expireAfterWrite(2, TimeUnit.MINUTES)
            .recordStats()
            .build(new CacheLoader<AdvertMergeDataQuery, MergeData>() {
                @Override
                public MergeData load(AdvertMergeDataQuery key) throws Exception {
                    throw new IllegalAccessException("not suppose single query");
                }

                @Override
                public Map<AdvertMergeDataQuery, MergeData> loadAll(Iterable<? extends AdvertMergeDataQuery> queries) {
                    Map<AdvertMergeDataQuery, String> queryKeyMap = StreamSupport
                            .stream(queries.spliterator(), false)
                            .collect(toMap(Function.identity(),
                                    query -> RedisKeyUtil.advertMergeStatKey(
                                            query.getAppId(),
                                            query.getAdvertId(),
                                            query.getMaterialId(),
                                            query.getTimes())));

                    return getAdvertMergeDataQueryMergeDataMap(queryKeyMap);

                }
            });

    @NotNull
    private Map<AdvertMergeDataQuery, MergeData> getAdvertMergeDataQueryMergeDataMap(Map<AdvertMergeDataQuery, String> queryKeyMap) {
        Map<String, MergeData> keyDataMap = StringRedisHelper
                .of(nezhaStringRedisTemplate)
                .valueMultiGet(queryKeyMap.values(),
                        MergeData.class,
                        MergeData::new);

        Map<AdvertMergeDataQuery, MergeData> queryMergeDataMap = MapUtils.translate(queryKeyMap, keyDataMap);
        queryMergeDataMap.forEach((query, data) -> {
            data.setAdvertId(query.getAdvertId());
            data.setAppId(query.getAppId());
            data.setMaterialId(query.getMaterialId());
            data.setTimes(query.getTimes());
        });
        return queryMergeDataMap;
    }


    @Override
    public List<MergeData> getMaterialDataInApp(Long advertId, Long appId, List<Long> materialIds) {
        try {
            DBTimeProfile.enter("getMaterialDataInApp");
            if (AssertUtil.isEmpty(materialIds)) {
                return new ArrayList<>();
            }

            List<AdvertMergeDataQuery> queries = materialIds
                    .stream()
                    .map(materialId -> AdvertMergeDataQuery.newBuilder()
                            .advertId(advertId)
                            .materialId(materialId)
                            .appId(appId)
                            .times(TIMES_GLOBAL)
                            .build())
                    .collect(toList());

            return Lists.newArrayList(advertMergeStatCache.getAll(queries).values());
        } catch (Exception e) {
            throw new RecommendEngineException("get materialDataInApp error", e);
        } finally {
            DBTimeProfile.release();
        }
    }

    @Override
    public Map<MergeDataType, List<MergeData>> getAdvertAndAppData(Collection<Advert> adverts, Long appId) {

        try {
            List<AdvertMergeDataQuery> queries = new ArrayList<>(adverts.size() * 2);
            List<AdvertMergeDataQuery> appQueries = new ArrayList<>(adverts.size());
            List<AdvertMergeDataQuery> globalQueries = new ArrayList<>(adverts.size());

            adverts.forEach(advert -> {
                Long id = advert.getId();
                Long launchCountToUser = advert.getCurrentCount();
                AdvertMergeDataQuery advertAppStatQuery = AdvertMergeDataQuery.newBuilder()
                        .advertId(id)
                        .appId(appId)
                        .times(launchCountToUser)
                        .build();

                AdvertMergeDataQuery advertGlobalStatQuery = AdvertMergeDataQuery.newBuilder()
                        .advertId(id)
                        .times(launchCountToUser)
                        .build();

                queries.add(advertAppStatQuery);
                queries.add(advertGlobalStatQuery);
                appQueries.add(advertAppStatQuery);
                globalQueries.add(advertGlobalStatQuery);
            });

            ImmutableMap<AdvertMergeDataQuery, MergeData> all ;
            // 融合数据为了 激励广告部分 和其他广告做隔离 使用新key
            all = advertMergeStatCache.getAll(queries);
            Map<MergeDataType, List<MergeData>> map = new EnumMap<>(MergeDataType.class);
            List<MergeData> appStatDoList = appQueries.stream().map(all::get).collect(toList());
            List<MergeData> globalStatDoList = globalQueries.stream().map(all::get).collect(toList());
            map.put(MergeDataType.APP, appStatDoList);
            map.put(MergeDataType.GLOBAL, globalStatDoList);
            return map;
        } catch (Exception e) {
            logger.error("advertMergeData error :{}", e);
            throw new RecommendEngineException("get advertMergeData error", e);
        }
    }

    @Override
    public Map<String, CacheInfo> getCacheInfo() {
        Map<String, CacheInfo> cacheInfoMap = new HashMap<>();
        cacheInfoMap.put("advertMergeDataCache", CacheInfo.generate(advertMergeStatCache));
        return cacheInfoMap;
    }


}