package cn.com.duiba.oto.enums.cust;

import com.google.common.collect.Sets;
import lombok.AllArgsConstructor;
import lombok.Getter;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author xiaotian
 * @version 1.0
 * @date 2022-06-27 09:35
 */
@Getter
@AllArgsConstructor
public enum CustFromSourceEnum {

    /**
     * 投放客户-自建落地页
     */
    PUB_SELF_BUILD_FORM("01_01", CustActionTypeEnum.FORM, "投放客户-自建落地页-表单"),
    PUB_SELF_BUILD_LIVE("01_01", CustActionTypeEnum.LAUNCH_LIVE, "投放客户-自建落地页-直播"),
    /**
     * 投放客户-飞鱼落地页
     */
    PUB_FEIYU_CLUE_INVALID("01_02", CustActionTypeEnum.INVALID, "投放客户-飞鱼落地页-未知行为"),
    PUB_FEIYU_CLUE_FORM("01_02", CustActionTypeEnum.FORM, "投放客户-飞鱼落地页-填单客户"),
    PUB_FEIYU_CLUE_CONSULT("01_02", CustActionTypeEnum.CONSULT, "投放客户-飞鱼落地页-咨询获客"),
    /**
     * 投放客户-腾讯落地页
     */
    PUB_TENCENT_CLUE_INVALID("01_03", CustActionTypeEnum.INVALID, "投放客户-腾讯落地页-未知行为"),
    PUB_TENCENT_CLUE_FORM("01_03", CustActionTypeEnum.FORM, "投放客户-腾讯落地页-填单客户"),
    PUB_TENCENT_CLUE_CONSULT("01_03", CustActionTypeEnum.CONSULT, "投放客户-腾讯落地页-咨询获客"),
    PUB_TENCENT_CLUE_PHONE("01_03", CustActionTypeEnum.INTELLIGENT_PHONE, "投放客户-腾讯落地页-智能电话"),
    /**
     * 投放客户-百度落地页
     */
    PUB_BAIDU_CLUE_INVALID("01_04", CustActionTypeEnum.INVALID, "投放客户-百度落地页-未知行为"),
    PUB_BAIDU_CLUE_FORM("01_04", CustActionTypeEnum.FORM, "投放客户-百度落地页-填单客户"),
    /**
     * 投放客户-历史表单导入
     */
    PUB_HISTORY_IMPORT_FORM("01_05", CustActionTypeEnum.FORM, "投放客户-历史表单导入-填单客户"),


    /**
     * 销售自建-缘故客户
     */
    SELLER_OLD_CUST("02_01", CustActionTypeEnum.OLD, "销售自建-缘故客户-缘故客户"),
    /**
     * 销售自建-转介绍客户
     */
    SELLER_INTRODUCE_CUST("02_02", CustActionTypeEnum.INTRODUCE, "销售自建-转介绍客户-转介绍客户"),
    /**
     * 销售自建-转介绍客户
     */
    SELLER_INTRODUCE_CUST_FOR_CHECK("02_03", CustActionTypeEnum.INTRODUCE, "销售自建-转介绍客户-盘点客户新增"),

    /**
     * 其他来源-线上营销裂变
     */
    OTHER_ONLINE_MARKET_LIVE("03_01", CustActionTypeEnum.LIVE_FISSION, "其他来源-线上营销裂变-直播裂变"),
    OTHER_ONLINE_MARKET_ASSIST("03_01", CustActionTypeEnum.ASSIST, "其他来源-线上营销裂变-助手客户"),
    OTHER_ONLINE_MARKET_CORP_WX("03_01", CustActionTypeEnum.CORP_WX, "其他来源-线上营销裂变-企微"),

    /**
     * 其他来源-线下来源
     */
    OTHER_OFFLINE_COMMUNITY("03_02", CustActionTypeEnum.COMMUNITY, "其他来源-线下来源-线下社区"),

    /**
     * 其他来源-视频号
     */
    OTHER_OFFLINE_VIDEO("03_03", CustActionTypeEnum.VIDEO, "其他来源-视频号-咨询客户"),
    ;

    /**
     * 客户来源
     */
    private final String fromSource;

    /**
     * 客户行为
     */
    private final CustActionTypeEnum actionType;

    /**
     * 描述
     */
    private final String desc;


    /**
     * 包含所有枚举的map Map<客户来源, Map<客户行为，枚举>>
     */
    private static Map<String, Map<String, CustFromSourceEnum>> SOURCE_MAP = null;

    /**
     * 包含三方建站线索推送的客户来源
     */
    private static final Set<String> CLUE_SOURCE_SET = Sets.newHashSet(Arrays.stream(values()).filter(custFromSourceEnum -> {
        if ("01_02".equals(custFromSourceEnum.getFromSource())) {
            return true;
        }
        if ("01_03".equals(custFromSourceEnum.getFromSource())) {
            return true;
        }
        if ("01_04".equals(custFromSourceEnum.getFromSource())) {
            return true;
        }
        return false;
    }).map(CustFromSourceEnum::getStringByEnum).collect(Collectors.toSet()));

    /**
     * 销售自己上交的客户来源
     */
    private static final Set<String> SELLER_SOURCE_SET = Sets.newHashSet(getStringByEnum(SELLER_OLD_CUST), getStringByEnum(SELLER_INTRODUCE_CUST));

    static {
        Arrays.stream(values()).forEach(sourceEnum -> {
            if (MapUtils.isEmpty(SOURCE_MAP)) {
                SOURCE_MAP = new HashMap<>();
            }
            Map<String, CustFromSourceEnum> actionTypeMap = SOURCE_MAP.computeIfAbsent(sourceEnum.getFromSource(), k -> new HashMap<>());
            actionTypeMap.put(sourceEnum.getActionType().getType(), sourceEnum);
        });
    }

    /**
     * 根据 客户来源枚举 获取 客户来源字符串
     * @param custFromSourceEnum 客户来源枚举
     * @return 客户来源字符串
     */
    public static String getStringByEnum(CustFromSourceEnum custFromSourceEnum){
        return custFromSourceEnum.getFromSource() + "-" + custFromSourceEnum.getActionType().getType();
    }

    /**
     * 根据 客户来源字符串 获取 客户来源枚举
     * @param custFromSource 客户来源枚举
     * @return 客户来源字符串
     */
    public static CustFromSourceEnum getEnumByString(String custFromSource){
        if (StringUtils.isBlank(custFromSource)) {
            return null;
        }
        final String[] params = custFromSource.split("-");
        if (params.length != 2) {
            return null;
        }
        if (!SOURCE_MAP.containsKey(params[0])){
            return null;
        }
        Map<String, CustFromSourceEnum> actionTypeMap = SOURCE_MAP.get(params[0]);
        if (MapUtils.isEmpty(actionTypeMap) || !actionTypeMap.containsKey(params[1])) {
            return null;
        }
        return actionTypeMap.get(params[1]);
    }

    /**
     * 根据 客户来源字符串 获取 行为类型
     * @param custFromSource 客户来源枚举
     * @return 客户行为类型
     */
    public static String getActionTypeByString(String custFromSource){
        if (StringUtils.isBlank(custFromSource)) {
            return null;
        }
        final CustFromSourceEnum enumByString = getEnumByString(custFromSource);
        if (enumByString == null) {
            return null;
        }
        return enumByString.getActionType().getType();
    }

    /**
     * 是否线索推送处理类型
     * @param custFromSource 客户来源类型
     * @return 是否线索推送处理类型
     */
    public static boolean isCluePush(String custFromSource){
        if (StringUtils.isBlank(custFromSource)) {
            return false;
        }
        return CLUE_SOURCE_SET.contains(custFromSource);
    }

    /**
     * 是否线索推送处理类型
     * @param custFromSource 客户来源类型
     * @return 是否线索推送处理类型
     */
    public static boolean isSellerBuild(String custFromSource) {
        if (StringUtils.isBlank(custFromSource)) {
            return false;
        }
        return SELLER_SOURCE_SET.contains(custFromSource);
    }

    private static final Set<String> SELLER_SOURCE_CHECK_PHONE_SET = Sets.newHashSet(getStringByEnum(SELLER_OLD_CUST), getStringByEnum(SELLER_INTRODUCE_CUST), getStringByEnum(OTHER_OFFLINE_COMMUNITY));

    /**
     * 是否线索推送处理类型
     *
     * @param custFromSource 客户来源类型
     * @return 是否线索推送处理类型
     */
    public static boolean isPhoneRepeatBuild(String custFromSource) {
        if (StringUtils.isBlank(custFromSource)) {
            return false;
        }
        return SELLER_SOURCE_CHECK_PHONE_SET.contains(custFromSource);
    }
}
