package cn.com.duiba.oto.oto.service.api.remoteservice.seller;

import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import cn.com.duiba.oto.dto.oto.seller.SellerIdAndNameDTO;
import cn.com.duiba.oto.param.oto.OtoSellerDto;
import cn.com.duiba.oto.param.oto.seller.OtoAttendancePageParam;
import cn.com.duiba.oto.param.oto.seller.OtoSellerPageListParam;

import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * OTO-销售信息
 *
 * @author ZhangJun
 * @date 2022/1/7 5:59 下午
 */
@AdvancedFeignClient
public interface RemoteOtoSellerService {

    /**
     * 条件查询销售总数量
     *
     * @param param 查询参数
     * @return 销售总数
     */
    Long pageCount(OtoSellerPageListParam param);

    /**
     * 条件查询人管系统销售总数量
     *
     * @param param 查询参数
     * @return 销售总数
     */
    Long pageSellerCount(OtoSellerPageListParam param);

    /**
     * 分页查询久久保销售信息
     *
     * @param param 分页查询参数
     * @return 分页查询结果
     */
    List<OtoSellerDto> pageJJBSellerList(OtoSellerPageListParam param);

    /**
     * 分页查询销售信息
     *
     * @param param 分页查询参数
     * @return 分页查询结果
     */
    List<OtoSellerDto> pageList(OtoSellerPageListParam param);

    /**
     * 分页查询未离职销售信息
     *
     * @param param 分页查询参数
     * @return 分页查询结果
     */
    List<OtoSellerDto> getPageListByCondition(OtoSellerPageListParam param);

    /**
     * 查询总监下属销售数量
     *
     * @param parentId 总监的销售id
     * @return 销售数量
     */
    Long attendanceCount(Long parentId);

    /**
     * 查询总监下属销售id列表
     *
     * @param parentId 总监的销售id
     * @return 销售数量
     */
    List<Long> attendanceListByParentId(Long parentId);

    /**
     * 分页查询出勤销售
     *
     * @param param 分页查询参数
     * @return 分页查询结果
     */
    List<OtoSellerDto> attendanceList(OtoAttendancePageParam param);


    /**
     * 根据销售id查询销售信息
     *
     * @param id 销售id
     * @return 销售信息
     */
    OtoSellerDto findById(Long id);

    /**
     * 根据手机号查询销售信息
     *
     * @param phoneNumber 销售手机号
     * @return 销售信息
     */
    OtoSellerDto findByPhone(String phoneNumber);

    /**
     * 查询销售名称
     *
     * @param ids 销售id
     * @return key销售id value销售名称
     */
    List<OtoSellerDto> listByIds(List<Long> ids);

    /**
     * 根据姓名右模糊搜索销售
     *
     * @param sellerName 销售姓名
     * @return 销售列表
     */
    List<OtoSellerDto> searchBySellerName(String sellerName);

    /**
     * 根据姓名右模糊搜索未离职销售
     *
     * @param sellerName 销售姓名
     * @return 销售列表
     */
    List<OtoSellerDto> getSellerInfoByName(String sellerName);

    /**
     * 根据姓名批量查询销售
     */
    List<OtoSellerDto> listBySellerNames(List<String> sellerNames);

    /**
     * 新增销售
     *
     * @param dto 销售信息
     * @return 新增结果
     */
    boolean save(OtoSellerDto dto);

    /**
     * 新增销售 返回新增的销售ID
     * @param dto 销售信息
     * @return 新增结果
     */
    OtoSellerDto saveToId(OtoSellerDto dto);

    /**
     * 编辑销售
     *
     * @param dto 销售信息
     * @return 编辑结果
     */
    boolean update(OtoSellerDto dto);

    /**
     * 启用/禁用销售账号
     *
     * @param id           销售id
     * @param sellerStatus 销售账号状态
     *                     {@link cn.com.duiba.oto.enums.OtoSellerStatusEnum}
     * @return 启用/禁用结果
     */
    boolean switchSellerStatus(Long id, Integer sellerStatus);

    /**
     * 启用/禁用销售电话权限
     *
     * @param id             销售id
     * @param callPermission 销售拨打电话权限
     *                       {@link cn.com.duiba.oto.enums.OtoSellerCallPermissionEnum}
     * @return 启用/禁用结果
     */
    boolean switchCallPermission(Long id, Integer callPermission);

    /**
     * 设置销售工作状态
     *
     * @param id         销售id
     * @param workStatus 工作状态
     * @return 设置结果
     */
    boolean switchWorkStatus(Long id, Integer workStatus);

    /**
     * 更新最后登录时间
     *
     * @param id            销售id
     * @param lastLoginTime 最后登录时间
     * @return 更新结果
     */
    boolean updateLastLoginTime(Long id, Date lastLoginTime);

    /**
     * 批量加锁
     *
     * @param sellerIds   销售id
     * @param lockSeconds 加锁时间，秒
     * @return 加锁结果
     */
    Map<Long, Boolean> batchLock(List<Long> sellerIds, Integer lockSeconds);

    /**
     * 获取销售当日累计分配次数
     *
     * @param sellerId 销售ID
     * @return 分配次数
     */
    @Deprecated
    Long getDailyAssignTimes(Long sellerId);

    /**
     * 获取销售当日累计分配客户数量
     *
     * @param sellerId    销售ID
     * @param assignTypes 分配类型 {@link cn.com.duiba.oto.enums.cust.follow.OtoCustAssignTypeEnum}
     * @return 分配次数
     */
    Long sellerDailyAssignTimes(Long sellerId, List<Integer> assignTypes);

    /**
     * 查询所有销售
     * 管理后台专用
     */
    List<OtoSellerDto> listAll();

    /**
     * 更新销售所在公司
     * 管理后台刷数据专用
     *
     * @param id  id
     * @param companyId 公司id
     * @return 跟新结果
     */
    boolean updateSellerCompany(Long id, Long companyId);

    /**
     * 设置销售职位状态（在职/离职）
     *
     * @param sellerId         销售id
     * @param dimissionStatus 职位状态 {@link cn.com.duiba.oto.enums.OtoSellerDimissionStatusEnum}
     * @return 设置结果
     */
    boolean switchDimissionStatus(Long sellerId, Integer dimissionStatus);

    /**
     * 修改手机号
     * @param sellerId 销售id
     * @param newPhoneNum 手机号
     * @return 操作结果
     */
    boolean updatePhoneNum(Long sellerId, String newPhoneNum);

    /**
     * 姓名模糊查询该总监所管辖的销售id和姓名（总监可以搜索自己）
     * @param sid 登录销售的id
     * @param sellerName 查找的销售姓名
     * @return list
     */
    List<SellerIdAndNameDTO> findMySellerByFuzzyName(Long sid, String sellerName);

    /**
     * 根据销售id取得客户系数，再根据行为类型分组求和
     * @param sid 销售id
     * @return key：行为类型，value客户系数之和
     */
    Map<String, Integer> coefficientGroupByActionType(Long sid);

    /**
     * 根据销售id取得客户系数，再根据客户地区分组求和
     * @param sid 销售id
     * @return key：客户地区，value客户系数之和
     */
    Map<Integer, Integer> coefficientGroupByCustRegion(Long sid);
}
