package cn.com.duiba.oto.oto.service.api.remoteservice.cust;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import cn.com.duiba.oto.bean.OtoSellerCustBean;
import cn.com.duiba.oto.dto.oto.cust.CustAgeDto;
import cn.com.duiba.oto.dto.oto.cust.MobileCustListDto;
import cn.com.duiba.oto.dto.oto.cust.OtoCustAbandonDto;
import cn.com.duiba.oto.dto.oto.cust.OtoCustDto;
import cn.com.duiba.oto.dto.oto.cust.OtoCustomerDto;
import cn.com.duiba.oto.dto.oto.cust.OtoCustomerExtDto;
import cn.com.duiba.oto.dto.oto.cust.OtoCustomerPageDto;
import cn.com.duiba.oto.dto.oto.cust.OtoRecycleCustomerPageDto;
import cn.com.duiba.oto.dto.oto.cust.init.CustInitResultDTO;
import cn.com.duiba.oto.dto.oto.cust.init.OtoCustomerInitDto;
import cn.com.duiba.oto.dto.oto.interview.CustomerSimpleInfoDTO;
import cn.com.duiba.oto.param.oto.common.PhoneLocationDto;
import cn.com.duiba.oto.param.oto.cust.OtoCustCheckListParam;
import cn.com.duiba.oto.param.oto.cust.OtoCustCountParam;
import cn.com.duiba.oto.param.oto.cust.OtoCustPageListParam;
import cn.com.duiba.oto.param.oto.cust.OtoCustRecyclePageListParam;
import cn.com.duiba.oto.param.oto.cust.OtoCustomerEditParam;
import cn.com.duiba.oto.param.oto.cust.OtoCustomerSearchParam;
import cn.com.duiba.oto.param.oto.cust.OtoManualAddCustomerParam;
import cn.com.duiba.oto.param.oto.cust.OtoPhoneCustomerListParam;
import cn.com.duiba.oto.param.oto.cust.OtoRecycleCustSearchParam;
import cn.com.duiba.oto.util.PageResult;
import org.apache.commons.lang3.tuple.Pair;

import java.util.List;
import java.util.Set;

/**
 * @author xiaotian
 * @version 1.0
 * @date 2022-01-10 13:44
 * 客户 rpc service
 */
@AdvancedFeignClient
public interface RemoteOtoCustService {

    /**
     * 根据客户id查询客户信息
     * @param custId 客户id
     * @return 客户信息
     */
    OtoCustomerDto findById(Long custId);

    /**
     * 根据客户id批量查询客户信息
     * @param custIds 客户id
     * @return 客户信息
     */
    List<OtoCustomerDto> listByIds(List<Long> custIds);

    /**
     * 根据客户id批量查询客户部门字段
     * @param custIds 客户id
     * @return 客户信息
     */
    List<OtoCustomerDto> listPartByIds(List<Long> custIds);

    /**
     * 根据客户id批量查询客户信息（过滤掉当前代理人未跟进的）
     * @param custIds 客户id
     * @param otoSid 销售id
     * @return 客户信息
     */
    List<OtoCustomerDto> listByIdsAndSid(List<Long> custIds, Long otoSid);

    /**
     * 模糊搜索客户名
     * @param custName 客户名
     * @param otoSid 销售id
     * @return 客户名列表
     */
    List<String> fuzzyCustName(String custName, Long otoSid);

    /**
     * 根据客户搜索条件获取客户列表
     * @param param 客户搜索条件
     * @return 客户列表
     */
    PageResult<OtoCustomerPageDto> searchByParam(OtoCustomerSearchParam param);

    /**
     * 根据客户搜索条件获取待回收客户列表
     * @param param 客户搜索条件
     * @return 客户列表
     */
    PageResult<OtoRecycleCustomerPageDto> getRecycleCustByCondition(OtoRecycleCustSearchParam param);

    /**
     * 给客户打星标/取消星标
     *
     * @param custId 客户id
     * @param otoSid 销售id
     * @return 操作结果
     */
    boolean starCust(Long custId, Long otoSid) throws BizException;

    /**
     * 修改客户价值custPageList
     * @param custId 客户id
     * @param otoSid 销售id
     * @param custValue 客户价值
     * @see cn.com.duiba.oto.enums.cust.ext.OtoCustValueEnum
     * @return 标记结果
     */
    boolean assessCust(Long custId, Long otoSid, Integer custValue);

    /**
     * 批量修改客户价值
     *
     * @param custIds  客户id集合
     * @param custValue 客户价值
     * @return 修改结果
     */
    boolean batchAssessCust(List<Long> custIds, Integer custValue);

    /**
     * 给客户打标签
     *
     * @param custId 客户id
     * @param otoSid 销售id
     * @param tagIds 标签id数组
     * @return 客户打标签
     */
    boolean tagCust(Long custId, Long otoSid, List<Long> tagIds);

    /**
     * 给客户打标签-crm
     *
     * @param custId    客户id
     * @param otoSid    销售id
     * @param newTagIds 新标签id数组
     * @param oldTagIds 老标签id数组
     * @return 客户打标签
     */
    boolean tagCustCrm(Long custId, Long otoSid, List<Long> newTagIds, List<Long> oldTagIds);

    /**
     * 给客户打标签
     *
     * @param custIds 客户id
     * @param otoSid  销售id
     * @param tagIds  标签id数组
     * @return 客户打标签
     */
    boolean batchTagCust(List<Long> custIds, Long otoSid, List<Long> tagIds);

    /**
     * 编辑客户详情资料页
     * @param param 表单参数
     * @param otoSid 销售id
     * @return 客户详情资料页
     */
    boolean editCust(OtoCustomerEditParam param, Long otoSid) throws BizException;

    /**
     * 新增客户
     *
     * @param dto 客户信息
     * @return 新增结果
     */
    OtoCustomerDto addCust(OtoCustomerDto dto);

    /**
     * 更新客户信息
     *
     * @param dto 客户信息
     * @return 更新结果
     */
    boolean updateCust(OtoCustomerDto dto);

    /**
     * 根据手机号查询客户
     *
     * @param phoneNum 手机号
     * @return 客户信息
     */
    OtoCustomerDto findByPhone(String phoneNum);

    /**
     * 客户管理-客户分页列表
     *
     * @param param 查询参数
     * @return 分页查询结果
     */
    PageResult<OtoCustomerDto> custPageList(OtoCustPageListParam param);

    /**
     * 客户管理-回收池客户分页列表
     *
     * @param param 查询参数
     * @return 分页结果
     */
    PageResult<OtoCustomerDto> recycleCustPageList(OtoCustRecyclePageListParam param);

    /**
     * 瀑布流查询，按照id倒序
     *
     * @param lastMaxId 上次查询的最大id
     * @param pageSize 查询数量
     * @return 查询结果
     */
    List<OtoCustomerDto> findWithWaterFall(Long lastMaxId, Long stopId, Integer pageSize);

    /**
     * 根据主键批量更新归属地
     *
     * @param locationDtos 归属地bean
     * @return 更新成功数量
     */
    int batchUpdateLocation(List<PhoneLocationDto> locationDtos);

    /**
     * 根据主键批量更新客户年龄
     *
     * @return 更新成功数量
     */
    int batchUpdateCustAge(List<CustAgeDto> custAgeDtos);

    /**
     * 根据查询条件统计客户数量
     *
     * @param param 查询条件
     * @return 客户数量
     */
    Long countByParam(OtoCustCountParam param);

    /**
     * 统计销售指定跟进状态的客户数量
     *
     * @param sellerId         销售id
     * @param followStatusList 跟进状态
     * @return 客户数量
     */
    Long countByFollowStatus(Long sellerId, List<Integer> followStatusList);

    /**
     * 查询销售名下价值为N的用户
     *
     * @param sellerId 销售id
     * @return 符合任一条件的客户集合
     */
    List<OtoCustomerDto> valueIsNCust(Long sellerId);

    /**
     * 批量存储投放池客户到redis中
     *
     * @param companyId 公司di
     * @param custIds 客户id集合
     */
    void batchPutCust2Redis(Long companyId, Set<Long> custIds);

    /**
     * 批量从redis投放池中移除客户
     *
     * @param companyId 公司id
     * @param custIds 客户id
     */
    void batchRemoveCustFromRedis(Long companyId, Set<Long> custIds);

    /**
     * 从redis获取投放池客户
     *
     * @param companyId 公司id
     * @return 客户id集合
     */
    Set<Long> freshMembersFromRedis(Long companyId);

    /**
     * 手机端我的客户列表
     * @param param  列表查询参数
     * @return 列表数据
     */
    MobileCustListDto myCustListForPhone(OtoPhoneCustomerListParam param);
    /**
     * 根据 客户模糊姓名 和 销售id 模糊查询：客户id + 客户全名
     * @param custName 客户模糊姓名
     * @param sellerId 销售id
     * @return info
     */
    List<CustomerSimpleInfoDTO> findCustomerInfoByFuzzy(String custName, Long sellerId);

    /**
     * 根据 客户模糊姓名 搜索客户信息
     * @param custName
     * @return
     */
    List<CustomerSimpleInfoDTO> fuzzyCustNames(String custName);

    /**
     * 地区左匹配，模糊搜索列表
     * @param custArea 模糊客户地区
     * @return 地区列表页
     */
    List<String> fuzzyCustArea(String custArea);

    /**
     * 手动添加客户
     * @param param 参数
     * @return ID
     * @throws BizException 业务异常
     */
    Long manualAddCustomer(OtoManualAddCustomerParam param) throws BizException;
    /**
     * 根据客户ID批量更新客户来源
     * @param ids
     * @param source
     */
    void batchUpdateCustSource(List<Long> ids, Integer source);

    /**
     * 查询客户手机号
     *
     * @param phoneNums 手机号
     * @return 手机号
     */
    List<String> listByPhones(List<String> phoneNums);

    /**
     * 查询客户手机号
     *
     * @param phoneNums 手机号
     * @return 手机号
     */
    List<OtoCustomerDto> listByPhoneNums(List<String> phoneNums);

    /**
     * Excel批量插入客户
     * @param list
     * @return
     */
    List<OtoCustomerDto> batchInsertByExcel(List<OtoCustomerDto> list);

    /**
     * 查询投放池未认领的客户
     * @param custIds
     * @return
     */
    List<OtoCustomerDto> filterSellerCust(List<Long> custIds);

    /**
     * 查询投放池所有未认领的客户
     * @return
     */
    List<OtoCustomerDto> filterAllSellerCust();

    /**
     * 初始化录入客户信息
     * @param initDto 客户信息
     * @return 保存结果
     */
    CustInitResultDTO initCust(OtoCustomerInitDto initDto);

    /**
     * 查询客户集合
     * @param param 参数
     */
    List<OtoCustomerDto> listBySearchParam(OtoCustomerSearchParam param);

    /**
     * 计算客户系数
     * @param custFromSource 客户来源
     * @param custRegion 客户地区
     * @return 客户系数
     */
    Integer judgeCustCoefficient(String custFromSource, Integer custRegion);

    /**
     * 处理客户地区
     * @param bean 处理客户地区
     */
    void processCustRegion(OtoSellerCustBean bean);

    /**
     * 根据客户加微状态获取客户ext信息
     * @param addWxStatus 加微状态
     * @return 客户ext信息
     */
    List<OtoCustomerExtDto> findByAddWxStatus(List<Integer> addWxStatus);

    /**
     * 从es搜索
     *
     * @param param 搜索条件
     * @return left：符合条件的总数 value：当前页结果
     */
    Pair<Long, List<OtoCustDto>> searchFromEs(OtoCustCheckListParam param);

    /**
     * 从es搜索 角色=销售/专家的全部客户列表
     * @param param 搜索条件
     * @return left：符合条件的总数 value：当前页结果
     */
    Pair<Long, List<OtoCustDto>> searchCustFromEs(OtoCustCheckListParam param);

    /**
     * 新增放弃记录
     *
     * @param otoCustAbandonDto
     * @return
     */
    int saveRecord(OtoCustAbandonDto otoCustAbandonDto);

    /**
     * 获取客户的放弃记录
     *
     * @param custIds
     * @return
     */
    List<OtoCustAbandonDto> selectAbandon(List<Long> custIds);

    /**
     * 模糊查询该销售名下的在顾客户
     *
     * @param custName 客户姓名
     * @param custIds  客户ID
     * @return 结果
     */
    List<OtoCustomerDto> selectByCustName(String custName, List<Long> custIds);
}
