package cn.com.duiba.oto.enums.cust.info;

import lombok.AllArgsConstructor;
import lombok.Getter;

/**
 * 不动产情况枚举
 *
 * @author xiaotian
 * @date 2024-12-18
 */
@Getter
@AllArgsConstructor
public enum RealEstateEnum {

    /**
     * 无
     */
    NONE(0, "无"),

    /**
     * 1-2套
     */
    ONE_TO_TWO(1, "1-2套"),

    /**
     * 3-4套
     */
    THREE_TO_FOUR(2, "3-4套"),

    /**
     * 5套及以上
     */
    FIVE_OR_MORE(3, "5套及以上");

    /**
     * 编码
     */
    private final Integer code;

    /**
     * 描述
     */
    private final String desc;

    /**
     * 根据编码获取枚举
     *
     * @param code 编码
     * @return 枚举
     */
    public static RealEstateEnum getByCode(Integer code) {
        if (code == null) {
            return null;
        }
        for (RealEstateEnum item : values()) {
            if (item.getCode().equals(code)) {
                return item;
            }
        }
        return null;
    }

    /**
     * 根据编码获取描述
     *
     * @param code 编码
     * @return 描述
     */
    public static String getDescByCode(Integer code) {
        RealEstateEnum item = getByCode(code);
        return item != null ? item.getDesc() : null;
    }

    /**
     * 判断是否有不动产
     *
     * @param code 编码
     * @return 是否有不动产
     */
    public static boolean hasRealEstate(Integer code) {
        return !NONE.getCode().equals(code);
    }

    /**
     * 判断是否为多套房产（3套及以上）
     *
     * @param code 编码
     * @return 是否为多套房产
     */
    public static boolean isMultipleProperties(Integer code) {
        return THREE_TO_FOUR.getCode().equals(code) || FIVE_OR_MORE.getCode().equals(code);
    }

    /**
     * 判断是否为大量房产（5套及以上）
     *
     * @param code 编码
     * @return 是否为大量房产
     */
    public static boolean isLargePropertyPortfolio(Integer code) {
        return FIVE_OR_MORE.getCode().equals(code);
    }
}
