package cn.com.duiba.oto.util;


import cn.com.duiba.kjy.base.api.utils.IdMakeUtil;
import cn.com.duiba.oto.enums.push.PushMessageTypeEnum;
import cn.com.duiba.wolf.utils.UrlUtils2;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

/**
 * @author lizhi
 * @date 2019/7/16 11:08 AM
 */
public class PushEmbedUtils {

    private static final String SEPARATOR = "_";

    private PushEmbedUtils() {
    }

    /**
     * 将推送参数转换成一个参数值
     *
     * @param oaId            公众号ID
     * @param eventType       推送事件类型
     * @param messageTypeEnum 推送消息类型
     * @param urlOrder        推送中第几个链接
     * @param dpm             dpm，为空则不拼接dpm参数
     * @return 推送参数值
     */
    public static String getPushParamStr(Long oaId, String eventType, PushMessageTypeEnum messageTypeEnum, Integer urlOrder, String dpm) {
        return (oaId == null ? "" : IdMakeUtil.encodingId(oaId)) + SEPARATOR + (StringUtils.isBlank(eventType) ? "" : eventType) + SEPARATOR
                + (messageTypeEnum == null ? "" : messageTypeEnum.getType()) + SEPARATOR + (urlOrder == null ? "" : urlOrder)
                + SEPARATOR + (dpm == null ? "" : dpm);
    }

    /**
     * 将推送参数转换成一个参数值（oaId不加密）
     *
     * @param oaId            公众号ID
     * @param eventType       推送事件类型
     * @param messageTypeEnum 推送消息类型
     * @param urlOrder        推送中第几个链接
     * @param dpm             dpm，为空则不拼接dpm参数
     * @return 推送参数值
     */
    public static String getPushParamStrShort(Long oaId, String eventType, PushMessageTypeEnum messageTypeEnum, Integer urlOrder, String dpm) {
        return (oaId == null ? "" : (oaId + 3)) + SEPARATOR + (eventType == null ? "" : eventType) + SEPARATOR
                + (messageTypeEnum == null ? "" : messageTypeEnum.getType()) + SEPARATOR + (urlOrder == null ? "" : urlOrder)
                + SEPARATOR + (dpm == null ? "" : dpm);
    }

    /**
     * 将推送参数值，转换成推送参数map
     *
     * @param pushParamStr 推送参数值
     * @return 推送参数map
     */
    public static Map<String, String> getPushParamMapByStr(String pushParamStr) {//NOSONAR
        if (StringUtils.isBlank(pushParamStr)) {
            return Collections.emptyMap();
        }
        Long oaId = null;
        String pushId = null;
        Integer messageType = null;
        Integer urlOrder = null;
        String dpm = null;
        String[] split = pushParamStr.split(SEPARATOR);
        for (int i = 0; i < split.length; i++) {
            String str = split[i];
            if (StringUtils.isBlank(str)) {
                continue;
            }
            if (i == 0) {
                if (NumberUtils.isNumber(str)) {
                    oaId = Long.parseLong(str) - 3;
                } else {
                    oaId = IdMakeUtil.decodingId(str);
                }
            }
            if (i == 1) {
                pushId = str;
            }
            if (i == 2 && NumberUtils.isNumber(str)) {
                messageType = Integer.parseInt(str);
            }
            if (i == 3 && NumberUtils.isNumber(str)) {
                urlOrder = Integer.parseInt(str);
            }
            if (i == 4) {
                dpm = str;
            }
        }
        return getPushParamMap(oaId, pushId, messageType, urlOrder, dpm, null);
    }

    /**
     * 获取推送埋点参数
     *
     * @param oaId            公众号ID
     * @param eventType       推送事件类型
     * @param messageTypeEnum 推送消息类型
     * @param urlOrder        推送中第几个链接
     * @param dpm             dpm，为空则不拼接dpm参数
     * @return 推送埋点参数
     */
    public static String getPushParam(Long oaId, String eventType, PushMessageTypeEnum messageTypeEnum, Integer urlOrder, String dpm) {
        Map<String, String> pushParamMap = getPushParamMap(oaId, eventType, messageTypeEnum, urlOrder, dpm, null);
        return UrlUtils2.buildUrlParams(pushParamMap);
    }

    /**
     * 获取map中的推送埋点参数（不包含dpm）
     *
     * @param parameterMap request中的所有参数
     * @return 推送埋点参数（不包含dpm）
     */
    public static String getPushParam(Map<String, String[]> parameterMap) {
        Map<String, String> pushParamMap = getPushParamMap(parameterMap);
        return UrlUtils2.buildUrlParams(pushParamMap);
    }

    /**
     * 获取推送埋点参数
     *
     * @param oaId            公众号ID
     * @param eventType       推送事件类型
     * @param messageTypeEnum 推送消息类型
     * @param urlOrder        推送中第几个链接
     * @param dpm             dpm，为空则不拼接dpm参数
     * @return 推送埋点参数
     */
    public static String getEncodePushParam(Long oaId, String eventType, PushMessageTypeEnum messageTypeEnum, Integer urlOrder, String dpm) {
        return encode(getPushParam(oaId, eventType, messageTypeEnum, urlOrder, dpm));
    }

    /**
     * URL增加推送埋点参数
     *
     * @param url             链接
     * @param oaId            公众号ID
     * @param eventType       推送事件
     * @param messageTypeEnum 消息类型
     * @param urlOrder        推送中第几个链接
     * @param dpm             dpm，为空则不拼接dpm参数
     * @return 带参数的URL
     */
    public static String buildPushParam(String url, Long oaId, String eventType, PushMessageTypeEnum messageTypeEnum, Integer urlOrder, String dpm) {
        return buildPushParam(url, oaId, eventType, messageTypeEnum, urlOrder, dpm, null);
    }

    /**
     * URL增加推送埋点参数
     *
     * @param url             链接
     * @param oaId            公众号ID
     * @param eventType       推送事件
     * @param messageTypeEnum 消息类型
     * @param urlOrder        推送中第几个链接
     * @param dpm             dpm，为空则不拼接dpm参数
     * @param markValue       唯一标识，为空则不加唯一标识
     * @return 带参数的URL
     */
    public static String buildPushParam(String url, Long oaId, String eventType, PushMessageTypeEnum messageTypeEnum, Integer urlOrder, String dpm, String markValue) {
        Map<String, String> params = getPushParamMap(oaId, eventType, messageTypeEnum, urlOrder, dpm, markValue);
        if (!url.contains("?")) {
            return UrlUtils2.appendParams(url, params);
        }
        Map<String, String> map = UrlUtils2.extractUrlParamsFromUrl(url);
        if (MapUtils.isEmpty(map)) {
            return UrlUtils2.appendParams(url, params);
        }
        String uriUrl = map.get("uri");
        if (StringUtils.isBlank(uriUrl)) {
            return UrlUtils2.appendParams(url, params);
        }
        String decode = decode(uriUrl);
        String decodePushUrl = UrlUtils2.appendParams(decode, params);
        map.put("uri", encode(decodePushUrl));
        return UrlUtils2.appendParams(UrlUtils2.extractUrl(url), map);
    }

    /**
     * 增加推送唯一标识
     *
     * @param page      小程序模板消息，跳转页面
     * @param markValue 唯一标识
     * @return 带参数的页面地址
     */
    public static String buildMpPushParam(String page, String markValue) {
        Map<String, String> params = new HashMap<>();
        if (markValue == null) {
            markValue = "";
        }
        params.put("unique_mark", markValue);
        return UrlUtils2.appendParams(page, params);
    }

    /**
     * URL追加map中的推送埋点参数（不包含dpm）
     *
     * @param url          链接
     * @param parameterMap request中的所有参数
     * @return 带推送参数的URL
     */
    public static String buildPushParam(String url, Map<String, String[]> parameterMap) {
        Map<String, String> pushParamMap = getPushParamMap(parameterMap);
        return UrlUtils2.appendParams(url, pushParamMap);
    }

    /**
     * 获取请求中推送埋点参数（不包含dpm）
     *
     * @param parameterMap request中的所有参数
     * @return 推送埋点参数map
     */
    public static Map<String, String> getPushParamMap(Map<String, String[]> parameterMap) {
        if (MapUtils.isEmpty(parameterMap)) {
            return Collections.emptyMap();
        }
        Map<String, String> params = new HashMap<>(16);
        Set<String> pushKey = getPushParamMap(0L, "", 0, 0, null, "").keySet();
        for (String key : pushKey) {
            String[] param = parameterMap.get(key);
            if (isNotEmpty(param)) {
                params.put(key, param[0]);
            }
        }
        return params;
    }

    /**
     * 封装推送埋点参数map
     *
     * @param oaId            公众号ID
     * @param eventType       推送事件
     * @param messageTypeEnum 消息类型
     * @param urlOrder        推送中第几个链接
     * @param dpm             dpm，为空则不拼接dpm参数
     * @return 推送埋点参数map
     */
    private static Map<String, String> getPushParamMap(Long oaId, String eventType, PushMessageTypeEnum messageTypeEnum, Integer urlOrder, String dpm, String markValue) {
        String pushId = StringUtils.isBlank(eventType) ? null : eventType;
        Integer messageType = Objects.isNull(messageTypeEnum) ? null : messageTypeEnum.getType();
        return getPushParamMap(oaId, pushId, messageType, urlOrder, dpm, markValue);
    }

    /**
     * 封装推送埋点参数map
     *
     * @param oaId        公众号ID
     * @param pushId      推送事件Id
     * @param messageType 推送消息类型
     * @param urlOrder    推送中第几个链接
     * @param dpm         dpm，为空则不拼接dpm参数
     * @return 推送埋点参数map
     */
    private static Map<String, String> getPushParamMap(Long oaId, String pushId, Integer messageType, Integer urlOrder, String dpm, String markValue) {
        Map<String, String> params = new HashMap<>(16);
        params.put("push", "push");
        if (Objects.nonNull(oaId)) {
            params.put("oa_id", IdMakeUtil.encodingId(oaId));
        }
        if (Objects.nonNull(pushId)) {
            params.put("push_id", pushId);
        }
        if (Objects.nonNull(messageType)) {
            params.put("message_type", String.valueOf(messageType));
        }
        if (Objects.nonNull(urlOrder)) {
            params.put("url_order", String.valueOf(urlOrder));
        }
        if (StringUtils.isNotBlank(dpm)) {
            params.put("dpm", dpm);
        }
        if (markValue != null) {
            params.put("unique_mark", markValue);
        }
        return params;
    }

    /**
     * URL编码
     *
     * @param url 链接
     * @return 编码后的链接
     */
    public static String encode(String url) {
        try {
            return URLEncoder.encode(url, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            return url;
        }
    }

    /**
     * URL解码
     *
     * @param url 链接
     * @return 解码后的链接
     */
    public static String decode(String url) {
        try {
            return URLDecoder.decode(url, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            return url;
        }
    }

    private static boolean isNotEmpty(String[] param) {
        return Objects.nonNull(param) && param.length > 0;
    }

    /**
     * 给文本信息插入唯一标识
     *
     * @param msg       文本信息
     * @param eventType 推送事件
     * @return 插入唯一标识后的文本
     */
    public static String insertUniqueMark(String msg, String eventType, String markValue) {
        if (StringUtils.isBlank(msg) || eventType == null) {
            return msg;
        }
        String pushIdKey = "push_id";
        //最终结果收集
        StringBuilder sb = new StringBuilder();
        //push_id的长度
        int pushIdKeyLength = pushIdKey.length();
        //push_id的值
        //push_id值的长度
        int eventTypeLength = eventType.length();
        int lastIndex = 0;
        while (lastIndex < msg.length()) {
            //push_id开始的下标
            int pushIdIndex = msg.indexOf(pushIdKey, lastIndex);
            if (pushIdIndex < 0) {
                //没有push_id
                break;
            }
            //push_id值开始的下标
            int eventTypeIndex = msg.indexOf(eventType, pushIdIndex);
            if (eventTypeIndex < 0) {
                //没有push_id值，非推送参数
                break;
            }
            //push_id结束的下标
            int pushIdLastIndex = pushIdIndex + pushIdKeyLength;
            //push_id值结束的下标
            int eventTypeLastIndex = eventTypeIndex + eventTypeLength;
            //获取push_id和值之间的符号，用于判断是否编码过
            String equalSign = msg.substring(pushIdLastIndex, eventTypeIndex);
            //获取编码次数
            int count = encodeCount(equalSign);
            //根据编码次数，获取要插入的唯一标识
            String mark = getMark(count, markValue);
            //追加push_id之前的数据
            sb.append(msg, lastIndex, pushIdIndex);
            //追加唯一标识
            sb.append(mark);
            //追加push_id及push_id的值
            sb.append(msg, pushIdIndex, eventTypeLastIndex);
            //将处理下标移动到push_id值后
            lastIndex = eventTypeLastIndex;
        }
        sb.append(msg.substring(lastIndex));
        return sb.toString();
    }

    /**
     * 根据编码次数，获取唯一标识
     *
     * @param encodeCount 编码次数
     * @return 唯一标识
     */
    private static String getMark(int encodeCount, String markValue) {
        String mark = "unique_mark=" + markValue + "&";
        if (encodeCount <= 0) {
            return mark;
        }
        for (int i = 0; i < encodeCount; i++) {
            mark = encode(mark);
        }
        return mark;
    }

    /**
     * 获取编码次数
     *
     * @param equalSign 编码后的 =
     * @return 编码次数
     */
    private static int encodeCount(String equalSign) {
        int encodeCount = 0;
        if (StringUtils.isBlank(equalSign)) {
            return encodeCount;
        }
        String sign = "=";
        int maxCount = 5;
        for (int i = 0; i < maxCount; i++) {
            if (StringUtils.equals(equalSign, sign)) {
                return encodeCount;
            }
            equalSign = decode(equalSign);
            encodeCount++;
        }
        return encodeCount;
    }
}
