package cn.com.duiba.projectx.sdk.repeatable;

import cn.com.duiba.projectx.sdk.BizRuntimeException;
import cn.com.duiba.projectx.sdk.ErrorCode;
import cn.com.duiba.projectx.sdk.InnerBizCode;
import cn.com.duiba.projectx.sdk.UserRequestApi;
import cn.com.duiba.projectx.sdk.repeatable.annotation.LocalInvoke;
import cn.com.duiba.projectx.sdk.repeatable.annotation.RemoteInvoke;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author ZhouFeng zhoufeng@duiba.com.cn
 * @version $Id: ComponentContext.java , v 0.1 2020-02-12 2:01 下午 ZhouFeng Exp $
 */
public class ComponentContext<I extends UserRequestApi> {

    /**
     * logger
     */
    private static final Logger LOGGER = LoggerFactory.getLogger(ComponentContext.class);

    private Component<I> component;

    private final Map<String, Method> localMethodMaps = new HashMap<>();

    private final Map<String, Method> remoteMethodMaps = new HashMap<>();


    public ComponentContext(Component<I> component) {
        this.component = component;

        Map<String, Method> map =
                Arrays.stream(component.getClass().getMethods()).filter(m -> m.isAnnotationPresent(LocalInvoke.class))
                        .collect(Collectors.toMap(Method::getName, Function.identity()));
        for (Method method : map.values()) {
            Class<?> returnType = method.getReturnType();
            if (!Response.class.isAssignableFrom(returnType)) {
                LOGGER.error("本地方法初始化失败，{}.{}方法返回值必须为Response", component.getClass().getName(), method.getName());
                throw new IllegalStateException("本地方法初始化失败");
            }
        }
        localMethodMaps.putAll(map);

        //远程调用的方法
        remoteMethodMaps.putAll(Arrays.stream(component.getClass().getMethods()).filter(m -> m.isAnnotationPresent(RemoteInvoke.class))
                .collect(Collectors.toMap(Method::getName, Function.identity())));
    }

    public Method getLocalAction(String name) {
        //当前不支持重载方法
        return localMethodMaps.get(name);
    }

    public Method getRemoteAction(String name) {
        //当前不支持重载方法
        return remoteMethodMaps.get(name);
    }

    public Object doRemoteAction(String actionName, Object... args) {
        try {
            Method action = remoteMethodMaps.get(actionName);
            if (action == null) {
                throw new BizRuntimeException(ErrorCode.ERR_10021, "方法不存在");
            }
            return action.invoke(component, args);
        } catch (BizRuntimeException biz) {
            throw biz;
        } catch (Throwable t) {
            LOGGER.error("执行失败", t);
            LOGGER.warn("方法内部执行失败:{}.{}", component.getClass(), actionName, t.getCause());
            throw new BizRuntimeException(InnerBizCode.SYSTEM_ERROR);
        }
    }

}
