package cn.com.duiba.projectx.sdk.repeatable.mvp;

import cn.com.duiba.projectx.sdk.BizRuntimeException;
import cn.com.duiba.projectx.sdk.ErrorCode;
import cn.com.duiba.projectx.sdk.UserRequestApi;
import cn.com.duiba.projectx.sdk.UserRequestContext;
import cn.com.duiba.projectx.sdk.playway.InitializationContext;
import cn.com.duiba.projectx.sdk.repeatable.ActionChain;
import cn.com.duiba.projectx.sdk.repeatable.Component;
import cn.com.duiba.projectx.sdk.repeatable.ComponentFactory;
import cn.com.duiba.projectx.sdk.repeatable.EventContext;
import cn.com.duiba.projectx.sdk.repeatable.EventContextType;
import cn.com.duiba.projectx.sdk.repeatable.FlowConfig;
import cn.com.duiba.projectx.sdk.repeatable.MultiConfig;
import cn.com.duiba.projectx.sdk.repeatable.Response;
import cn.com.duiba.projectx.sdk.repeatable.annotation.DevConfig;
import cn.com.duiba.projectx.sdk.repeatable.utils.ReflectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;

import java.lang.reflect.Field;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * MVP统一接口
 *
 * @author ZhouFeng zhoufeng@duiba.com.cn
 * @version $Id: Mvp.java , v 0.1 2020-01-21 9:59 上午 ZhouFeng Exp $
 */
public interface Mvp extends EventContextType {

    InitializationContext getInitializationContext();

    default <I extends UserRequestApi> Response<I> invokeChain(ActionChain<I> actionChain, EventContext eventContext,
                                                               UserRequestContext userRequestContext, I api) {
       return actionChain.invoke(eventContext, userRequestContext, api);
    }

    @SuppressWarnings("unchecked")
    default void init() {
        boolean inited = false;

        //暂时通过读取当前路径下的CustomConfig.class,后续改用从配置文件中解析
        String className = this.getClass().getName();
        int index = className.lastIndexOf(".");
        String configClassName = className.substring(0, index + 1) + "CustomConfig";
        MultiConfig newConfig = null;
        try {
            Class<?> clazz = Class.forName(configClassName, true, this.getClass().getClassLoader());
            Object config = clazz.newInstance();
            if (FlowConfig.class.isAssignableFrom(clazz)) {
                FlowConfig<Mvp> oldConfig = (FlowConfig<Mvp>) config;
                oldConfig.config(this);
                inited = true;
            } else if (MultiConfig.class.isAssignableFrom(clazz)) {
                newConfig = ((MultiConfig) config);
            } else {
                throw new RuntimeException(className + "玩法配置不存在");
            }
        } catch (Throwable e) {
            throw new RuntimeException(className + "玩法解析配置错误", e);
        }

        if (!inited) {
            //如果 inited 为false，说明没有使用旧的配置方式，需要走新的配置
            Field[] fields = this.getClass().getDeclaredFields();
            for (Field field : fields) {
                if (field.getType() == ActionChain.class) {
                    field.setAccessible(true);
                    try {
                        ActionChain<?> actionChain = (ActionChain<?>) field.get(this);
                        String name = actionChain.getName();
                        if (StringUtils.isBlank(name)) {
                            name = field.getName();
                        }
                        ActionChains.put(this.getClass(), name, actionChain);
                    } catch (IllegalAccessException e) {
                        throw new IllegalStateException(e);
                    }
                }
            }
            newConfig.config(this);
        }
    }

    /**
     * 获取 ActionChain
     *
     * @param name
     * @return
     */
    default <T extends UserRequestApi> ActionChain<T> getChain(String name) {
        return ActionChains.getChain(this.getClass(), name);
    }

    default void populate(Mvp mvp, Map<String, String> configMap) {
        Field[] fields = mvp.getClass().getDeclaredFields();
        for (Field field : fields) {
            DevConfig devConfig = field.getAnnotation(DevConfig.class);
            if (devConfig == null) {
                continue;
            }
            String key = devConfig.key();
            String value = configMap.get(key);
            if (StringUtils.isBlank(value)) {
                throw new BizRuntimeException("找不到[" + mvp.getClass().getName() + "." + key + "]配置");
            }
            if (NumberUtils.isCreatable(value)) {
                //如果是数字，则做值校验
                double aDouble = Double.parseDouble(value);
                if (aDouble < devConfig.minInclude() || aDouble >= devConfig.maxExclude()) {
                    throw new BizRuntimeException(String.format("%s值校验出错,要求[%s,%s),实际:%s", key, devConfig.minInclude(),
                            devConfig.maxExclude(), value));
                }
            }
            ReflectUtils.assign(field, mvp, value);
        }
    }


    /**
     * 调用实际的组件方法
     *
     * @param componentClass 要调用的组件
     * @param context        请求上下文
     * @param args           实际的方法请求参数
     * @return
     */
    default Object remoteInvoke(Class<? extends Component<? extends UserRequestApi>> componentClass,
                                UserRequestContext context,
                                Object... args) {
        StackTraceElement[] stackTrace = Thread.currentThread().getStackTrace();
        String methodName = stackTrace[2].getMethodName();
        if (methodName.contains("_")) {
            //方法名可能是AA_BB，只取_后面部分
            methodName = methodName.substring(methodName.lastIndexOf("_") + 1);
        }
        return remoteInvoke(componentClass, methodName, context, args);

    }

    /**
     * 调用实际的组件方法
     *
     * @param componentClass 要调用的组件
     * @param methodName     方法名
     * @param context        请求上下文
     * @param args           实际的方法请求参数
     * @return
     */
    default Object remoteInvoke(Class<? extends Component<? extends UserRequestApi>> componentClass, String methodName,
                                UserRequestContext context, Object... args) {
        //取出组件id，可能为空
        String cpId = context.getHttpRequest().getParameter("cp_id");
        if (cpId == null) {
            cpId = "";
        }
        Component<?> component = ComponentFactory.get(componentClass, cpId);
        if (component == null) {
            throw new BizRuntimeException(ErrorCode.ERR_10021, "组件不存在");
        }
        return component.doRemoteAction(methodName, args);

    }

    /**
     * 获取当前项目中的组件
     *
     * @return
     */
    default List<String> currentComponents() {
        return ComponentFactory.currentComponents(this.getClass().getClassLoader()).stream()
                .map(Object::getClass).map(Class::getSimpleName).distinct().collect(Collectors.toList());
    }
}
