package cn.com.duiba.projectx.sdk.repeatable;

import cn.com.duiba.projectx.sdk.BizRuntimeException;
import cn.com.duiba.projectx.sdk.ErrorCode;
import cn.com.duiba.projectx.sdk.UserRequestApi;
import cn.com.duiba.projectx.sdk.playway.InitializationContext;
import cn.com.duiba.projectx.sdk.repeatable.annotation.DevConfig;
import cn.com.duiba.projectx.sdk.repeatable.utils.ConfigUtils;
import cn.com.duiba.projectx.sdk.repeatable.utils.ReflectUtils;
import com.google.common.collect.HashBasedTable;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Properties;

/**
 * @author ZhouFeng zhoufeng@duiba.com.cn
 * @version $Id: ComponentFactory.java , v 0.1 2020-02-03 4:07 下午 ZhouFeng Exp $
 */
public class ComponentFactory {

    /**
     * logger
     */
    private static final Logger LOGGER = LoggerFactory.getLogger(ComponentFactory.class);

    private static final HashBasedTable<Class<? extends Component<? extends UserRequestApi>>, String,
            Component<? extends UserRequestApi>> CONTAINER = HashBasedTable.create();

    private static final Multimap<ClassLoader, Component<? extends UserRequestApi>> CLASS_LOADER_CLASS_MULTIMAP =
            HashMultimap.create();

    /**
     * 创建并注册组件，如果组件已存在，则直接返回已有的
     *
     * @param componentClass
     * @param initializationContext
     * @param <I>
     * @return
     */
    public static <I extends UserRequestApi> Component<I> createAndRegister(Class<? extends Component<I>> componentClass,
                                                                            InitializationContext initializationContext) {
        return createAndRegister(componentClass, initializationContext, "");
    }

    /**
     * 创建并注册组件，如果组件已存在，则直接返回已有的
     *
     * @param componentClass
     * @param initializationContext
     * @param id
     * @param <I>
     * @return
     */
    public static <I extends UserRequestApi> Component<I> createAndRegister(Class<? extends Component<I>> componentClass,
                                                                            InitializationContext initializationContext,
                                                                            String id) {
        Component<? extends UserRequestApi> component = CONTAINER.get(componentClass, id);
        if (component == null) {
            synchronized (componentClass) {
                component = CONTAINER.get(componentClass, id);
                if (component == null) {
                    component = newInstance(componentClass, initializationContext, id);
                    CONTAINER.put(componentClass, id, component);
                    CLASS_LOADER_CLASS_MULTIMAP.put(componentClass.getClassLoader(), component);
                }
            }
        }
        return (Component<I>) component;
    }

    /**
     * 获取组件
     *
     * @param <I>
     * @param componentClass 组件类型
     * @param id             组件id
     * @return
     */
    public static <I extends UserRequestApi> Component<I> get(Class<? extends Component<? extends UserRequestApi>> componentClass,
                                                              String id) {
        return (Component<I>) CONTAINER.get(componentClass, id);
    }

    /**
     * 获取当前项目已注册的组件
     *
     * @param classLoader
     * @return
     */
    public static List<Component<? extends UserRequestApi>> currentComponents(ClassLoader classLoader) {
        return new ArrayList<>(CLASS_LOADER_CLASS_MULTIMAP.get(classLoader));
    }

    private static <I extends UserRequestApi> Component<I> newInstance(Class<? extends Component<I>>
                                                                               componentClass,
                                                                       InitializationContext initializationContext,
                                                                       String id) {
        try {
            Constructor<? extends Component<I>> constructor = componentClass.getConstructor(String.class);
            Component<I> component = constructor.newInstance(id);
            populate(component, initializationContext);
            return component;
        } catch (BizRuntimeException e) {
            throw e;
        } catch (Throwable t) {
            LOGGER.error("创建组件失败", t);
            throw new BizRuntimeException(ErrorCode.ERR_10000, "系统繁忙");
        }

    }

    private static void populate(Component<?> component, InitializationContext initializationContext) {

        Class<? extends Component> clazz = component.getClass();
        Field[] fields = clazz.getDeclaredFields();
        Properties devProperties = initializationContext.getDevProperties();
        Map<String, String> operationConfig = initializationContext.getOperationConfig();
        Map<String, String> config = ConfigUtils.merge(operationConfig, devProperties);
        for (Field field : fields) {
            if (!field.isAnnotationPresent(DevConfig.class)) {
                continue;
            }
            DevConfig devConfig = field.getAnnotation(DevConfig.class);
            String key = configKey(component, devConfig.key());
            String value = config.get(key);
            if (StringUtils.isBlank(value)) {
                throw new BizRuntimeException("找不到[" + component.getClass().getName() + "." + key + "]配置");
            }
            if (NumberUtils.isCreatable(value)) {
                //如果是数字，则做值校验
                double aDouble = Double.parseDouble(value);
                if (aDouble < devConfig.minInclude() || aDouble >= devConfig.maxExclude()) {
                    throw new BizRuntimeException(String.format("%s值校验出错,要求[%s,%s),实际:%s", key, devConfig.minInclude(),
                            devConfig.maxExclude(), value));
                }
            }
            ReflectUtils.assign(field, component, value);
        }

    }

    private static String configKey(Component<?> component, String baseKey) {
        if (StringUtils.isBlank(component.getId())) {
            return baseKey;
        }
        return baseKey + "." + component.getId();
    }
}
