package cn.com.duiba.projectx.sdk.utils;

import cn.com.duiba.projectx.sdk.Api;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * @Author: DSG
 * @Date: 18:52 2021/6/24
 * @Description: RedisApi
 */
public interface RedisApi extends Api {
    /**
     * 设置一个键值对,键值为{@code String}类型
     * 如果键不存在，则新增一条记录；如果键已经存在，则会修改值，并且设置失效时间
     * 默认失效时间为6年
     *
     * @param key   键
     * @param value 值
     */
    void set(String key, String value);

    /**
     * 设置一个键值对,键值为{@code String}类型,并设置失效时间
     * 如果键不存在，则新增一条记录；如果键已经存在，则会修改值，并且设置失效时间
     *
     * @param key      键
     * @param value    值
     * @param l        失效时间
     * @param timeUnit 时间单位
     */
    void set(String key, String value, long l, TimeUnit timeUnit);

    /**
     * 设置一个键值对，值为{@code String}类型，默认失效时间为6年
     * 只有当键不存在时才会设置成功
     *
     * @param key   键
     * @param value 值
     * @return 设置记录成功时返回 {@code true};否则返回 {@code false}
     */
    Boolean setIfAbsent(String key, String value);

    /**
     * 设置一个键值对，值为{@code String}类型，并设置失效时间
     * 只有当键不存在时才会设置成功
     *
     * @param key   键
     * @param value 值
     * @return 设置记录成功时返回 {@code true};否则返回 {@code false}
     */
    Boolean setIfAbsent(String key, String value, long l, TimeUnit timeUnit);

    /**
     * 通过键,获取{@code String}类型的值
     *
     * @param key 键
     * @return 值
     */
    String get(String key);

    /**
     * 增长键对应的值，默认失效时间为6年
     * 如果键不存在，则将创建一调记录再自增，并设置默认失效时间；
     * offset可以为负数，表示自减
     *
     * @param key      键
     * @param offset   自增的值(为负数代表自减)
     * @param timeOut  超时时间, 不设置传null
     * @param timeUnit 超时时间单位, 不设置传null
     * @return 自增之后的值
     */
    Long increment(String key, long offset, Long timeOut, TimeUnit timeUnit);

    /**
     * 批量获取{@code String}类型的值，如果某个键不存在，将使用{@code null}填充对应位置
     *
     * @param keys 键组成的列表
     * @return 由值组成的{@code List}，一定不会为{@code null}；如果值存在且尚未过期，目标位置为对应值；否则目标位置为{@code null}
     */
    List<String> multiGet(List<String> keys);

    /**
     * 向list类型的数据结构中新增一个元素并返回list的长度,默认失效时间为6年
     * 如果键不存在,则创建一个list
     * 如果键的类型不是list,则报错
     *
     * @param key   键
     * @param value 增加的值
     * @param timeOut  超时时间, 不设置传null
     * @param timeUnit 超时时间单位, 不设置传null
     * @return list的长度
     */
    Long listAdd(String key, String value, Long timeOut, TimeUnit timeUnit);

    /**
     * 向list类型的数据结构中新增多个元素并返回list的长度,默认失效时间为6年
     * 如果键不存在,则创建一个list
     * 如果键的类型不是list,则报错
     *
     * @param key    键
     * @param values 增加的值
     * @param timeOut  超时时间, 不设置传null
     * @param timeUnit 超时时间单位, 不设置传null
     * @return list的长度
     */
    Long listAdd(String key, List<String> values, Long timeOut, TimeUnit timeUnit);

    /**
     * 获取list类型的长度
     *
     * @param key 键
     * @return list的长度
     */
    Long listSize(String key);

    /**
     * 获取某个键的list
     *
     * @param key 键
     * @return list
     */
    List<String> getList(String key);

    /**
     * 向set类型的数据结构中新增一个元素并返回set的长度,默认失效时间为6年
     * 如果键不存在,则创建一个set
     * set数据结构是无序的,去重的
     * 如果键的类型不是set,则报错
     *
     * @param key   键
     * @param value 增加的值
     * @param timeOut  超时时间, 不设置传null
     * @param timeUnit 超时时间单位, 不设置传null
     * @return 本次add操作新元素的数量, 已存在的元素不计算
     */
    Long setAdd(String key, String value, Long timeOut, TimeUnit timeUnit);

    /**
     * 向set类型的数据结构中新增多个元素并返回set的长度,默认失效时间为6年
     * 如果键不存在,则创建一个set
     * set数据结构是无序的,去重的
     * 如果键的类型不是set,则报错
     *
     * @param key    键
     * @param values 增加的值
     * @param timeOut  超时时间, 不设置传null
     * @param timeUnit 超时时间单位, 不设置传null
     * @return 本次add操作新元素的数量, 已存在的元素不计算
     */
    Long setAdd(String key, Set<String> values, Long timeOut, TimeUnit timeUnit);

    /**
     * 获取set的长度
     *
     * @param key 键
     * @return set的长度
     */
    Long setSize(String key);

    /**
     * 删除set类型中的值
     * 如果set中包含传入的值,则删除,如果不包含,则什么都不做
     *
     * @param key    键
     * @param values 要删除的值
     * @return 成功被删除的元素数量
     */
    Long setRemove(String key, String... values);

    /**
     * 随机从set中取出一个元素,随后从set中删除该元素
     * 当key不存在,或者set为空时, 返回null
     *
     * @param key 键
     * @return 取出的值
     */
    String setPop(String key);

    /**
     * 查找set键中是否包含传入的值
     *
     * @param key   键
     * @param value 要查找的值
     * @return 如果包含返回true, 否则false
     */
    boolean setContains(String key, String value);

    /**
     * 通过key,获取set
     *
     * @param key 键
     * @return set
     */
    Set<String> getSet(String key);

    /**
     * 设置一个hash结构的键值对,key的默认失效时间是6年
     * 如果key不存在则创建一个hash表
     * 如果hashkey的值已存在,则会修改
     *
     * @param key     键
     * @param hashKey hash键
     * @param value   hash值
     * @param timeOut  超时时间, 不设置传null
     * @param timeUnit 超时时间单位, 不设置传null
     */
    void hashPut(String key, String hashKey, String value, Long timeOut, TimeUnit timeUnit);

    /**
     * 批量设置hash键值对,key的默认失效时间是6年
     * 如果key不存在则创建一个hash表
     * 如果其中有hashkey的值已存在,则会修改
     *
     * @param key 键
     * @param map 键值对
     * @param timeOut  超时时间, 不设置传null
     * @param timeUnit 超时时间单位, 不设置传null
     */
    void hashPutAll(String key, Map<String, String> map, Long timeOut, TimeUnit timeUnit);

    /**
     * 自增某个hashkey的值,offset为负数则是自减,key的默认失效时间是6年
     * 如果key不存在则创建一个hash表,且该hashkey初始化为0
     * 如果该hashkey不是数字,则报错
     *
     * @param key     键
     * @param hashKey hash键
     * @param offset  自增量
     * @param timeOut  超时时间, 不设置传null
     * @param timeUnit 超时时间单位, 不设置传null
     */
    void hashIncr(String key, String hashKey, long offset, Long timeOut, TimeUnit timeUnit);

    /**
     * 删除hash中指定的hashkey,可以一次删除多个hashkey
     *
     * @param key     键
     * @param hashKey hash键
     */
    void hashDel(String key, String... hashKey);

    /**
     * 获取hash中指定的hashkey的值
     * 如果值不存在或者key不存在,则返回null
     *
     * @param key     键
     * @param hashKey hash键
     * @return 值
     */
    String hashGet(String key, String hashKey);

    /**
     * 获取键下的全部hash键值对
     * 如果key不存在或者hash为空, 则返回空map
     *
     * @param key 键
     * @return 键值对
     */
    Map<String, String> hashGetAll(String key);

    /**
     * 删除
     * @param key
     */
    void delete(String key);

    /**
     * 批量删除, 一次不超过100条
     * @param key
     */
    void delete(List<String> keys);

    /**
     * 设置键对应的失效时间
     *
     * @param key      键
     * @param l        失效时间
     * @param timeUnit 时间单位
     * @return 如果键存在则返回{@code true}，如果不存在返回{@code false}
     */
    Boolean expire(String key, long l, TimeUnit timeUnit);

    /**
     * 设置键对应的失效时间,指定失效日期
     *
     * @param key  键
     * @param date 失效日期
     * @return 如果键存在则返回{@code true}，如果不存在返回{@code false}
     */
    Boolean expire(String key, Date date);
}
