package cn.com.duiba.projectx.sdk.component.rank;

import cn.com.duiba.projectx.sdk.BizRuntimeException;
import cn.com.duiba.projectx.sdk.component.rank.dto.LeaderboardArchiveDto;
import cn.com.duiba.projectx.sdk.component.rank.dto.RankConfigDto;
import cn.com.duiba.projectx.sdk.playway.base.Ranking;

import java.util.List;

/**
 * 定时任务场景下使用的任务组件Api
 * 方法和接口线程中使用的组件Api高度一致，因为历史原因就不做api结构上继承复用等设计了
 * <p>
 * RankTimerApi 设计服务于定时任务 设计的历史原因即：
 * RankApi和RankTimerApi的作用域有别，前者在Request（接口线程）作用域，后者在Timer（定时任务线程）。
 * <p>
 * 更好的设计可以是将依赖关系梳理清楚，减少这种“带状态”的对象的出现，以便于上层扩展，否则就会出现这样的重复逻辑的api。
 * <p>
 * 后期维护的成本也非常高，且容易出错，如改了一处，忘改另一处
 * <p>
 * 注：后期如果RankApi或RankTimerApi有任一改动，需要两边进行同步修改
 *
 * @author zhuxj
 * @since 2023/6/6
 */
public interface RankTimerApi {

    /**
     * 获取前几名的排行榜信息
     * <p>
     * 此方法用于获取指定排行榜组件下的topN数据，有别于{@link cn.com.duiba.projectx.sdk.component.rank.RankApi#getTopN}，
     * 此方法不通过 ScoringUserRequestApi#getTopN 间接获得topN数据。
     *
     * @param rankPlaywayId 排行榜玩法组件ID
     * @param topN          前几名
     *                      不允许调用超过3000名的排名，会有性能问题
     * @return ranking 列表
     * @see cn.com.duiba.projectx.sdk.component.rank.RankApi#getTopN(String, int)
     */
    List<Ranking> getTopN(String rankPlaywayId, int topN);

    /**
     * 更新排行榜分数
     * ⚠️ 接口内部不作单用户并发处理，由业务方决定
     *
     * @param score 分值
     */
    void updateRankingScore(String rankPlaywayId, String userId, int score);


    /**
     * 更新排行榜分数自动+1
     * 使用从高到低的排序
     * ⚠️ 接口内部不作单用户并发处理，由业务方决定
     *
     * @param rankPlaywayId 排行榜玩法组件ID
     */
    void updateRankingScoreByIncr(String rankPlaywayId, String userId, int score);


    /**
     * 替换排行榜分数 - 针对新榜
     * 使用从高到低的排序
     * ⚠️ 接口内部不作单用户并发处理，由业务方决定
     *
     * @param rankPlaywayId 排行榜玩法组件ID
     */
    void replaceRankingScore(String rankPlaywayId, String userId, int score);

    /**
     * 清零我的分数 - 针对新榜
     * ⚠️ 接口内部不作单用户并发处理，由业务方决定
     *
     * @param rankPlaywayId 排行榜玩法组件ID
     */
    void clearRankingScore(String rankPlaywayId, String userId);

    /**
     * 获取新排行榜当前正在进行中的rankingType
     * 如果排行榜已结束，返回的是最后一期
     * 如果排行榜未开始，返回的是第一期
     *
     * @param rankPlaywayId 排行榜玩法组件ID
     * @return
     * @throws BizRuntimeException
     */
    String getCurrentRankingType(String rankPlaywayId);

    /**
     * 获取新排行榜上一期的rankingType
     *
     * @param rankPlaywayId 排行榜玩法组件ID
     * @return
     * @throws BizRuntimeException
     */
    String getPrefixRankingType(String rankPlaywayId);

    /**
     * 获取组件配置
     *
     * @param rankPlaywayId
     * @return
     */
    RankConfigDto getRankConfig(String rankPlaywayId);

    /**
     * 获取排行榜期次奖励
     *
     * @param rankPlaywayId
     * @param phaseType     0 获取当前期次 -1 获取上一期次
     * @return
     */
    LeaderboardArchiveDto getRankOptionConfig(String rankPlaywayId, int phaseType);
}
