package cn.com.duiba.projectx.sdk.utils;

import org.apache.commons.lang3.StringUtils;

import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Date;

public class CommonUtils {
    // 手机号正则表达式
    public static final String MOBILE_REG = "^1\\d{10}$";
    // 通常日期时间格式
    public static final DateTimeFormatter COMMON_DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
    // 通常日期格式
    public static final DateTimeFormatter COMMON_DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    // 数字日期格式
    public static final DateTimeFormatter NUMBER_DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyyMMdd");
    // 通常时间格式
    public static final DateTimeFormatter COMMON_TIME_FORMATTER = DateTimeFormatter.ofPattern("HH:mm:ss");
    // 短时间格式
    public static final DateTimeFormatter SHORT_TIME_FORMATTER = DateTimeFormatter.ofPattern("HH:mm");

    private CommonUtils() {
    }

    /**
     * 手机号脱敏处理
     * @param mobile
     * @return
     */
    public static String desensitizeMobile(String mobile) {
        if(StringUtils.isBlank(mobile) || !mobile.matches(MOBILE_REG)) {
            return mobile;
        }
        return mobile.replaceAll("^(\\d{3})\\d{4}(\\d{4})$", "$1****$2");
    }

    /**
     * 昵称脱敏（第一位显示，后续位用*替换）
     * @param nickname
     * @return
     */
    public static String desensitizeNickname(String nickname) {
        if(StringUtils.isBlank(nickname) || nickname.length() == 1) {
            return nickname;
        }
        StringBuilder sb = new StringBuilder(nickname.substring(0, 1));
        for(int i = 1; i < nickname.length(); i++) {
            sb.append("*");
        }
        return sb.toString();
    }

    /**
     * java.util.Date 转成 LocalDateTime
     * @param date
     * @return
     */
    public static LocalDateTime date2LocalDateTime(Date date) {
        return LocalDateTime.ofInstant(date.toInstant(), ZoneId.systemDefault());
    }

    /**
     * LocalDateTime 转成 java.util.Date
     * @param localDateTime
     * @return
     */
    public static Date localDateTime2Date(LocalDateTime localDateTime) {
        return Date.from(localDateTime.atZone(ZoneId.systemDefault()).toInstant());
    }

    /**
     * LocalDateTime 转成 时间戳（精确到毫秒）
     * @param localDateTime
     * @return
     */
    public static long localDateTime2Timestamp(LocalDateTime localDateTime) {
        return localDateTime.atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
    }

    /**
     * 精确到毫秒的时间戳 转成 LocalDateTime
     * @param timestamp
     * @return
     */
    public static LocalDateTime timestamp2LocalDateTime(long timestamp) {
        return LocalDateTime.ofInstant(Instant.ofEpochMilli(timestamp), ZoneId.systemDefault());
    }

    /**
     * 获取今日的开始时间戳（精确到毫秒）
     * @return
     */
    public static long getTodayStartTimestamp() {
        return LocalDate.now().atTime(LocalTime.MIN).atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
    }

    /**
     * 获取今日的结束时间戳（精确到毫秒）
     * @return
     */
    public static long getTodayEndTimestamp() {
        return LocalDate.now().atTime(LocalTime.MAX).atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
    }
}
