package cn.com.duiba.projectx.sdk.component.enums;

import cn.com.duiba.wolf.utils.DateUtils;

import java.time.DayOfWeek;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.temporal.TemporalAdjusters;
import java.util.Date;

/**
 * @program: projectx
 * @description: 组件限制 - 时间粒度枚举类
 * @author: Simba
 * @create: 2021-02-24 13:50
 **/
public enum TimeUnitEnum {
    NO_LIMIT(0, "不限制"),

    DAY(1, "日"),

    WEEK(2, "周"),

    MONTH(3, "月"),

    FOREVER(4, "永久"),

    CUSTOM(5, "自定义"),

    ;

    private int type;

    private String desc;

    TimeUnitEnum(int type, String desc) {
        this.type = type;
        this.desc = desc;
    }

    public static TimeUnitEnum toEnum(int type) {
        for (TimeUnitEnum p : values()) {
            if (p.type == type) {
                return p;
            }
        }
        return null;
    }

    public int getType() {
        return type;
    }

    public String getDesc() {
        return desc;
    }


    /**
     * 获取限制的到期时长
     *
     * ! 不支持自定义周期的计算
     *
     * @return
     */
    public Long getNowToEndTime() {
        Date now = new Date();
        if (this == DAY) {
            return DateUtils.getDayEndTime(now).getTime()-now.getTime();
        } else if (this == WEEK) {
            return getWeekLastDay(now).getTime()-now.getTime();
        } else if (this == MONTH) {
            return getMonthLastDay(now).getTime()-now.getTime();
        } else if (this == FOREVER) {
            return DateUtils.daysAddOrSub(now,365*6).getTime()-now.getTime();
        } else {
            return null;
        }
    }


    private static Date getWeekLastDay(Date date) {
        LocalDate localDate = date.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
        LocalDateTime localDateTime = LocalDateTime.of(localDate.with(DayOfWeek.SUNDAY), LocalTime.MAX);
        return Date.from(localDateTime.atZone(ZoneId.systemDefault()).toInstant());
    }

    private static Date getMonthLastDay(Date date) {
        LocalDate localDate = date.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
        LocalDateTime lastDateTime = LocalDateTime.of(localDate.with(TemporalAdjusters.lastDayOfMonth()), LocalTime.MAX);
        return Date.from(lastDateTime.atZone(ZoneId.systemDefault()).toInstant());
    }


}
