package cn.com.duiba.quanyi.center.api.utils.wxcoupon;

import cn.com.duiba.quanyi.center.api.dto.coupon.autocreateext.DiscountAmountRuleDto;
import cn.com.duiba.quanyi.center.api.enums.coupon.autocreate.CashierFullCutPeriodTypeEnum;
import cn.com.duiba.quanyi.center.api.utils.PriceUtils;
import cn.com.duiba.quanyi.center.api.utils.StringUtil;
import cn.com.duiba.wolf.utils.DateUtils;
import com.alibaba.fastjson.JSON;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import javax.validation.constraints.NotBlank;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * @author xuzhigang
 * @date 2024/4/8 14:33
 **/
@Slf4j
public class WxCouponAutoCreateUtils {

    private static final int INIT_DAYS = 3;

    private static final int ONE_WEEK_DAYS = 7;

    private static final int ONE_YEAR_DAYS = 365;

    private static final String TIME_FORMAT_PATTERN = "^(?:[01]\\d|2[0-4]):[0-5]\\d$";

    /**
     * 根据活动预算计算实际金额
     * 公式：活动预算/活动周期 * 3（创建时预留3天的预算），且金额必须是面额的整数倍
     *
     * @param budgetAmount          活动总预算
     * @param couponAmount          面额
     * @param startDate             活动开始时间
     * @param endDate               活动结束时间
     * @param stockDistributePeriod 实际活动周期
     * @param dayReceiveLimit       每日领取上限
     * @param autoAddMoney          true-自动加款 false-不自动加款
     * @return 实际金额（分）
     */
    public static Long calTotalAmount(Long budgetAmount, Long couponAmount, String startDate, String endDate, Integer stockDistributePeriod, Integer dayReceiveLimit, boolean autoAddMoney) {
        // 不自动加款 直接取活动预算
        if (!autoAddMoney) {
            return budgetAmount;
        }
        // 计算初始化需要消耗的金额
        long initAmount = (budgetAmount / calDiffDays(startDate, endDate, stockDistributePeriod)) * INIT_DAYS;
        // 加款金额必须是面额的整数倍 取余向上取整
        initAmount = initAmount + (couponAmount - (initAmount % couponAmount));
        // 初始化预算与10倍面额取最大值，不能超过总预算
        if (dayReceiveLimit == null) {
            return Math.min(Math.max(initAmount, 10 * couponAmount), budgetAmount);
        }
        // 单日领取上限数量 * 面额 与 初始化预算取最大值，不能超过总预算
        return Math.min(Math.max(Math.max(initAmount, 10 * couponAmount), dayReceiveLimit * couponAmount), budgetAmount);
    }

    /**
     * 根据活动预算计算最大发放数量 公式：活动预算/活动周期 * 3 / 面额
     *
     * @param budgetAmount          活动总预算
     * @param couponAmount          面额
     * @param startDate             活动开始时间
     * @param endDate               活动结束时间
     * @param stockDistributePeriod 实际活动周期
     * @param dayReceiveLimit       每日领取上限
     * @param autoAddMoney          true-自动加款 false-不自动加款
     * @return 最大发放数量
     */
    public static Integer calMaxNum(Long budgetAmount, Long couponAmount, String startDate, String endDate, Integer stockDistributePeriod, Integer dayReceiveLimit, boolean autoAddMoney) {
        Long totalAmount = calTotalAmount(budgetAmount, couponAmount, startDate, endDate, stockDistributePeriod, dayReceiveLimit, autoAddMoney);
        return Math.toIntExact(totalAmount.intValue() / couponAmount);
    }

    private static Integer calDiffDays(String startDate, String endDate, Integer stockDistributePeriod) {
        if (stockDistributePeriod != null) {
            return stockDistributePeriod;
        }
        return DateUtils.daysBetween(DateUtils.getDayDate(startDate), DateUtils.getDayDate(endDate)) + 1;
    }

    /**
     * 活动名称是否非法（活动名称不能超过9个字符）
     *
     * @param stockName 活动名称
     * @return true-合法 false-非法
     */
    public static boolean isValidStockName(String stockName) {
        return stockName != null && strLength(stockName) <= 18;
    }

    private static int strLength(@NotBlank String value) {
        int valueLength = 0;
        String chinese = "[\u4e00-\u9fa5]";
        for (int i = 0; i < value.length(); i++) {
            String temp = value.substring(i, i + 1);
            if (temp.matches(chinese)) {
                valueLength += 2;
            } else {
                valueLength += 1;
            }
        }
        return valueLength;
    }

    public static boolean oneOrTwo(Integer num) {
        return num != null && (num == 1 || num == 2);
    }

    public static boolean zeroOrOne(Integer num) {
        return num != null && (num == 0 || num == 1);
    }

    public static boolean isValidThemeColor(Integer themeColor) {
        return themeColor != null && themeColor >= 1 && themeColor <= 10;
    }

    public static boolean isValidCardBin(@NotBlank String cardBinList) {
        List<String> cardBins = StringUtil.splitStringWithComma(cardBinList);
        if (cardBins.size() > 100) {
            return false;
        }
        for (String cardBin : cardBins) {
            if (cardBin.length() < 6 || cardBin.length() > 13) {
                return false;
            }
        }
        return true;
    }

    public static boolean isValidBudgetAmount(String budgetAmount, Long discountAmount) {
        if (StringUtils.isBlank(budgetAmount)) {
            return false;
        }
        Long budget = PriceUtils.convertY2F(budgetAmount);
        return budget % discountAmount == 0;
    }

    /**
     * 转换随机立减规则对象
     *
     * @param discountAmountRules 随机减价规则（优惠方式为随机立减时填写，格式10&1-8,10&1-9）
     * @return 转换随机立减规则
     */
    public static List<DiscountAmountRuleDto> parseDiscountAmountRules(String discountAmountRules) {
        if (StringUtils.isBlank(discountAmountRules)) {
            return null;
        }
        List<String> ruleStrList = StringUtil.splitStringWithComma(discountAmountRules);
        return ruleStrList.stream().map(ruleStr -> {
            DiscountAmountRuleDto dto = new DiscountAmountRuleDto();
            String[] ruleArr = ruleStr.split("&");
            if (ruleArr.length != 2) {
                log.info("parseDiscountAmountRules not valid ruleStr={}", ruleStr);
                return null;
            }
            dto.setAmount(PriceUtils.convertY2F(ruleArr[0]));
            String[] rangeArr = ruleArr[1].split("-");
            if (rangeArr.length != 2) {
                log.info("parseDiscountAmountRules not valid rangeArr={}", ruleArr[1]);
                return null;
            }
            dto.setStartRange(PriceUtils.convertY2F(rangeArr[0]));
            dto.setEndRange(PriceUtils.convertY2F(rangeArr[1]));
            return dto;
        }).filter(Objects::nonNull).collect(Collectors.toList());
    }

    public static Pair<Long, Long> getAmountRange(String discountAmountRules) {
        List<DiscountAmountRuleDto> ruleList = parseDiscountAmountRules(discountAmountRules);
        if (CollectionUtils.isEmpty(ruleList)) {
            return Pair.of(0L, 0L);
        }
        long min = ruleList.stream().mapToLong(DiscountAmountRuleDto::getStartRange).min().orElse(0L);
        long max = ruleList.stream().mapToLong(DiscountAmountRuleDto::getEndRange).max().orElse(0L);
        return Pair.of(min, max);
    }

    public static boolean isValidDiscountAmountRules(String discountAmountRules) {
        try {
            List<DiscountAmountRuleDto> ruleList = parseDiscountAmountRules(discountAmountRules);
            if (CollectionUtils.isEmpty(ruleList) || ruleList.size() > 5) {
                log.info("isValidDiscountAmountRules not valid ruleList size is empty or more then 5 discountAmountRules={}", discountAmountRules);
                return false;
            }
            // 预估名额 不能小于5
            long totalNum = 0;
            for (DiscountAmountRuleDto rule : ruleList) {
                long amount = rule.getAmount();
                Long startRange = rule.getStartRange();
                Long endRange = rule.getEndRange();
                if (amount <= 0 || startRange <= 0 || endRange <= 0 || endRange <= startRange || endRange >= amount) {
                    log.info("isValidDiscountAmountRules not valid amount startRange endRange is not valid discountAmountRule={}", JSON.toJSONString(rule));
                    return false;
                }
                totalNum += amount / ((startRange + endRange) / 2);
            }
            return totalNum >= 5;
        } catch (Exception e) {
            log.error("isValidDiscountAmountRules discountAmountRules={}", discountAmountRules, e);
            return false;
        }
    }

    public static boolean isValidIrregularPeriod(String validTimeRange, String validTimePeriod) {
        if (StringUtils.isBlank(validTimeRange)) {
            log.warn("isValidIrregularPeriod validTimeRange is blank! validTimeRange={} validTimePeriod={}", validTimeRange, validTimePeriod);
            return false;
        }
        List<String> timeRanges = StringUtil.splitStringWithComma(validTimeRange);
        if (org.apache.commons.collections.CollectionUtils.isEmpty(timeRanges)) {
            log.warn("isValidIrregularPeriod timeRanges is empty! validTimeRange={} validTimePeriod={}", validTimeRange, validTimePeriod);
            return false;
        }
        return isValidTimePeriod(validTimePeriod);
    }

    public static boolean isValidRulePeriod(String validWeekDay, String validTimePeriod) {
        if (StringUtils.isAnyBlank(validWeekDay, validTimePeriod)) {
            log.warn("isValidRulePeriod param is blank! validWeekDay={} validTimePeriod={}", validWeekDay, validTimePeriod);
            return false;
        }
        // 校验
        List<Integer> weekDays = StringUtil.splitIntegerWithComma(validWeekDay);
        if (org.apache.commons.collections.CollectionUtils.isEmpty(weekDays) || weekDays.size() != ONE_WEEK_DAYS) {
            log.warn("isValidRulePeriod validWeekDay is not valid! validWeekDay={}", validWeekDay);
            return false;
        }
        // 必须选中一天
        boolean mustSelectOne = weekDays.stream().anyMatch(weekDay -> weekDay == 1);
        if (!mustSelectOne) {
            log.warn("isValidRulePeriod validWeekDay is must select one! validWeekDay={}", validWeekDay);
            return false;
        }
        // 只能选择0或1
        boolean zeroOrOne = weekDays.stream().allMatch(weekDay -> weekDay == 1 || weekDay == 0);
        if (!zeroOrOne) {
            log.warn("isValidRulePeriod validWeekDay is must zero or one! validWeekDay={}", validWeekDay);
            return false;
        }
        return isValidTimePeriod(validTimePeriod);
    }

    private static boolean isValidTimePeriod(String validTimePeriod) {
        // 只能填3个时间
        List<String> validTimePeriods = StringUtil.splitStringWithComma(validTimePeriod);
        if (org.apache.commons.collections.CollectionUtils.isEmpty(validTimePeriods) || validTimePeriods.size() > 3) {
            log.warn("isValidTimePeriod validTimePeriods size must less 3! validTimePeriod={}", validTimePeriod);
            return false;
        }
        for (String timePeriod : validTimePeriods) {
            String[] split = StringUtils.split(timePeriod, "-");
            if (split.length != 2) {
                log.warn("isValidTimePeriod validTimePeriods split size must less 2! validTimePeriod={}", validTimePeriod);
                return false;
            }
            if (!isValidTime(split[0]) || !isValidTime(split[1])) {
                log.warn("isValidTimePeriod validTimePeriods time not valid! validTimePeriod={}", validTimePeriod);
                return false;
            }
        }
        return true;
    }

    private static boolean isValidTime(String time) {
        Pattern pattern = Pattern.compile(TIME_FORMAT_PATTERN);
        return pattern.matcher(time).matches();
    }

    public static boolean isValidPeriod(String beginTime, String endTime, Integer validPeriodType, String
            validTimeRange, String validTimePeriod, String validWeekDay) {
        if (validPeriodType == null) {
            return false;
        }
        try {
            Date now = DateUtils.getDayDate(new Date());
            Date beginDate = DateUtils.getDayDate(beginTime);
            if (beginDate.before(now)) {
                return false;
            }
            Date endDate = DateUtils.getDayDate(endTime);
            int diffDays = DateUtils.daysBetween(beginDate, endDate) + 1;
            // 活动周期不能超过365天
            if (diffDays > ONE_YEAR_DAYS || diffDays <= 0) {
                return false;
            }
            CashierFullCutPeriodTypeEnum periodTypeEnum = CashierFullCutPeriodTypeEnum.getByType(validPeriodType);
            if (periodTypeEnum == null) {
                log.warn("isValidPeriod periodTypeEnum not found! validPeriodType={}", validPeriodType);
                return false;
            }
            switch (periodTypeEnum) {
                case UNLIMITED:
                    return true;
                case RULE:
                    return WxCouponAutoCreateUtils.isValidRulePeriod(validWeekDay, validTimePeriod);
                case IRREGULAR:
                    return WxCouponAutoCreateUtils.isValidIrregularPeriod(validTimeRange, validTimePeriod);
                default:
                    return false;
            }
        } catch (Exception e) {
            log.error("isValidPeriod error!", e);
            return false;
        }
    }

    public static void main(String[] args) {
        System.out.println(isValidDiscountAmountRules("10&1-10,10&1-5"));
    }
}
