package cn.com.duiba.quanyi.center.api.utils.login;

import cn.com.duiba.quanyi.center.api.dto.login.LoginBean;
import cn.com.duiba.quanyi.center.api.dto.login.TokenKeyDto;
import cn.com.duiba.quanyi.center.api.remoteservice.login.RemoteTokenKeyService;
import cn.com.duiba.wolf.utils.BlowfishUtils;
import cn.com.duiba.wolf.utils.NumberUtils;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.github.benmanes.caffeine.cache.CacheLoader;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

import java.util.Date;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;

/**
 * 使用方法：以下两种方法任意一个即可
 * 1.在spring配置文件中声明:<bean id="tokenUtils" class="cn.com.duiba.quanyi.center.api.utils.login.TokenUtils"/>
 * 2.使用@Bean注解：
 *     @Bean
 *     public TokenUtils getTokenUtils() {
 *         return new TokenUtils();
 *     }
 * 实例化时会自动寻找RemoteTokenKeyService并注入，找不到则报错要求声明这个bean
 * 
 * @author dugq
 * @date 2020/12/28 8:49 下午
 */
@Slf4j
public class TokenUtils implements ApplicationContextAware, InitializingBean {

    private ApplicationContext applicationContext;
    
    private static RemoteTokenKeyService remoteTokenKeyService;

    private static final LoadingCache<Integer, Optional<TokenKeyDto>> cache = Caffeine.newBuilder().expireAfterWrite(1, TimeUnit.MINUTES).refreshAfterWrite(10, TimeUnit.SECONDS).maximumSize(100).build(buildCacheLoader(key -> loadTokenKey()));

    private static <K, V> CacheLoader<K, V> buildCacheLoader(@NonNull Function<K, V> function) {
        return key -> {
            try {
                return function.apply(key);
            } catch (Exception e) {
                log.error("key={}", key, e);
                return null;
            }
        };
    }

    private static Optional<TokenKeyDto> loadTokenKey() {
        TokenKeyDto tokenKey = remoteTokenKeyService.getTokenKey();
        return Optional.ofNullable(tokenKey);
    }

    private static final ObjectMapper objectMapper = new ObjectMapper();
    static {
        objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    }
    
    public static String encodeToken(LoginBean loginBean, String keyEncrypt, int days) {
        if (Objects.isNull(loginBean)){
            return null;
        }
        Date now = new Date();
        loginBean.setLoginTime(now.getTime());
        loginBean.setDisableTime(DateUtils.addDays(now, days).getTime());
        try {
            TokenKeyDto tokenKey = getTokenKey();
            if (tokenKey == null) {
                return encodeToken(loginBean, keyEncrypt);
            }
            String key = getKey(loginBean.getLoginTime(), tokenKey);
            if (StringUtils.isBlank(key)) {
                return encodeToken(loginBean, keyEncrypt);
            }
            return loginBean.getLoginTime() + "_" + encodeToken(loginBean, key);
        } catch (JsonProcessingException e) {
            log.warn("[Token], 加密token错误, loginBean={}", loginBean, e);
            return null;
        }
    }
    
    private static String encodeToken(LoginBean loginBean, String key) throws JsonProcessingException {
        return BlowfishUtils.encryptBlowfish(objectMapper.writeValueAsString(loginBean), key);

    }

    public static <T extends LoginBean> T decodeToken(String value, String keyEncrypt, Class<T> clazz) {
        if (StringUtils.isBlank(value)){
            return null;
        }
        try {
            TokenKeyDto tokenKey = getTokenKey();
            if (tokenKey == null) {
                return doDecodeToken(value, keyEncrypt, clazz);
            }
            Pair<Long, String> pair = parseValue(value);
            if (pair.getLeft() != null) {
                // 新token
                return decodeTokenNew(value, pair, tokenKey, clazz);
            }
            if (tokenKey.isAllSwitch()) {
                // 已全部切换，不再支持老的token
                log.info("[Token], not support old token, value={}", value);
                return null;
            }
            return doDecodeToken(value, keyEncrypt, clazz);
        }catch (Exception e){
            log.info("[Token], 解密token错误",e);
            return null;
        }
    }
    
    private static <T extends LoginBean> T decodeTokenNew(String value, Pair<Long, String> pair, TokenKeyDto tokenKey, Class<T> clazz) throws JsonProcessingException {
        String key = getKey(pair.getLeft(), tokenKey);
        T bean = doDecodeToken(pair.getRight(), key, clazz);
        if (bean == null) {
            return null;
        }
        if (!Objects.equals(pair.getLeft(), bean.getLoginTime())) {
            log.info("[Token], token error, value = {}", value);
            return null;
        }
        if (bean.getLoginTime() < tokenKey.getTime() && System.currentTimeMillis() > tokenKey.getOldKeyNotAccessibleTime()) {
            // 老的key生成的，并且当前时间已到老密钥不可访问时间
            log.info("[Token], token not accessible, value = {}", value);
            return null;
        }
        return bean;
    }
    
    private static Pair<Long, String> parseValue(String value) {
        int index = value.indexOf("_");
        if (index < 1) {
            return Pair.of(null, value);
        }
        String timeStr = value.substring(0, index);
        int timeLength = 13;
        if (!NumberUtils.isNumeric(timeStr) || timeStr.length() != timeLength) {
            return Pair.of(null, value);
        }
        return Pair.of(Long.parseLong(timeStr), value.substring(index + 1));
    }
    
    private static <T extends LoginBean> T  doDecodeToken(String value, String key, Class<T> clazz) throws JsonProcessingException {
        final String decodeStr = BlowfishUtils.decryptBlowfish(value, key);
        return objectMapper.readValue(decodeStr, clazz);
    }

    private static String getKey(long time, TokenKeyDto tokenKey) {
        if (time >= tokenKey.getTime()) {
            return tokenKey.getNewKey();
        }
        return tokenKey.getOldKey();
    }

    private static TokenKeyDto getTokenKey() {
        if (remoteTokenKeyService == null) {
            log.error("[Token], remoteTokenKeyService is null");
            return null;
        }
        Optional<TokenKeyDto> optional = cache.get(1);
        if (Objects.isNull(optional)) {
            log.warn("[Token], Optional is null ");
            return null;
        }
        TokenKeyDto tokenKey = optional.orElse(null);
        if (tokenKey == null) {
            log.warn("[Token], token key not config");
        }
        return tokenKey;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        RemoteTokenKeyService tokenKeyService = applicationContext.getBean(RemoteTokenKeyService.class);
        if(tokenKeyService == null){
            throw new IllegalStateException("there must exists a bean of class RemoteTokenKeyService(in quanyi-center)");
        }
        TokenUtils.remoteTokenKeyService = tokenKeyService;
        log.info("[Token], remoteTokenKeyService load");
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}
