package cn.com.duiba.quanyi.center.api.enums.goods;

import lombok.AllArgsConstructor;
import lombok.Getter;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author lizhi
 * @date 2023/8/7 16:26
 */
@Getter
@AllArgsConstructor
public enum GoodsOrderStatusEnum {
    //订单状态, 1-处理中，2-成功，3-失败, 4-异常
    /**
     * 发放页面展示：处理中
     * 领取页面展示：处理中
     */
    PROCESS(1, "处理中"),
    /**
     * 发放页面展示：领取成功
     * 领取页面展示：领取成功
     */
    SUCCESS(2, "成功"),
    /**
     * 明确的不能重试成功的错误，比如：用户领取次数超过上限等用户原因
     * 发放页面展示：领取失败
     * 领取页面展示：领取失败
     */
    FAIL(3, "失败，无法自动重试成功"),
    /**
     * 发放页面展示：领取中
     * 领取页面展示：领取中
     */
    EXCEPTION(4, "异常"),
    /**
     * 发放页面展示：待领取
     * 领取页面展示：待领取
     */
    NOT_RECEIVE(5, "待领取"),
    /**
     * 可重试的错误，比如：预算不足、熔断等
     * 发放页面展示：处理中
     * 领取页面展示：处理中
     */
    FAIL_CAN_RETRY(6, "失败，自动重试可能会成功"),
    /**
     * 发放页面展示：待领取
     * 领取页面展示：领取中
     */
    SHORT_URL_RECEIVE_PROCESS(7, "短链领取中"),
    /**
     * 发放页面展示：待领取
     * 领取页面展示：领取失败
     */
    SHORT_URL_RECEIVE_FAIL(8, "短链领取失败，无法自动重试成功"),
    /**
     * 发放页面展示：待领取
     * 领取页面展示：领取中
     */
    SHORT_URL_RECEIVE_FAIL_CAN_RETRY(9, "短链领取失败，自动重试可能会成功"),
    /**
     * 发放页面展示：待领取
     * 领取页面展示：领取中
     */
    SHORT_URL_RECEIVE_EXCEPTION(10, "短链领取异常"),

    /**
     * 核销
     */
    VERIFICATION(11, "已核销"),

    /**
     * 已失效
     * 已过期的虚拟充值商品
     * 主动作废的虚拟充值商品
     */
    EXPIRED(12, "已作废"),

    /**
     * 已领取
     * 红包
     */
    RECEIVED(13, "已领取"),

    /**
     * 已退款
     * 红包
     */
    REFUND(14, "已退款"),
    /**
     * 短链未领取导致过期
     */
    SHORT_URL_EXPIRE(15, "短链未领取过期"),

    // 实物状态-5X
    UNSHIPPED(51, "待发货"),
    SHIPPED(52, "待收货"),
    COMPLETED(53, "已签收"),
    CANCELED(54, "已取消"),
    RETURNED(55, "已退货"),
    
    
    CLOSE(99, "已关闭"),
    ;
    
    private final Integer status;
    
    private final String desc;
    
    private static final Map<Integer, GoodsOrderStatusEnum> ENUM_MAP = Collections.unmodifiableMap(Arrays.stream(values()).collect(Collectors.toMap(GoodsOrderStatusEnum::getStatus, Function.identity(), (v1, v2) -> v2)));
    
    /**
     * 根据枚举值获取枚举
     * @param status 枚举值
     * @return 枚举
     */
    public static GoodsOrderStatusEnum getByStatus(Integer status) {
        if (status == null) {
            return null;
        }
        return ENUM_MAP.get(status);
    }
    private static final Set<Integer> SHORT_URL_STATUS_SET = Collections.unmodifiableSet(new HashSet<>(Arrays.asList(
            NOT_RECEIVE.getStatus()
            , SHORT_URL_RECEIVE_PROCESS.getStatus()
            , SHORT_URL_RECEIVE_FAIL.getStatus()
            , SHORT_URL_RECEIVE_FAIL_CAN_RETRY.getStatus()
            , SHORT_URL_RECEIVE_EXCEPTION.getStatus()
            , SHORT_URL_EXPIRE.getStatus()
    )));
    public static boolean isShortUrlStatus(Integer status) {
        if (status == null) {
            return false;
        }
        return SHORT_URL_STATUS_SET.contains(status);
    }

    private static final Set<Integer> ALREADY_SUCCESS_STATUS_SET = Collections.unmodifiableSet(new HashSet<>(Arrays.asList(SUCCESS.status, VERIFICATION.status, EXPIRED.getStatus(), RECEIVED.getStatus(), REFUND.getStatus(),
            UNSHIPPED.getStatus(),SHIPPED.getStatus(),COMPLETED.getStatus(),CANCELED.getStatus(),RETURNED.getStatus())));
    
    
    public static boolean isAlreadySuccessStatus(Integer status) {
        if (status == null) {
            return false;
        }
        return ALREADY_SUCCESS_STATUS_SET.contains(status);
    }
    
    public static boolean isAlreadyClose(Integer status) {
        return CLOSE.getStatus().equals(status);
    }
    public static boolean isAlreadyShortExpire(Integer status) {
        return SHORT_URL_EXPIRE.getStatus().equals(status);
    }

    private static final Set<Integer> CAN_RETRY_GRANT_STATUS_SET = Collections.unmodifiableSet(new HashSet<>(Arrays.asList(
            PROCESS.getStatus()
            , FAIL.getStatus()
            , EXCEPTION.getStatus()
            , FAIL_CAN_RETRY.getStatus()
    )));

    private static final Set<Integer> CAN_RETRY_RECEIVE_STATUS_SET = Collections.unmodifiableSet(new HashSet<>(Arrays.asList(
            SHORT_URL_RECEIVE_PROCESS.getStatus()
            , SHORT_URL_RECEIVE_FAIL_CAN_RETRY.getStatus()
            , SHORT_URL_RECEIVE_FAIL.getStatus()
            , SHORT_URL_RECEIVE_EXCEPTION.getStatus()
    )));

    public static boolean canRetryGrant(Integer status) {
        return CAN_RETRY_GRANT_STATUS_SET.contains(status);
    }

    public static boolean canRetryReceive(Integer status) {
        return CAN_RETRY_RECEIVE_STATUS_SET.contains(status);
    }
}
