package cn.com.duiba.quanyi.center.api.utils.alipay;

import cn.com.duiba.quanyi.center.api.dto.coupon.autocreateext.alipay.AlipayActivityAutoCreateExtInfoDto;
import cn.com.duiba.quanyi.center.api.dto.coupon.autocreateext.alipay.CardPayCutAutoCreateExtInfoDto;
import cn.com.duiba.quanyi.center.api.enums.coupon.autocreate.alipay.GamePlaySingleTypeEnum;
import cn.com.duiba.quanyi.center.api.utils.StringUtil;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author xuzhigang
 * @date 2024/7/30 10:49
 **/
@Slf4j
public class AlipayActivityAutoCreateUtils {
    private static final String JOINER = "-";

    /**
     * 银行名称-银行卡类型（1-借记卡 2-信用卡）
     * 格式：xx银行-1,xx银行-2，多个银行英文逗号分割
     */
    public static List<AlipayActivityAutoCreateExtInfoDto.Bank> parseBankList(String bankListStr) {
        if (StringUtils.isBlank(bankListStr)) {
            return Collections.emptyList();
        }
        List<String> bankStrList = StringUtil.splitStringWithComma(bankListStr);
        return bankStrList.stream().map(AlipayActivityAutoCreateUtils::parseBank).filter(Objects::nonNull).collect(Collectors.toList());
    }

    /**
     * 银行名称-银行卡类型
     */
    public static AlipayActivityAutoCreateExtInfoDto.Bank parseBank(String bankStr) {
        List<String> bankNameAndBankCardType = StringUtil.splitStringWithSeparator(bankStr, JOINER);
        if (CollectionUtils.isEmpty(bankNameAndBankCardType)) {
            return null;
        }
        AlipayActivityAutoCreateExtInfoDto.Bank bank = new AlipayActivityAutoCreateExtInfoDto.Bank();
        bank.setName(bankNameAndBankCardType.get(0));
        bank.setCardType(Integer.valueOf(bankNameAndBankCardType.get(1)));
        return bank;
    }

    /**
     * 可用时间设置 每周指定
     *
     * @param weeks 0,1,0,1,0,1,0
     * @param times 00:00:00-01:00:00
     */
    public static AlipayActivityAutoCreateExtInfoDto.EnableTime parseEnableTime(String weeks, String times) {
        if (StringUtils.isAnyBlank(weeks, times)) {
            return null;
        }
        AlipayActivityAutoCreateExtInfoDto.EnableTime enableTime = new AlipayActivityAutoCreateExtInfoDto.EnableTime();
        enableTime.setWeeks(StringUtil.splitIntegerWithComma(weeks));
        enableTime.setTimes(StringUtil.splitStringWithComma(times));
        return enableTime;
    }

    /**
     * 预警设置
     *
     * @param alarmAmount 预警金额
     * @param alarmPhone 预警手机号 多个逗号分割
     */
    public static AlipayActivityAutoCreateExtInfoDto.Alarm parseAlarm(String alarmAmount, String alarmPhone) {
        if (StringUtils.isAnyBlank(alarmAmount, alarmPhone)) {
            return null;
        }
        AlipayActivityAutoCreateExtInfoDto.Alarm alarm = new AlipayActivityAutoCreateExtInfoDto.Alarm();
        alarm.setAmount(alarmAmount);
        alarm.setPhoneList(StringUtil.splitStringWithComma(alarmPhone));
        return alarm;
    }

    /**
     * 每日预算上限
     * @param dayBudgetType 每日预算上限类型
     * @param dayBudgetAmount 每日预算上限（元/份）
     * @return 每日预算上限
     */
    public static CardPayCutAutoCreateExtInfoDto.DayBudget parseDayBudget(Integer dayBudgetType, String dayBudgetAmount) {
        if (dayBudgetType == null || StringUtils.isBlank(dayBudgetAmount)) {
            return null;
        }
        CardPayCutAutoCreateExtInfoDto.DayBudget dayBudget = new CardPayCutAutoCreateExtInfoDto.DayBudget();
        dayBudget.setAmount(dayBudgetAmount);
        dayBudget.setType(dayBudgetType);
        return dayBudget;
    }

    /**
     * 单一门槛优惠
     *
     * @param gamePlaySingleType 单一玩法类型 {@link GamePlaySingleTypeEnum}
     * @param gamePlaySingleMaxAmount 单一玩法最大优惠金额
     * @param gamePlaySingleThreshold 单一玩法门槛
     * @param gamePlaySingleAveragePrice 单一玩法优惠金额均价
     * @param gamePlaySingleAverage 单一玩法优惠区间
     * @param gamePlaySingleRule 单一玩法规则
     * @return 单一门槛优惠
     */
    public static CardPayCutAutoCreateExtInfoDto.GamePlaySingle parseGamePlaySingle(Integer gamePlaySingleType, String gamePlaySingleMaxAmount, String gamePlaySingleThreshold, String gamePlaySingleAveragePrice, String gamePlaySingleAverage, String gamePlaySingleRule) {
        if (GamePlaySingleTypeEnum.getByType(gamePlaySingleType) == null) {
            return null;
        }
        CardPayCutAutoCreateExtInfoDto.GamePlaySingle gamePlaySingle = new CardPayCutAutoCreateExtInfoDto.GamePlaySingle();
        gamePlaySingle.setType(gamePlaySingleType);
        gamePlaySingle.setThreshold(gamePlaySingleThreshold);
        if (GamePlaySingleTypeEnum.FIXED_REDUCE.getType().equals(gamePlaySingleType)) {
            gamePlaySingle.setAmount(gamePlaySingleRule);
        }
        if (GamePlaySingleTypeEnum.DISCOUNT_REDUCE.getType().equals(gamePlaySingleType)) {
            gamePlaySingle.setRule(gamePlaySingleRule);
            gamePlaySingle.setMaxAmount(gamePlaySingleMaxAmount);
        }
        if (GamePlaySingleTypeEnum.SMART_REDUCE.getType().equals(gamePlaySingleType)) {
            gamePlaySingle.setAverage(StringUtil.splitStringWithSeparator(gamePlaySingleAverage, JOINER));
            gamePlaySingle.setAveragePrice(gamePlaySingleAveragePrice);
        }
        return gamePlaySingle;
    }

    /**
     * 阶梯门槛优惠
     *
     * @param gamePlayStepRule 阶梯门槛优惠规则: x-y-1-1，满x元-y元-优惠类型1减至2定额立减-优惠规则，多组根据英文逗号分割
     * @param gamePlayStepMaxAmount 最高优惠金额（元）
     * @return 阶梯门槛优惠
     */
    public static CardPayCutAutoCreateExtInfoDto.GamePlayStep parseGamePlayStep(String gamePlayStepRule, String gamePlayStepMaxAmount) {
        if (StringUtils.isAnyBlank(gamePlayStepRule, gamePlayStepMaxAmount)) {
            return null;
        }
        CardPayCutAutoCreateExtInfoDto.GamePlayStep gamePlayStep = new CardPayCutAutoCreateExtInfoDto.GamePlayStep();
        gamePlayStep.setMaxAmount(gamePlayStepMaxAmount);
        gamePlayStep.setArray(StringUtil.splitStringWithComma(gamePlayStepRule).stream().map(rule -> {
            List<String> rules = StringUtil.splitStringWithSeparator(rule, JOINER);
            CardPayCutAutoCreateExtInfoDto.GamePlayRule gamePlayRule = new CardPayCutAutoCreateExtInfoDto.GamePlayRule();
            gamePlayRule.setThresholdArea(Lists.newArrayList(rules.get(0), rules.get(1)));
            gamePlayRule.setType(Integer.parseInt(rules.get(2)));
            gamePlayRule.setRule(rules.get(3));
            return gamePlayRule;
        }).collect(Collectors.toList()));
        return gamePlayStep;
    }

    /**
     * 频率限制
     *
     * @param timeLimitStr 频率限制 x-y，x表示周期 1-自然日 2-自然周 3-自然月 4-自然年，y表示次数，如果多种限制根据英文逗号分割
     * @return 频率限制
     */
    public static List<CardPayCutAutoCreateExtInfoDto.TimeLimit> parseTimeLimit(String timeLimitStr) {
        if (StringUtils.isBlank(timeLimitStr)) {
            return null;
        }
        List<String> timeLimits = StringUtil.splitStringWithComma(timeLimitStr);
        if (CollectionUtils.isEmpty(timeLimits)) {
            return null;
        }
        return timeLimits.stream().map(timeLimitSingleStr -> {
            CardPayCutAutoCreateExtInfoDto.TimeLimit timeLimit = new CardPayCutAutoCreateExtInfoDto.TimeLimit();
            List<Integer> timeLimitSingle = StringUtil.splitIntegerWithSeparator(timeLimitSingleStr, JOINER);
            timeLimit.setType(timeLimitSingle.get(0));
            timeLimit.setNum(timeLimitSingle.get(1));
            return timeLimit;
        }).collect(Collectors.toList());
    }
}
