package cn.com.duiba.quanyi.center.api.enums.coupon;

import lombok.AllArgsConstructor;
import lombok.Getter;
import org.apache.commons.lang3.tuple.Pair;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Objects;

/**
 * 美团券批次状态
 * 1-立减券
 * 3-满减券
 * 4-折扣券
 * 7-口价券
 */
@Getter
@AllArgsConstructor
public enum MeiTuanCouponTypeEnum {
    DISCOUNT_COUPON(1, "立减券"),
    FULL_REDUCTION_COUPON(3, "满减券"),
    PERCENTAGE_DISCOUNT_COUPON(4, "折扣券"),
    FIXED_PRICE_COUPON(7, "口价券"),
    ;

    private int code;
    private String description;

    /**
     * 根据code获取枚举对象
     *
     * @param code
     * @return
     */
    public static MeiTuanCouponTypeEnum getByCode(int code) {
        for (MeiTuanCouponTypeEnum status : values()) {
            if (status.code == code) {
                return status;
            }
        }
        return null;
    }

    /**
     * 根据code,判断面额类型（定额，不定额）
     *
     * @param code 券类型
     * @return 1-定额券，2-不定额券
     */
    public static Integer isAmountType(int code) {
        if (DISCOUNT_COUPON.getCode() == code || FULL_REDUCTION_COUPON.getCode() == code) {
            return AmountTypeEnum.FIXED.getType();
        }
        if (PERCENTAGE_DISCOUNT_COUPON.getCode() == code || FIXED_PRICE_COUPON.getCode() == code) {
            return AmountTypeEnum.UNFIXED.getType();
        }
        return null;
    }

    /**
     * 根据券类型，计算优惠金额范围（仅计算不固定面额的）
     *
     * @param couponType  券类型
     * @param quotaValue  优惠券额度
     * @param reduceValue 满减金额
     * @param discountValue 折扣值 8.8折，5折
     * @return
     */
    public static Pair<Long, Long> isMinAmountAndMaxAmount(Integer couponType, Long quotaValue, Long reduceValue, Long discountValue) {

        if (Objects.isNull(couponType)) {
            return Pair.of(0L, 0L);
        }
        MeiTuanCouponTypeEnum byCode = getByCode(couponType);
        if (Objects.isNull(byCode)) {
            return Pair.of(0L, 0L);
        }
        switch (byCode) {
            case PERCENTAGE_DISCOUNT_COUPON:
                Long discountAmount = getDiscountAmount(quotaValue, discountValue);
                return Pair.of(discountAmount, reduceValue);
            case FIXED_PRICE_COUPON:
                Long buyItNowAmount = getBuyItNowAmount(quotaValue, reduceValue);
                return Pair.of(buyItNowAmount, reduceValue);
            default:
                return Pair.of(0L, 0L);
        }
    }

    /**
     * 计算折扣优惠价格
     *
     * @param quotaValue    优惠券额度
     * @param discountValue 折扣值 8.8折，5折
     * @return
     */
    private static Long getDiscountAmount(Long quotaValue, Long discountValue) {
        // 参数校验
        if (quotaValue == null || quotaValue <= 0 || discountValue == null || discountValue <= 0) {
            throw new IllegalArgumentException("Invalid parameters: quotaValue and discountValue must be positive.");
        }
        BigDecimal amount = new BigDecimal(quotaValue);
        BigDecimal rate = new BigDecimal(discountValue);

        BigDecimal discount = amount.multiply(rate).divide(new BigDecimal(10),2, RoundingMode.HALF_UP);
        return amount.subtract(discount).longValue();
    }

    /**
     * 计算一口价优惠价格
     *
     * @param quotaValue  优惠券额度
     * @param reduceValue 满减金额
     * @return
     */
    private static Long getBuyItNowAmount(Long quotaValue, Long reduceValue) {
        // 参数校验
        if (quotaValue == null || reduceValue == null) {
            throw new IllegalArgumentException("quotaValue and reduceValue cannot be null");
        }
        if (quotaValue < 0 || reduceValue < 0) {
            throw new IllegalArgumentException("quotaValue and reduceValue must be non-negative");
        }
        BigDecimal amount = new BigDecimal(quotaValue);
        BigDecimal reduce = new BigDecimal(reduceValue);
        return amount.subtract(reduce).longValue();
    }
}
