package cn.com.duiba.quanyi.center.api.enums.api;

import lombok.AllArgsConstructor;
import lombok.Getter;

import java.util.Arrays;
import java.util.Collections;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * API订单状态
 * 
 * @author lizhi
 * @date 2023/12/21 16:06
 */
@Getter
@AllArgsConstructor
public enum ApiOrderStatusEnum {

    /**
     * 支持该状态的商品类型：  
     *     ○ 所有商品
     */
    PROCESS(1,"处理中"),

    /**
     * 支持该状态的商品类型： 
     *     ○ 卡密
     *     ○ 链接券/输入账号 发放类型
     */
    ISSUED(2,"已发放，针对卡密、链接券等商品有该状态，支持该状态的商品类型"),

    /**
     * 支持该状态的商品类型：  
     *     ○ 所有商品
     * 
     */
    FAIL(3,"失败，一般是外部原因，比如充值账号不正确"),

    /**
     * 支持该状态的商品类型：   
     *     ○ 非卡密类型商品
     */
    RECEIVED(4,"已到账，一般是直冲类商品，并非所有直冲商品都有该状态"),

    /**
     * 支持该状态的商品类型： 
     *     ○ 微信立减金
     *     ○ 支付宝立减金
     *     ○ 霸王茶姬
     *     ○ 奈雪直冲
     *     ○ 古茗代金券
     */
    VERIFICATION(5,"已核销"),

    /**
     * 支持该状态的商品类型： 
     *     ○ 霸王茶姬
     */
    EXPIRED(6,"已过期"),

    /**
     * 支持该状态的商品类型： 
     *     ○ 微信红包
     */
    RED_PACK_RECEIVED(7, "已领取"),

    /**
     * 支持该状态的商品类型： 
     *     ○ 微信红包
     *     ○ 支付宝立减金 
     *     ○ 古茗代金券
     */
    REFUND(8, "已退款"),

    /**
     * 支持该状态的商品类型： 
     *     ○ 实物
     */
    UNSHIPPED(9, "待发货"),

    /**
     * 支持该状态的商品类型： 
     *     ○ 实物
     */
    SHIPPED(10, "待收货"),

    /**
     * 支持该状态的商品类型： 
     *     ○ 实物
     */
    COMPLETED(11, "已签收"),

    /**
     * 支持该状态的商品类型： 
     *     ○ 实物
     */
    CANCELED(12, "已取消"),

    /**
     * 支持该状态的商品类型： 
     *     ○ 实物
     */
    RETURNED(13, "已退货"),

    /**
     * 支持该状态的商品类型： 
     *     ○ 非实物
     */
    CLOSE(14, "已关闭"),

    /**
     * 支持该状态的商品类型： 
     *     ○ 非实物、非卡密
     */
    INVALID(15, "已作废"),
    ;
    
    private final Integer status;
    
    private final String desc;

    private static final Map<Integer, ApiOrderStatusEnum> ENUM_MAP = Collections.unmodifiableMap(Arrays.stream(values()).collect(Collectors.toMap(ApiOrderStatusEnum::getStatus, Function.identity(), (v1, v2) -> v2)));

    /**
     * 根据枚举值获取枚举
     * @param status 枚举值
     * @return 枚举
     */
    public static ApiOrderStatusEnum getByStatus(Integer status) {
        //默认返回处理中
        if (status == null) {
            return PROCESS;
        }
        return ENUM_MAP.get(status);
    }

}
