package cn.com.duiba.quanyi.center.api.enums.coupon;

import cn.com.duiba.quanyi.center.api.dto.alipay.AlipayDiscountIntelligentAlgorithmJsonDto;
import cn.com.duiba.quanyi.center.api.dto.alipay.AlipayDiscountJsonDto;
import cn.com.duiba.quanyi.center.api.dto.alipay.AlipayDiscountMultiStagedDiscountJsonDto;
import cn.com.duiba.quanyi.center.api.dto.alipay.AlipayDiscountRandomJsonDto;
import cn.com.duiba.quanyi.center.api.dto.alipay.StagedThresholdInfo;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import lombok.AllArgsConstructor;
import lombok.Getter;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 立减类型
 * reduce:满立减(满5元减3元)
 * 定额立减(定额满减): single
 * 折扣立减: discount
 * 随机金额立减(随机满减): random
 * 智能立减: intelligentAlgorithm
 * 多阶梯立减: multiStagedDiscount
 *
 * @see <a>https://opendocs.alipay.com/apis/07yvpj</>
 */
@Getter
@AllArgsConstructor
public enum AlipayDiscountTypeEnum {
    REDUCE("reduce", "满立减"),
    SINGLE("single", "定额立减(定额满减)"),
    DISCOUNT("discount", "折扣立减"),
    RANDOM("random", "随机金额立减(随机满减)"),
    INTELLIGENT_ALGORITHM("intelligentAlgorithm", "智能立减"),
    MULTI_STAGED_DISCOUNT("multiStagedDiscount", "多阶梯立减"),
    ;

    private final String type;

    private final String desc;

    public static Integer getByType(String discountType) {
        if (REDUCE.getType().equals(discountType) || SINGLE.getType().equals(discountType)) {
            return AmountTypeEnum.FIXED.getType();
        }
        if (DISCOUNT.getType().equals(discountType) || RANDOM.getType().equals(discountType) || INTELLIGENT_ALGORITHM.getType().equals(discountType) || MULTI_STAGED_DISCOUNT.getType().equals(discountType)) {
            return AmountTypeEnum.UNFIXED.getType();
        }
        return null;
    }

    /**
     * 获取不固定金额，最小和最大金额
     * @param discountType 优惠类型 以分为单位
     * @param discountValue 优惠值 以分为单位
     * @param discountThresholdAmt 优惠立减门槛金额 以分为单位
     * @return 左边：最小金额，右边：最大金额
     */
    public static Pair<Long, Long> getMinAmountAndMaxAmount(String discountType, String discountValue,Long discountThresholdAmt) {
        if (StringUtils.isEmpty(discountType) || StringUtils.isEmpty(discountValue)) {
            return Pair.of(0L, 0L);
        }
        if (DISCOUNT.getType().equals(discountType)){
            // 先将整个 JSON 字符串解析为AlipayDiscountJsonDto类型
            AlipayDiscountJsonDto jsonDto = JSON.parseObject(discountValue, AlipayDiscountJsonDto.class);
            if (jsonDto != null) {
                Long ratio = jsonDto.getRatio();
                Long minAmount = getRatioAmount(discountThresholdAmt, ratio);
                Long maxAmount = jsonDto.getMaxAmount();
                return Pair.of(minAmount, maxAmount);
            }
        }
        if (RANDOM.getType().equals(discountType)){
            // 先将整个 JSON 字符串解析为 JSONObject
            JSONObject jsonObject = JSONObject.parseObject(discountValue);
            // 从 JSONObject 中提取 "rangeInfoList" 数组
            JSONArray jsonArray = jsonObject.getJSONArray("rangeInfoList");
            // 将 jsonArray 转换为 List<AlipayDiscountRandomJsonDto>
            List<AlipayDiscountRandomJsonDto> rangeInfoList = jsonArray.toJavaList(AlipayDiscountRandomJsonDto.class);

            if (CollectionUtils.isNotEmpty(rangeInfoList)) {
                AlipayDiscountRandomJsonDto alipayDiscountRandomJsonDto = rangeInfoList.get(0);
                Long minAmount = alipayDiscountRandomJsonDto.getMinAmount();
                Long maxAmount = alipayDiscountRandomJsonDto.getMaxAmount();
                return Pair.of(minAmount, maxAmount);
            }
        }
        if (INTELLIGENT_ALGORITHM.getType().equals(discountType)){
            // 解析 JSON 字符串
            AlipayDiscountIntelligentAlgorithmJsonDto jsonDto = JSON.parseObject(discountValue, AlipayDiscountIntelligentAlgorithmJsonDto.class);
            if (jsonDto != null) {
                Long minAmount = jsonDto.getMinAmount();
                Long maxAmount = jsonDto.getMaxAmount();
                return Pair.of(minAmount, maxAmount);
            }
        }
        if (MULTI_STAGED_DISCOUNT.getType().equals(discountType)){
            // 解析 JSON 字符串
            AlipayDiscountMultiStagedDiscountJsonDto jsonDto = JSON.parseObject(discountValue, AlipayDiscountMultiStagedDiscountJsonDto.class);
            Long minAmount = 1L;
            List<StagedThresholdInfo> stagedDiscountList = jsonDto.getStagedDiscountList();
            if (CollectionUtils.isNotEmpty(stagedDiscountList)){
                Set<Long> collect = stagedDiscountList.stream().map(stagedThresholdInfo ->{
                    String priceCalcType = stagedThresholdInfo.getPriceCalcType();
                    if ("REDUCE_TO_AMOUNT".equals(priceCalcType)){
                        return stagedThresholdInfo.getMinThresholdAmount()-stagedThresholdInfo.getReduceAmount();
                    }
                    return stagedThresholdInfo.getReduceAmount();
                }).collect(Collectors.toSet());
                minAmount = collect.stream().min(Long::compareTo).orElse(1L);
            }
            Long maxAmount = jsonDto.getMaxReduceAmount();
            return Pair.of(minAmount, maxAmount);
        }
        return Pair.of(0L, 0L);
    }

    /**
     * 获取比率金额
     *
     * @param discountThresholdAmt 优惠立减门槛金额 以分为单位
     * @param ratio                比率
     * @return
     */
    private static Long getRatioAmount(Long discountThresholdAmt, Long ratio) {
        if (discountThresholdAmt == null || ratio == null) {
            throw new IllegalArgumentException("Input parameters cannot be null");
        }

        if (ratio == 0) {
            throw new IllegalArgumentException("Ratio cannot be zero");
        }

        BigDecimal amount = new BigDecimal(discountThresholdAmt);
        BigDecimal rate = new BigDecimal(ratio);

        BigDecimal discount = amount.multiply(rate).divide(new BigDecimal(100), 2, RoundingMode.HALF_UP);
        return amount.subtract(discount).longValue();
    }

}
