package cn.com.duiba.quanyi.center.api.enums.coupon;

import lombok.AllArgsConstructor;
import lombok.Getter;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 支付宝立减金后台类型
 *
 * @see <a>https://p.alipay.com/page/dtbank/manage</a>
 */
@Getter
@AllArgsConstructor
public enum AlipayActivityTypeEnum {
    CARD_PAY_CUT(1, "银行卡支付立减", false),
    GENERAL_COUPON(2, "通用渠道红包", true),
    FIRST_BIND(3, "首绑有礼", false),
    MANY_CARD_SWIPING(4, "多刷有礼", false),
    ;

    /**
     * 批次类型
     */
    private final Integer type;

    /**
     * 批次类型名称
     */
    private final String typeName;

    /**
     * 是否支持API发放
     */
    private final boolean supportApi;
    
    private static final Map<Integer, AlipayActivityTypeEnum> ENUM_MAP = Collections.unmodifiableMap(Arrays.stream(values()).collect(Collectors.toMap(AlipayActivityTypeEnum::getType, Function.identity(), (v1, v2) -> v2)));

    /**
     * 根据枚举值获取枚举
     *
     * @param type 枚举值
     * @return 枚举
     */
    public static AlipayActivityTypeEnum getByType(Integer type) {
        if (type == null) {
            return null;
        }
        return ENUM_MAP.get(type);
    }

    /**
     * 根据枚举值获取枚举
     *
     * @param type 枚举值
     * @return 枚举
     */
    public static String getNameByType(Integer type) {
        AlipayActivityTypeEnum typeEnum = getByType(type);
        if (typeEnum == null) {
            return "未知";
        }
        return typeEnum.getTypeName();
    }

    /**
     * 是否支持api发放
     *
     * @param type 枚举值
     * @return true-是，false-否
     */
    public static boolean isSupportApi(Integer type) {
        AlipayActivityTypeEnum typeEnum = getByType(type);
        if (typeEnum == null) {
            return false;
        }
        return typeEnum.isSupportApi();
    }

    public static List<Integer> getSupportApiList() {
        return Arrays.stream(values()).filter(AlipayActivityTypeEnum::isSupportApi).map(AlipayActivityTypeEnum::getType).collect(Collectors.toList());
    }
}
