package cn.com.duiba.quanyi.center.api.redis;

import cn.com.duiba.wolf.redis.RedisAtomicClient;
import cn.com.duiba.wolf.redis.RedisLock;
import cn.com.duiba.wolf.utils.UUIDUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.connection.RedisStringCommands;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.data.redis.core.types.Expiration;

/**
 * @Resource(name="stringRedisTemplate")
 * private QyRedisAtomicClient qyRedisAtomicClient;
 *
 * 提供Redis一些不直接支持的原子性的操作,很多实现采用了lua脚本
 * Created by wenqi.huang on 2017/3/6.
 */
@Slf4j
public class QyRedisAtomicClient extends RedisAtomicClient {

    private final StringRedisTemplate stringRedisTemplate;
    private static final String COMPARE_AND_DELETE =
            "if redis.call('get',KEYS[1]) == ARGV[1]\n" +
            "then\n" +
            "    return redis.call('del',KEYS[1])\n" +
            "else\n" +
            "    return 0\n" +
            "end";

    public QyRedisAtomicClient(RedisTemplate redisTemplate){
        super(redisTemplate);
        this.stringRedisTemplate = new StringRedisTemplate();
        this.stringRedisTemplate.setConnectionFactory(redisTemplate.getConnectionFactory());
        this.stringRedisTemplate.afterPropertiesSet();
    }

    /**
     * 获取redis的自动续期分布式锁，内部实现使用了redis的setnx。只会尝试一次，如果锁定失败返回null，如果锁定成功则返回RedisLock对象，调用方需要调用RedisLock.unlock()方法来释放锁.
     * <br/>
     * 注意：请勿在数据库事务内获取和释放分布式锁，否则由于事务可见性的原因，可能会导致数据错乱。
     * <br/>使用方法：
     * <pre>
     * RedisLock lock = redisAtomicClient.getAutomaticRenewalLock(key, 2);
     * if(lock != null){
     *      try {
     *          //lock succeed, do something
     *      }finally {
     *          lock.unlock();
     *      }
     * }
     * </pre>
     * 由于RedisLock实现了AutoCloseable,所以可以使用更简介的使用方法:
     * <pre>
     *  try(RedisLock lock = redisAtomicClient.getAutomaticRenewalLock(key, 2)) {
     *      if (lock != null) {
     *          //lock succeed, do something
     *      }
     *  }
     * </pre>
     * @param key 要锁定的key
     * @param expireSeconds key的失效时间
     * @param interval 续期的间隔间隔时间
     *                 比如expireSeconds=10，interval=5，那么每隔5秒会重新将失效时间设置为10秒，直到锁释放
     * @return 获得的锁对象（如果为null表示获取锁失败），后续可以调用该对象的unlock方法来释放锁.
     */
    public RedisLock getAutomaticRenewalLock(final String key, long expireSeconds, long interval){
        return getAutomaticRenewalLock(key, expireSeconds, 0, 0, interval);
    }

    /**
     * 获取redis的自动续期分布式锁，内部实现使用了redis的setnx。如果锁定失败返回null，如果锁定成功则返回RedisLock对象，调用方需要调用RedisLock.unlock()方法来释放锁
     *  <br/>
     *     注意：请勿在数据库事务内获取和释放分布式锁，否则由于事务可见性的原因，可能会导致数据错乱。<br/>
     * <span style="color:red;">此方法在获取失败时会自动重试指定的次数,由于多次等待会阻塞当前线程，请尽量避免使用此方法</span>
     *
     * @param key 要锁定的key
     * @param expireSeconds key的失效时间
     * @param maxRetryTimes 最大重试次数,如果获取锁失败，会自动尝试重新获取锁；
     * @param retryIntervalTimeMillis 每次重试之前sleep等待的毫秒数
     * @param interval 续期的间隔间隔时间
     *                 比如expireSeconds=10，interval=5，那么每隔5秒会重新将失效时间设置为10秒，直到锁释放
     * @return 获得的锁对象（如果为null表示获取锁失败），后续可以调用该对象的unlock方法来释放锁.
     */
    public RedisLock getAutomaticRenewalLock(final String key, final long expireSeconds, int maxRetryTimes,
                                             long retryIntervalTimeMillis, long interval){
        final String value =  UUIDUtils.createUUID();
        int maxTimes = maxRetryTimes + 1;
        for(int i = 0;i < maxTimes; i++) {
            Boolean status = stringRedisTemplate.execute((RedisCallback<Boolean>) connection ->
                    setNx(connection, key, value, expireSeconds)
            );
            if (status != null && status) {//抢到锁
                return new AutomaticRenewalRedisLock(stringRedisTemplate, key, value, expireSeconds, interval);
            }

            if(retryIntervalTimeMillis > 0) {
                try {
                    Thread.sleep(retryIntervalTimeMillis);
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
            }
            if(Thread.currentThread().isInterrupted()){
                break;
            }
        }

        return null;
    }

    private Boolean setNx(RedisConnection connection, String key, String value, long expireSeconds){
        return connection.set(key.getBytes(), value.getBytes(), Expiration.seconds(expireSeconds), RedisStringCommands.SetOption.ifAbsent());
    }
}
