package cn.com.duiba.quanyi.center.api.utils;

import cn.com.duiba.quanyi.center.api.dto.api.ApiOrderResult;
import cn.com.duiba.quanyi.center.api.dto.api.ApiQuerySubOrderResult;
import cn.com.duiba.quanyi.center.api.dto.api.doc.ApiDocDto;
import cn.com.duiba.quanyi.center.api.dto.api.doc.create.ApiDocOrderCreateDto;
import cn.com.duiba.quanyi.center.api.dto.api.doc.create.ApiDocPkgOrderCreateDto;
import cn.com.duiba.quanyi.center.api.dto.api.doc.create.ApiDocSingleOrderCreateDto;
import cn.com.duiba.quanyi.center.api.dto.api.doc.result.ApiDocOrderResultDto;
import cn.com.duiba.quanyi.center.api.dto.api.doc.result.ApiDocPkgOrderResultDto;
import cn.com.duiba.quanyi.center.api.dto.api.doc.result.async.ApiDocPkgSubAsyncOrderResultDto;
import cn.com.duiba.quanyi.center.api.dto.api.doc.result.async.ApiDocSingleAsyncOrderResultDto;
import cn.com.duiba.quanyi.center.api.dto.api.doc.result.sync.ApiDocPkgSubSyncOrderResultDto;
import cn.com.duiba.quanyi.center.api.dto.api.doc.result.sync.ApiDocSingleSyncOrderResultDto;
import cn.com.duiba.quanyi.center.api.enums.api.ApiOrderStatusEnum;
import cn.com.duiba.quanyi.center.api.enums.api.ApiSpecTypeEnum;
import cn.com.duiba.quanyi.center.api.enums.goods.GoodsOrderStatusEnum;
import cn.com.duiba.quanyi.goods.service.api.enums.goods.SpuTypeEnum;
import cn.com.duiba.wolf.utils.BeanUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;

import java.io.Serializable;
import java.util.Collections;
import java.util.List;
import java.util.function.Supplier;
import java.util.stream.Collectors;

/**
 * @author lizhi
 * @date 2024/7/17 14:29
 */
@Slf4j
public class ApiSupplyUtils {
    
    private ApiSupplyUtils() {}

    /**
     * 将下单结果转换为API文档要求的下单结果对象
     * @param result 下单结果
     * @return API文档要求的下单结果
     */
    public static ApiDocDto<ApiDocOrderCreateDto> convertCreate(ApiOrderResult result) {
        return convert(result, () -> {
            if (isPkg(result)) {
                ApiDocPkgOrderCreateDto createDto = BeanUtils.copy(result, ApiDocPkgOrderCreateDto.class);
                createDto.setGoodsPkgFlag(true);
                return createDto;
            }
            ApiDocSingleOrderCreateDto createDto = BeanUtils.copy(result, ApiDocSingleOrderCreateDto.class);
            createDto.setOrderNum(result.getMainOrderNum());
            return createDto;
        });
    }

    /**
     * 转换成异步结果（查询订单、订单状态变更通知）
     * @param result 订单结果
     * @return 异步结果
     */
    public static ApiDocDto<ApiDocOrderResultDto> convertAsyncResult(ApiOrderResult result) {
        return convertResult(result, ApiDocPkgSubAsyncOrderResultDto.class, ApiDocSingleAsyncOrderResultDto.class);
    }

    /**
     * 转换成同步结果（同步接口返回）
     * @param result 订单结果
     * @return 同步结果
     */
    public static ApiDocDto<ApiDocOrderResultDto> convertSyncResult(ApiOrderResult result) {
        return convertResult(result, ApiDocPkgSubSyncOrderResultDto.class, ApiDocSingleSyncOrderResultDto.class);
    }

    private static <P extends ApiDocPkgSubSyncOrderResultDto, S extends ApiDocSingleSyncOrderResultDto> ApiDocDto<ApiDocOrderResultDto> convertResult(ApiOrderResult result, Class<P> pkgSubClass, Class<S> singleClass) {
        return convert(result, () -> {
            if (isPkg(result)) {
                ApiDocPkgOrderResultDto<P> resultDto = new ApiDocPkgOrderResultDto<>();
                BeanUtils.copy(result, resultDto);
                resultDto.setGoodsPkgFlag(true);
                resultDto.setSubOrderDataList(convertPkgSub(result.getSubOrderDataList(), pkgSubClass));
                return resultDto;
            }
            return convertSingle(result, singleClass);
        });
    }
    
    private static <T extends ApiDocSingleSyncOrderResultDto> T convertSingle(ApiOrderResult result, Class<T> clazz) {
        List<ApiQuerySubOrderResult> subOrderDataList = result.getSubOrderDataList();
        if (CollectionUtils.isEmpty(subOrderDataList)) {
            T dto = BeanUtils.copy(result, clazz);
            dto.setOrderNum(result.getMainOrderNum());
            return dto;
        }
        if (subOrderDataList.size() > 1) {
            log.error("[ApiSupply], not pkg, but subOrderDataList limit 1, result={}", result);
        }
        ApiQuerySubOrderResult subOrderResult = subOrderDataList.get(0);
        T dto = BeanUtils.copy(subOrderResult, clazz);
        dto.setOrderStatus(convertGoodsStatus(subOrderResult).getStatus());
        dto.setOrderNum(result.getMainOrderNum());
        dto.setRemark(result.getRemark());
        dto.setThirdOrderId(result.getThirdOrderId());
        return dto;
    }
    
    private static <T extends ApiDocPkgSubSyncOrderResultDto> List<T> convertPkgSub(List<ApiQuerySubOrderResult> subOrderDataList, Class<T> clazz) {
        if (CollectionUtils.isEmpty(subOrderDataList)) {
            return Collections.emptyList();
        }
        return subOrderDataList.stream().map(result -> {
            T sub = BeanUtils.copy(result, clazz);
            sub.setSubOrderStatus(convertGoodsStatus(result).getStatus());
            return sub;
        }).collect(Collectors.toList());
    }
    
    
    
    private static boolean isPkg(ApiOrderResult result) {
        return ApiSpecTypeEnum.PKG.getType().equals(result.getSpecType()) || ApiSpecTypeEnum.QUANTITY.getType().equals(result.getSpecType());
    }
    
    private static <T extends Serializable> ApiDocDto<T> convert(ApiOrderResult result, Supplier<T> func) {
        if (result == null) {
            return null;
        }
        ApiDocDto<T> dto = new ApiDocDto<>();
        if (!result.isSuccess()) {
            return dto.buildError(result.getErrCode(), result.getErrMsg());
        }
        return dto.buildSuccess(func);
    }

    private static ApiOrderStatusEnum convertGoodsStatus(ApiQuerySubOrderResult subOrderResult) {
        if (isIssued(subOrderResult)) {
            return ApiOrderStatusEnum.ISSUED;
        }
        GoodsOrderStatusEnum goodsOrderStatusEnum = GoodsOrderStatusEnum.getByStatus(subOrderResult.getOrderStatus());
        if (goodsOrderStatusEnum == null) {
            return ApiOrderStatusEnum.PROCESS;
        }
        switch (goodsOrderStatusEnum) {
            case FAIL:
                return ApiOrderStatusEnum.FAIL;
            case SUCCESS:
                return ApiOrderStatusEnum.RECEIVED;
            case VERIFICATION:
                return ApiOrderStatusEnum.VERIFICATION;
            case EXPIRED:
                return ApiOrderStatusEnum.EXPIRED;
            case RECEIVED:
                return ApiOrderStatusEnum.RED_PACK_RECEIVED;
            case REFUND:
                return ApiOrderStatusEnum.REFUND;
            case INVALID:
                return ApiOrderStatusEnum.INVALID;
            case UNSHIPPED:
                return ApiOrderStatusEnum.UNSHIPPED;
            case SHIPPED:
                return ApiOrderStatusEnum.SHIPPED;
            case COMPLETED:
                return ApiOrderStatusEnum.COMPLETED;
            case CANCELED:
                return ApiOrderStatusEnum.CANCELED;
            case RETURNED:
                return ApiOrderStatusEnum.RETURNED;
            case CLOSE:
                return ApiOrderStatusEnum.CLOSE;
            default:
                return ApiOrderStatusEnum.PROCESS;
        }
    }

    private static boolean isIssued(ApiQuerySubOrderResult subOrderResult) {
        if (GoodsOrderStatusEnum.isShortUrlStatus(subOrderResult.getOrderStatus())) {
            // 短链状态
            return true;
        }
        // 卡密，且是发放成功状态
        return SpuTypeEnum.KA_MI.getType().equals(subOrderResult.getSpuType()) && GoodsOrderStatusEnum.SUCCESS.getStatus().equals(subOrderResult.getOrderStatus());
    }
}
