package cn.com.duiba.quanyi.center.api.utils;

import java.util.Calendar;
import java.util.Date;

import org.apache.commons.lang3.time.DateUtils;

import cn.com.duiba.quanyi.center.api.dto.activity.ClaimTimeLimitDto;
import cn.com.duiba.quanyi.center.api.enums.activity.ClaimTimeLimitTypeEnum;
import lombok.extern.slf4j.Slf4j;

/**
 * 领取时间限制校验工具类
 * 
 * @author yangyukang
 * @date 2025-01-21
 */
@Slf4j
public class ClaimTimeLimitValidatorUtils {
    
    /**
     * 错误提示信息
     */
    private static final String CLAIM_TIME_OUT_MESSAGE = "对不起，当前时间不在使用时间范围内！";
    
    /**
     * 校验当前时间是否在允许的领取时间范围内
     * 
     * @param claimTimeLimitDto 领取时间限制配置
     * @param orderCreateTime 兑换时间
     * @return true表示可以领取，false表示不可领取
     */
    public static boolean isValidClaimTime(ClaimTimeLimitDto claimTimeLimitDto, Date orderCreateTime) {
        if (claimTimeLimitDto == null || claimTimeLimitDto.getLimitType() == null) {
            // 未配置限制，默认允许
            return true;
        }
        
        ClaimTimeLimitTypeEnum limitType = ClaimTimeLimitTypeEnum.getByType(claimTimeLimitDto.getLimitType());
        if (limitType == null || limitType == ClaimTimeLimitTypeEnum.NO_LIMIT) {
            // 不限制，允许领取
            return true;
        }

        Date currentTime = new Date();

        try {
            switch (limitType) {
                case LATEST_CLAIM_TIME:
                    return validateLatestClaimTime(claimTimeLimitDto.getLatestClaimTime(), currentTime);
                    
                case DAYS_AFTER_ORDER:
                    return validateDaysAfterOrder(claimTimeLimitDto.getDaysAfterOrder(), orderCreateTime, currentTime);
                    
                case MONTH_DAY_AFTER_ORDER:
                    return validateMonthDayAfterOrder(claimTimeLimitDto.getMonthDayAfterOrder(), orderCreateTime, currentTime);
                    
                case QUARTER_AFTER_ORDER:
                    return validateQuarterAfterOrder(orderCreateTime, currentTime);
                    
                case CUSTOM_TIME:
                    return validateCustomTime(claimTimeLimitDto.getCustomStartTime(), claimTimeLimitDto.getCustomEndTime(), currentTime);
                    
                default:
                    log.warn("未知的领取时间限制类型: {}", claimTimeLimitDto.getLimitType());
                    return false;
            }
        } catch (Exception e) {
            log.error("领取时间校验异常, claimTimeLimitDto: {}, currentTime: {}", claimTimeLimitDto, currentTime, e);
            // 异常情况下，为了不影响业务，默认允许领取
            return false;
        }
    }
    
    /**
     * 校验最晚可领取时间
     */
    private static boolean validateLatestClaimTime(Date latestClaimTime, Date currentTime) {
        if (latestClaimTime == null) {
            return false;
        }
        // 当前时间必须小于等于最晚可领取时间
        return !currentTime.after(latestClaimTime);
    }
    
    /**
     * 校验订单生成后x天内可领取
     */
    private static boolean validateDaysAfterOrder(Integer daysAfterOrder, Date orderCreateTime, Date currentTime) {
        if (daysAfterOrder == null || orderCreateTime == null) {
            return false;
        }
        Date deadline = DateUtils.addDays(orderCreateTime, daysAfterOrder);
        // 当前时间必须在订单生成时间到截止时间之间（包含边界）
        return !currentTime.before(orderCreateTime) && !currentTime.after(deadline);
    }
    
    /**
     * 校验订单生成后当月x号内领取
     */
    private static boolean validateMonthDayAfterOrder(Integer monthDayAfterOrder, Date orderCreateTime, Date currentTime) {
        if (monthDayAfterOrder == null || orderCreateTime == null) {
            return false;
        }
        
        // 当前时间必须大于等于订单创建时间
        if (currentTime.before(orderCreateTime)) {
            return false;
        }
        
        Calendar orderCal = Calendar.getInstance();
        orderCal.setTime(orderCreateTime);
        int orderYear = orderCal.get(Calendar.YEAR);
        int orderMonth = orderCal.get(Calendar.MONTH);
        
        Calendar currentCal = Calendar.getInstance();
        currentCal.setTime(currentTime);
        int currentYear = currentCal.get(Calendar.YEAR);
        int currentMonth = currentCal.get(Calendar.MONTH);
        int currentDay = currentCal.get(Calendar.DAY_OF_MONTH);
        
        // 如果当前时间在订单生成的当月，且当前日期小于等于配置的号数
        if (orderYear == currentYear && orderMonth == currentMonth) {
            return currentDay <= monthDayAfterOrder;
        }
        // 不在当月的，不允许领取
        return false;
    }
    
    /**
     * 校验订单生成后当季度可领取
     */
    private static boolean validateQuarterAfterOrder(Date orderCreateTime, Date currentTime) {
        if (orderCreateTime == null) {
            return false;
        }
        
        // 当前时间必须大于等于订单创建时间
        if (currentTime.before(orderCreateTime)) {
            return false;
        }
        
        Calendar orderCal = Calendar.getInstance();
        orderCal.setTime(orderCreateTime);
        int orderYear = orderCal.get(Calendar.YEAR);
        int orderQuarter = getQuarter(orderCal.get(Calendar.MONTH));
        
        Calendar currentCal = Calendar.getInstance();
        currentCal.setTime(currentTime);
        int currentYear = currentCal.get(Calendar.YEAR);
        int currentQuarter = getQuarter(currentCal.get(Calendar.MONTH));
        
        // 当前时间必须在订单生成的当季度
        return orderYear == currentYear && orderQuarter == currentQuarter;
    }
    
    /**
     * 获取季度（1-4）
     * 1月=季度1, 2月=季度1, 3月=季度1
     * 4月=季度2, 5月=季度2, 6月=季度2
     * 7月=季度3, 8月=季度3, 9月=季度3
     * 10月=季度4, 11月=季度4, 12月=季度4
     */
    private static int getQuarter(int month) {
        return (month / 3) + 1;
    }
    
    /**
     * 校验自定义时间
     */
    private static boolean validateCustomTime(Date customStartTime, Date customEndTime, Date currentTime) {
        if (customStartTime == null || customEndTime == null) {
            return false;
        }
        // 当前时间必须在自定义时间范围内（包含边界）
        return !currentTime.before(customStartTime) && !currentTime.after(customEndTime);
    }
    
    /**
     * 获取校验失败的错误提示信息
     */
    public static String getErrorMessage() {
        return CLAIM_TIME_OUT_MESSAGE;
    }
}

