package cn.com.duiba.quanyi.center.api.utils.activity;

import cn.hutool.core.date.DateUnit;
import cn.hutool.core.date.DateUtil;

import java.text.SimpleDateFormat;
import java.time.DayOfWeek;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAdjusters;
import java.util.Calendar;
import java.util.Date;

/**
 * DateUtils
 *
 * @author fmk
 * @since 2024/4/23
 */
public class QyDateTimeUtils {

    private QyDateTimeUtils() {}

    /**
     * 获取上周一时间
     *
     * @return 上周一
     */
    public static Date getMondayOfPreWeek() {
        return convertLocalDateToDate(LocalDate.now().minusWeeks(1).with(DayOfWeek.MONDAY));
    }

    /**
     * 获取本周一时间
     *
     * @return 本周一
     */
    public static Date getMondayOfThisWeek() {
        return convertLocalDateToDate(LocalDate.now().with(DayOfWeek.MONDAY));
    }

    /**
     * 获取上周日时间
     *
     * @return 本周一
     */
    public static Date getSundayOfPreWeek() {
        return convertLocalDateToDate(LocalDate.now().minusWeeks(1).with(DayOfWeek.SUNDAY));
    }

    /**
     * 获取下周一时间
     *
     * @return 本周一
     */
    public static Date getMondayOfNextWeek() {
        return convertLocalDateToDate(LocalDate.now().plusWeeks(1).with(DayOfWeek.MONDAY));
    }

    public static Date convertLocalDateToDate(LocalDate localDate) {
        return Date.from(localDate.atStartOfDay(ZoneId.systemDefault()).toInstant());
    }

    public static LocalDate convertDateToLocalDate(Date date) {
        return date.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
    }

    public static Date calculateTime(int weekOfDay, Date time) {
        String zoneId = "Asia/Shanghai";
        // 将给定的时间转换为 ZonedDateTime，并设置时区为 "Asia/Shanghai"
        ZonedDateTime zonedDateTime = time.toInstant().atZone(ZoneId.of(zoneId));

        // 获取当前日期
        LocalDate currentDate = LocalDate.now(ZoneId.of(zoneId));

        // 获取给定的周几日期
        LocalDate targetDate = currentDate.with(TemporalAdjusters.nextOrSame(DayOfWeek.of(weekOfDay)));

        // 如果目标日期在当前日期之前，说明目标日期在下一周，需要加一周
        if (targetDate.isBefore(currentDate)) {
            targetDate = targetDate.plusWeeks(1);
        }

        // 将给定时间的时分秒设置到目标日期上
        ZonedDateTime targetZonedDateTime = targetDate.atTime(zonedDateTime.toLocalTime()).atZone(ZoneId.of(zoneId));

        // 将 ZonedDateTime 转换为 Date
        return Date.from(targetZonedDateTime.toInstant());
    }

    public static String parse(String date, String oldFormat, String newFormat) {
        DateTimeFormatter inputFormatter = DateTimeFormatter.ofPattern(oldFormat);
        LocalDateTime dateTime = LocalDateTime.parse(date, inputFormatter);

        DateTimeFormatter outputFormatter = DateTimeFormatter.ofPattern(newFormat);
        return dateTime.format(outputFormatter);
    }

    public static int getWeekOfDay() {
        Calendar calendar = Calendar.getInstance();
        // 设置周一为一周的第一天
        calendar.setFirstDayOfWeek(Calendar.MONDAY);
        // 获取当前日期
        int today = calendar.get(Calendar.DAY_OF_WEEK);
        return today == Calendar.SUNDAY ? 7 : today - 1;
    }

    /**
     * 获取23:59:59时间点
     * @param date
     * @return Date 年月日不变，时分秒改为当天的23:59:59
     */
    public static Date getDayEndTime(Date date) {
        if (null == date) {
            return null;
        }
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.set(Calendar.HOUR_OF_DAY, 23);
        calendar.set(Calendar.MINUTE, 59);
        calendar.set(Calendar.SECOND, 59);
        calendar.set(Calendar.MILLISECOND, 0);
        return calendar.getTime();
    }

    /**
     * 获取开始时间，即零点（00:00:00）。
     */
    public static Date getDayStartTime(Date date) {
        if (null == date) {
            return null;
        }
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        return calendar.getTime();
    }

    /**
     * 获取当前加或者减几个小时的时间
     */
    public static Date getNowAddHours(int amount) {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.HOUR_OF_DAY, amount);
        return calendar.getTime();
    }

    /**
     * 将日期去掉毫秒
     */
    public static Date removeMillisecond(Date date) {
        if (null == date) {
            return null;
        }
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.set(Calendar.MILLISECOND, 0);
        return calendar.getTime();
    }

    /**
     * 根据传入的格式，获取昨天的日期字符串
     * @param pattern 格式
     */
    public static String getYesterday(String pattern) {
        DateTimeFormatter fmt = DateTimeFormatter.ofPattern(pattern);
        LocalDate lastDayDate = LocalDate.now().minusDays(1);
        return lastDayDate.format(fmt);
    }

    /**
     * 获取date
     * @param date
     * @param pattern
     * @return
     */
    public static Date getDate(String date, String pattern) {
        DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern(pattern);
        LocalDate localDate = LocalDate.parse(date, dateTimeFormatter);
        // 将LocalDate转换为java.util.Date
        return convertLocalDateToDate(localDate);
    }

    /**
     * 将日期格式化成字符串
     *
     * @param date          日期
     * @param formatPattern 格式
     */
    public static String getStr(Date date, String formatPattern) {
        if (date == null) {
            return "";
        }
        SimpleDateFormat format = new SimpleDateFormat(formatPattern);
        return format.format(date);
    }

    /**
     * 获取两个日期之间间隔的天数
     */
    public static long daysBetween(Date startDate, Date endDate) {
        return DateUtil.between(startDate, endDate, DateUnit.DAY, true);
    }

    /**
     * 获取指定日期添加天数后的日期
     */
    public static Date addDays(Date availableBeginTime, Integer consumeStockDays) {
        return DateUtil.offsetDay(availableBeginTime, consumeStockDays);
    }
    /**
     * 获取当前的时间，格式为：yyyy-MM-dd HH:mm:ss
     */
    public static String getCurrentTime() {
        return DateUtil.now();
    }

    /**
     * 获取当前月底时间 23:59:59
     */
    public static Date getCurrentMonthEndTime(Date date) {
        return getDayEndTime(DateUtil.endOfMonth(date));
    }

    /**
     * 获取当前整小时时间
     * @return
     */
    public static Date getHourlyDate() {
        // 获取当前时间（带时区）
        LocalDateTime now = LocalDateTime.now();

        // 将分钟、秒和纳秒归零，保留年月日时
        LocalDateTime truncatedTime = now
                .withMinute(0)
                .withSecond(0)
                .withNano(0);

        // 转换为 java.util.Date
        return Date.from(truncatedTime.atZone(ZoneId.systemDefault()).toInstant());
    }

    /**
     * 获取本周周日 23:59:59时间
     *
     * @return 本周日
     */
    public static Date getSundayOfThisWeek() {
        return getDayEndTime(convertLocalDateToDate(LocalDate.now().with(DayOfWeek.SUNDAY)));
    }

}
